function [x, f, loc, X, G, w, H] = hanso(pars, options)
%  HANSO: Hybrid Algorithm for Nonsmooth Optimization, Version 3.0, 2021.
%  Minimization algorithm intended for nonsmooth, nonconvex functions,
%   but also applicable to functions that are smooth, convex or both.
%   Based on BFGS and Gradient Sampling: see refs below for more details.
%  [X,F] = HANSO(PARS) returns approximate minimizer and function value 
%   of the function with PARS.NVAR variables coded in mfile PARS.FGNAME.
%  [X,F,LOC] = HANSO(PARS,OPTIONS) allows specification of options and
%   also returns an approximate Local Optimality Certificate.
%  [X,F,LOC,X,G,W,H] = HANSO(PARS,OPTIONS) returns additional data 
%   supporting the approximate Local Optimality Certificate and the
%   final inverse Hessian approximation matrix generated by BFGS.
% 
%   Input parameters
%    pars is a required struct, with two required fields
%      pars.nvar: the number of variables
%      pars.fgname: string giving the name of m-file (in single quotes) 
%         that returns the function and its gradient at a given input x, 
%         with call   [f,g] = fgtest(x,pars)  if pars.fgname is 'fgtest'.
%         Any data required to compute the function and gradient may be
%         encoded in other fields of pars. The user does not have to worry
%         about the nondifferentiable case or identify subgradients. 
%         The basic assumption is that the nondifferentiable case arises
%         with probability zero, and in the event that it does occur, it is
%         fine to return the gradient of the function at a nearby point.
%         For example, the user does not have to worry about ties in "max".
%    options is an optional struct, with no required fields
%      options.x0: columns are one or more starting vector of variables, 
%          used to intialize the BFGS phase of the algorithm
%          (default: 10 starting points are generated randomly)
%      options.maxit: max number of iterations for each BFGS starting point
%          (default: 1000)
%      options.maxit_gradsamp: max number of iterations for gradient sampling
%          phase, initalized from the best result found by BFGS
%          (default: 0 (no gradient sampling phase, as it may be expensive)
%      options.nvec: 0 for full BFGS matrix update in BFGS phase, otherwise 
%          number of vectors to save and use in the limited memory updates
%          (default: 0)
%      options.normtol: termination tolerance for norm of smallest vector 
%          in convex hull of a set of saved gradients: these are gradients 
%          at the current and sufficiently close previous iterates in the 
%          case of BFGS, or gradients at the current iterate and nearby 
%          randomly generated points, in the case of gradient sampling.
%          In the latter case, options.normtol is used to define nu_opt
%          in Algorithm GS: the stationarity termination tolerance.
%          See description of the next option for more details.
%          (default: 1e-4)
%      options.evaldist: evaluation distance used to determine the gradients
%          in the set described in the previous option: these are gradients
%          at the current iterate x and at the previous iterates approximately
%          within distance options.evaldist of x, in the case of BFGS, or
%          gradients at the current iterate x and randomly generated points
%          whose displacement from x is uniformly drawn from
%          options.evaldist times the unit ball, in the case of gradient
%          sampling. In the latter case, options.evaldist is used to define
%          eps_opt in Algorithm GS: the sampling radius termination tolerance. 
%          (default: 1e-4)
%      options.samprad0: initial sampling radius for gradient sampling,
%          known as eps0 in Algorithm GS.
%          (default: 0.1)
%      options.target0: initial stationarity target for gradient sampling,
%          known as nu0 in Algorithm GS.
%          (default: 0.1)
%      options.prtlevel: print level, between 0 (none) and 3 (most verbose).
%          (default: 1)
%
%   Output parameters 
%    x: column vector, the best point found
%    f: the value of the function at x
%    loc: local optimality certificate, structure with 2 fields:
%      loc.dnorm: norm of a vector in the convex hull of gradients of the 
%          function evaluated at and near x 
%      loc.evaldist: specifies max distance from x at which these gradients 
%       were evaluated (approximately, if there is no gradient sampling phase). 
%       loc is a measure of Clarke stationarity. The smaller loc.dnorm and 
%       loc.evaldist are, the more likely it is that x is an approximate 
%       local minimizer.
%    X: columns are point where these gradients were evaluated, including x
%    G: the gradients of the function at the columns of X
%    w: vector of nonnegative weights summing to one such that 
%        loc.dnorm = ||G*w||
%    H: the final BFGS inverse Hessian approximation for the best result
%       found by BFGS, typically ill-conditioned if the function is nonsmooth
%
%   Method: 
%      BFGS phase: BFGS is run from multiple starting points, taken from
%       the columns of options.x0, if provided, and otherwise 10 points
%       generated randomly. If options.maxit_gradsamp = 0 (the default)
%       HANSO terminates; otherwise, it continues to:
%      Gradient sampling phase, with automatic reduction of 
%       sampling radius and stationarity target from initial values
%       provided in options.samprad0 and options.target0, 
%       using Algorithm GS from reference [1] below.
%       Unlike BFGS, this algorithm has guaranteed convergence properties
%       for a general class of nonsmooth functions: see [1] for details.
%   References: 
%    [1] J.V. Burke, F.E. Curtis, A.S. Lewis, M.L. Overton and L.E.A. Simões, 
%      Gradient Sampling Methods for Nonsmooth Optimization,
%      In: Numerical Nonsmooth Optimization, edited by A. Bagirov et al, 
%      Springer (2020), pp. 201-225, https://arxiv.org/abs/1804.11003
%    [2] A.S. Lewis and M.L. Overton, Nonsmooth Optimization via Quasi-Newton
%     Methods, Math Programming 141 (2013), pp. 135-163
%    [3] J.V. Burke, A.S. Lewis and M.L. Overton, A Robust Gradient Sampling 
%     Algorithm for Nonsmooth, Nonconvex Optimization
%      SIAM J. Optimization 15 (2005), pp. 751-779
%   Send comments/bug reports to Michael Overton, mo1@nyu.edu.
%   Version 3.0, 2021, see GPL license info below.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  HANSO 3.0 Copyright (C) 2021  Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Version notes:
%
% Starting with HANSO 3.0, the gradient sampling phase of the algorithm
% implements Algorithm GS in our survey paper [1] (see above).
% As explained there, Algorithm GS is based on a 2007 paper of Kiwiel 
% which incorporated several modifications to our 2005 paper [3].
% Consequently, the user does not provide a vector of sampling radii
% as in earlier versions, but only an initial sampling radius which is
% then decreased adaptively. See the comments in gradsamp.m for details.
% As noted above, Algorithm GS has guaranteed convergence properties for a
% general class of nonsmooth functions. However, by default there is no 
% gradient sampling phase of the algorithm since it is usually significantly 
% more costly than the BFGS phase.
% 
% The BFGS phase of the algorithm, which is based on our 2013 paper [2],
% has not changed since version 2.2, except that the tolerances in the
% quadratic program solver qpspecial.m (used to check the termination
% condition) have been tightened, and minor cosmetic changes have been made.
%
% The cpumax and fvalquit options in HANSO 2.2 were eliminated in version 3.0.

%
% parameter defaults
%
if nargin == 0
   error('hanso: "pars" is a required input parameter')
end
if nargin == 1
   options = [];
end
%
% Set defaults for HANSO 3.0.  No longer calling "setdefaults".
% Additional BFGS and its line search defaults are set by setdefaults_bfgs, 
% called later by bfgs. Additional gradient sampling and its line search
% defaults are set by setdefaults_gradsamp, called later by gradsamp if
% invoked.
%
options = setdefaults_hanso(pars, options);
prtlevel = options.prtlevel; % used by BFGS and gradient sampling
if prtlevel > 0  
    fprintf('HANSO Version 3.0, optimizing objective %s over %d variables with options\n', ...
        pars.fgname, pars.nvar')
    disp(options)
end
%
% BFGS phase with possibly multiple starting points (10 if not provided by user)
% The max number of iterations per starting point is options.maxit.
% We considered changing this option to be called options.maxit_bfgs, but
% retained the name options.maxit for backwards compatibility with HANSO 2.2
%
if prtlevel > 0
    if options.nvec == 0
        fprintf('BFGS Phase\n\n')
    else
        fprintf('BFGS Phase (Limited Memory version with memory size %d)\n\n',options.nvec)
    end
    
end
%
% In the following, x(:,j) is the best point found by BFGS using starting
% point j, with function value f(j), while the columns of X{j} include both
% x(:,j) and other nearby points, and the columns of G{j} are the gradients
% of the function at these points. Furthermore, d(:,j) is the smallest
% vector in the convex hull of the columns in G{j}, with weights w{j}, so
% d(:,j) = G{j}*w{j} and w{j}>=0. Finally H{j} is the final inverse Hessian
% approximation computed by BFGS using starting point j.
% 
[x, f, d, H, ~, ~, X, G, w] = bfgs(pars, options); 
if length(f) > 1 % more than one starting point
    [f,indx] = min(f); % throw away all but the best result
    x = x(:,indx);
    d = d(:,indx);
    H = H{indx}; % bug if do this when only one start point: H already a matrix in that case
    X = X{indx};
    G = G{indx};
    w = w{indx};
end
if isnaninf(f) 
    error('hanso: f is inf or nan at all starting points')
end
%
% Compute local optimality certificate for BFGS phase: loc.dnorm is the
% norm of d = G*w, and loc.evaldist is the max distance from x (the first
% column of X) to the other columns of X. We pass x, G and w as well as 
% X and d to verify the properties just claimed. Notice that although BFGS
% attempted to compute an optimality certificate satisfying loc.dnorm <=
% options.normtol and loc.evaldist <= options.evaldist, there is no guarantee
% that this was done, partly because of the specific way BFGS chooses the 
% iterates that are "nearby" x (see p.158-159 of [2]), and partly because
% BFGS may have quit before satisfying the termination criteria.
%
loc = getloc(x, X, G, d, w); 
if prtlevel > 0
    fprintf('hanso: best point found by bfgs phase has f = %10.7e\n', f);
    fprintf(' with local optimality measure: dnorm = %5.1e, evaldist = %5.1e\n',loc.dnorm, loc.evaldist)
    if loc.dnorm <= options.normtol && loc.evaldist <= options.evaldist
        fprintf('hanso: verified optimality within requested tolerance in bfgs phase\n')
    end
end
%
% on to gradient sampling phase, if requested
%
if options.maxit_gradsamp == 0 % the default is 0: no gradient sampling
    if prtlevel > 0
        fprintf('\nhanso: no Gradient Sampling phase was requested\n')
        fprintf('hanso: if Gradient Sampling phase is desired, set options.maxit_gradsamp > 0\n')
    end
    return 
else
    f_BFGS = f; % only used for print statements below
    %
    % set options for gradient sampling
    % (use default values for other options)
    %
    options.maxit = options.maxit_gradsamp; 
    options = rmfield(options,'maxit_gradsamp'); 
    options.x0 = x; % start at best point found by BFGS
    options.eps0 = options.samprad0; % initial value of sampling radius 
    options.nu0 = options.target0; % initial stationarity target
    options.eps_opt = options.evaldist; % final sampling radius 
    options.nu_opt = options.normtol; % final stationarity target
    % 
    % gradient sampling: Algorithm GS in our survey paper: see [1] above
    %
    if prtlevel > 0
        fprintf('\nGradient Sampling Phase\n\n')
    end
    %
    % In the following, x is the best point found by gradient sampling,
    % with function value f, while the columns of X include both
    % x and other nearby points, and the columns of G are the gradients
    % of the function at these points. Furthermore, d is the smallest
    % vector in the convex hull of the columns in G, with weights w, so
    % d = G*w. Note that because gradient sampling starts at the best point
    % found by BFGS, the best function value found by gradient sampling
    % cannot be larger than the best found by BFGS.
    %
    [x, f, d, X, G, w] = gradsamp(pars, options);
    if f < f_BFGS
        if prtlevel > 0
            fprintf('hanso: gradient sampling reduced f below best value found by BFGS\n')
        end
    elseif f == f_BFGS
        if prtlevel > 0
            fprintf('hanso: gradient sampling did not reduce f below best value found by BFGS\n')
        end
    else
        error('hanso: gradient sampling increased f above the best point found by BFGS, which should not happen: email mo1@nyu.edu') 
    end
    %
    % Although in principle the certificate for BFGS computed above could 
    % be "better" than the one computed by gradient sampling, it is less
    % reliable, and so there is no point in saving it.
    %
    % Compute local optimality certificate for gradient sampling phase:
    % loc.dnorm is the norm of d = G*w, and loc.evaldist is the max
    % distance from x (the first column of X) to the other columns of X. 
    % We pass x, G and w as well as X and d to verify the properties just
    % claimed. Notice that although gradient sampling attempted to compute
    % an optimality certificate satisfying loc.dnorm <= options.normtol and
    % loc.evaldist <= options.evaldist, there is no guarantee that this was
    % done, since it may have quit before satisfying the termination criteria.
    %
    loc = getloc(x, X, G, d, w); 
    if options.prtlevel > 0
        fprintf('hanso: best point found by gradient sampling phase has f = %10.7e\n', f);
        fprintf(' with local optimality measure: dnorm = %5.1e, evaldist = %5.1e\n',loc.dnorm, loc.evaldist)
        if loc.dnorm <= options.normtol && loc.evaldist <= options.evaldist
            fprintf('hanso: verified optimality within requested tolerance in gradient sampling phase\n')
        end
    end
end