function [coeff,M,QQ] = divergingpoly(n1,n2,z,b,prtlevel)
% Called by affpolymin only.
% Set up the polynomial defining the "big M" for the real abscissa
% problem where a multiple root goes to minus infinity
% Consider the equation
% p(s) = s^n + c_1 s^n-1 + ... + c_n
%      = (s + z)^n1 (s + M)^n2  
%      = (s^n1 + n1 z s^n1-1 + ... + z^n1)(s^n2 + n2 M s^n2-1 + ... + M^n2)
%      = s^n + (n1 z + n2 M) s^n-1 + 
%         + ((n1 choose 2)z^2 + n1 n2 z M + (n2 choose 2)) M^2 s^n-2 + ...
% This defines the coeffs c_i in terms of z (which we know, from solving
% the derivative polynomial equation previously) and M (which we want).
% The equation b'*c = beta then gives a polynomial equation for M.
% The closer z is chosen to optroot, the larger M will be, with a positive
% sign as long as z > optroot.
%
% Notice that z and M have signs consistent with optroot, that is the signs
% are NOT reversed, but later, in affpolymin, the signs of M and optroot will 
% be changed.  This is to avoid powers of -1. 
%
% Notation guide:  
%    divergingpoly.m:      n1   n2   z             M   QQ(1)....QQ(n2+1)   
%    the BGMO paper:       n-m  m    alpha*+eps    K   eta_0...eta_m
%    except that in the paper, we assumed WLOG alpha*=0 to simplify eta_k
%
% In the calling program affpolymin, z has been set to optroot + eps,
% that is alpha* + eps, where eps is small.  Normally, n2 has been set to
% the index of the derivative of Q for which optroot is the rightmost root,
% called l (ell) in the BGMO paper.  However, it might be necessary to set
% n2 to l + 1 (ell + 1), as explained in the proof of the relevant theorem
% in the paper.  This is done in affpolymin if n2=l (ell) fails.
%
%  Send comments or bug reports to Michael Overton, overton@cs.nyu.edu,
%  with subject header containing the string "affpolymin".
%  Version 1.0, 2010, see GPL license info below.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  AFFPOLYMIN 1.0 Copyright (C) 2010  Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
V = zeros(n1+1,1);
for j=0:n1
    V(j+1) = nchoosek(n1,j)*z^j;
end
bb = b'; % note the conjugate!!!
for j=0:n2
    QQ(j+1) = nchoosek(n2,j)*bb(j+1:j+1+n1)'*V;
end
QQ = fliplr(QQ);
if QQ(1) == 0
    if prtlevel > 0  % z was chosen to be equal to optroot - eps
        fprintf('divergingpoly: cannot find a root, need to increase eps\n')
    end
    M = nan;
    coeff = nan;
    return
end
% M is the largest root of QQ, which should be real, positive and O(1/eps)
r = roots(QQ);
[M,idx] = max(real(r));
if imag(r(idx)) ~= 0
    if prtlevel > 0
        fprintf('divergingpoly: largest real part of root defining M is not real\n')
    end
    % this may happen because we had to make eps big
    % probably increasing eps will not help, but may as well try
    % alternatively, it might be because n2=ell and it needs to be ell+1
    % (see above).  In either case, corrective action is taken in
    % affpolymin
    coeff = nan;
end
% the following convolution generalizes what was done in convolvebinomials
V = zeros(n1+1,1); % result has to be a column vector
for j=0:n1
    V(j+1) = nchoosek(n1,j)*z^j;
end
W = zeros(n2+1,1);
for j=0:n2
    W(j+1) = nchoosek(n2,j)*M^j;
end
coeff = conv(V,W); % retain the leading one
coeff = conj(coeff');  % transpose without conjugation
