function [optroot, coeff, k, M, allroots, Q, QQ, countrecord] = affpolymin(b,radius,eps,prtlevel)
% OPTROOT = AFFPOLYMIN(b) computes the global minimum of 
%  the ROOT RADIUS (max modulus of the roots) of the 
%  monic real or complex polynomial family of degree n 
%       p(s) = z^n + coeff(2) z^n-1 + ... + coeff(n+1),
%  with one affine constraint on the coefficients, namely
%              b(1) + b(2)coeff(2) + ... + b(n+1)coeff(n+1) = 0.
%  If b is real, the optimization is over real coefficients and the
%   optimal polynomial is (z - optroot)^(n-k) * (z + optroot)^k 
%   for some k between 0 and n.
%  If b is complex, the optimization is over complex coefficients and the 
%   optimal polynomial is (z - optroot)^n.
%  For optimizing over complex coefficients when b is real, multiply b by i
%  b must be a ROW vector.
%
% OPTROOT = AFFPOLYMIN(b,0) does the same thing for the ROOT ABSCISSA
%  (largest real part of the roots).
%  If b is real, the optimization is over real coefficients and
%  if the optimal value is attained, the optimal polynomial is
%   (z - optroot)^n; for the case that it is not attained, see below.
%  If b is complex, the optimization is over complex coefficients, the
%  optimal value is attained and the optimal polynomial is (z - optroot)^n.
%
% [OPTROOT, COEFF] = AFFPOLYMIN(b,radius) returns the optimal coefficients 
%  as well, with COEFF(1)=1 since the polynomial is monic, except in the
%  real abscissa case if the optimal value is not attained: see below. 
%  COEFF is a row vector, so in exact arithmetic sum(b.*coeff) would be 0.
%  However, computing the roots of COEFF may result in rounding errors 
%  that are of the order of the nth root of machine precision, and 
%  furthermore the coefficients can be enormous so the constraint residual
%  sum(b.*coeff) may be huge.  Nonetheless, the optimal root OPTROOT 
%  is generally computed accurately.
%
% [OPTROOT, COEFF, k, M] = AFFPOLYMIN(b,radius,eps) works as follows.
%  In the complex cases, k should be 0 and M should be NaN.
%  In the real radius case, k should be the integer for which the optimal
%  polynomial is (z - optroot)^(n-k) * (z + optroot)^k  and M should be NaN.
%  For the real abscissa case, the optimal value may not be attained:
%  in theory there is one root with multiplicity n-k converging to 
%  OPTROOT while the real parts of the other k roots go to -INF, so
%  extracting approximate coefficients is an unstable process. AFFPOLYMIN
%  computes coefficients for which the roots consist of n-k roots at 
%  OPTROOT + EPS which implies setting the other k roots to a large negative
%  real number M with magnitude O(1/EPS).  If EPS is too small rounding 
%  errors may result in M >> OPTROOT instead of M << OPTROOT, in which case 
%  EPS is increased and the process repeated.  Small EPS also results in 
%  rounding error swamping the constraint on the coefficients. 
%  The default value for EPS is (1e-16)^(1/n).
%
% [OPTROOT, COEFF, k, M, ALLROOTS, Q, QQ, COUNTRECORD] = AFFPOLYMIN(b,radius) 
%  returns ALLROOTS, all of
%    the real roots of the polynomials g_j (real radius case) 
%    the real roots of polynomial h and its derivatives (real abscissa case)
%    all the roots of the polynomial h (complex radius or abscissa case)
%  of which -OPTROOT is one with the largest radius or real part respectively, 
%  as well as Q, the vector of coefficients of h or the relevant g_k, 
%  QQ, the vector of coefficients of the polynomial defining M when relevant, 
%  and COUNTRECORD, which is needed to reconstruct the coefficients of the
%  polynomials corresponding to the roots in ALLROOTS. See the paper
%  referenced below for definitions of the polynomials g_j and h.
%
% OPTROOT = AFFPOLYMIN(b,radius,eps,prtlevel) 
%  allows the user to control output: prtlevel = 0 (nothing), prtlevel = 1
%  (informative), prtlevel = 2 (default: also shows plot of h in the 
%  real abscissa case only) 
%
%  Reference:
%   V.D. Blondel, M. Gurbuzbalaban, A. Megretski and M.L. Overton,
%    Explicit Solutions for Root Optimization of a Polynomial Family with
%    One Affine Constraint, 2010
%  The indexing is shifted by 1 here, since zero subscripts are illegal.
%  Thus the correspondences:
%    code   b(1),...,b(n+1)   coeff(1),...,coeff(n+1)    optroot
%    paper  B_0,...,B_n          1, a_1, ..., a_n         gamma
%  and, in the real abscissa case:
%    code:   k     z            M       QQ(1)....QQ(k+1)  
%    paper:  m  alpha*+eps  M_eps = -K  eta_0...eta_m
%
%  Send comments or bug reports to Michael Overton, overton@cs.nyu.edu,
%  with subject header containing the string "affpolymin".
%  Version 1.0, 2010, see GPL license info below.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  AFFPOLYMIN 1.0 Copyright (C) 2010  Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
M = nan;
QQ = nan;
countrecord = nan;
if nargin < 1
    error('b is a required input parameter')
end
if size(b,1) ~= 1
    error('b must be a row vector')
end
n = length(b) - 1; % polynomial degree
if n == 0 % n=1 is a trivial case, but the program works fine
    error('affpolymin: length of b must be at least two')
end
b_real = isreal(b);
if nargin < 2
    radius = 1;  % default
end
if nargin < 3 | isempty(eps)
    eps = (1e-16)^(1/n); % this models sensitivity of roots to coeffs
end
if eps < 0
    error('eps must be nonnegative')
end
if nargin < 4
    prtlevel = 2;  % default level of printing (shows plot in real abscissa case only)
end
%
% define a polynomial Q(z): note that ordering of coefficients
% is the standard Matlab ordering, with constant entry last.
% The polynomial Q is called h in the BGMO paper.
% If this polynomial has all real roots, the largest is the point
% at which the abscissa optimization problem for the affine polynomial
% family is attained, with a multiple root (Chen, 1979).
% Idea: we want p(s) = (s+z)^n = s^n + n z s^n-1 + ... + n z^n-1 s + z^n
% so imposing the affine constraint defines z by a polynomial Q(z)=0
% (Q is not monic.)
v = convolvebinomials(n,0); 
Q = v .* fliplr(b); 
Qsave = Q;  % for plotting below
% There are several cases.  In the complex case, the roots of Q are all
% we need.  In the real case, for the abscissa, they are all we need if
% they are all real, but if not, we have to supplement them with roots of 
% Q's derivatives. In the real case, for the radius, we look not only
% at the roots of Q but also of some other polynomials, each defined by
% a partition n = n1+n2 (the Q above is the case n1=n, n2=0). We only need
% to consider the cases n1 >= n2, because n1 = k, n2 = n-k gives the
% same result as n1 = n-k, n2 = k.  These polynomials are called g_k in
% the BGMO paper.
if b_real
    count = 0;
    allroots = [];
    countrecord = [];
    n1 = n;  
    n2 = 0;
    % it is NOT OK to terminate when allroots reaches length n, as we
    % found by experiment (counterexample with n=5)
    % In the abscissa case we can stop if all the roots of Q are real,
    % because of the interlacing theorem; apparently this extends to
    % the radius case too, by experimental observation, but its much
    % less likely to occur, because the degree of Q does not drop in
    % this case, but remains equal to n.
    allrootsQreal = -1; %  
    while (~radius & count < n & allrootsQreal < 0)| (radius & n1 >= n2)
        r = roots(Q)'; % conjugation of roots is irrelevant
        indx = find(imag(r) == 0);
        if length(indx) == length(r) & allrootsQreal < 0
            allrootsQreal = count; % sometimes interesting to look at this value
        end
        allroots = [allroots r(indx)];
        countrecord = [countrecord count*ones(1,length(indx))];
        if ~radius
            Q = polyderiv(Q); % derivative of polynomial
        else % see comment above
            n1 = n1 - 1;
            n2 = n2 + 1; % same as count
            Q = convolvebinomials(n1,n2);
            % Q = fliplr(Q) has no effect on roots: same Q=-Q if n2 is odd 
            Q = Q .* fliplr(b); 
        end
        count = count + 1;
    end
    n1 = nan; n2 = nan; % make sure final values not used
    maxroot = max(allroots); 
    minroot = min(allroots);
    if ~radius 
        [optroot,indx] = max(allroots);  % relevant one for abscissa since change sign below
        % countrecord(indx) is the index of the derivative of Q for which optroot 
        % is the rightmost root....called l (ell) in the BGMO paper, and n2 in
        % divergingpoly.m, called below
        optn1 = n - countrecord(indx);   
    else
        [minabsroot,indx] = min(abs(allroots));
        optroot = allroots(indx); % sign is relevant
        optn1 = n - countrecord(indx); % need for reconstructing coefficients
    end
    if ~radius & b_real & prtlevel > 1 % only in the real abscissa case
        % plot in active figure if any
        t = linspace(minroot,maxroot);
        y = polyval(Qsave,t);
        plot(t,zeros(size(t)),'r-') % plot axis first so doesn't overwrite polynomial
        hold on
        plot(t,y)
        title('Polynomial h: stars show real roots of h and its derivatives')
        plot(real(allroots),0,'*')
        plot(max(real(allroots)),0,'o')
        xlabel('rightmost star is at minus the optimal abscissa value')
        hold off
    end
else % complex case is much simpler
    allroots = roots(Q);
    allrootsQreal = nan; % irrelevant
    optn1 = n;
    if ~radius % abscissa case
        [maxrealval,indx] = max(real(allroots));
        optroot = allroots(indx); % the relevant one, because change sign below
    else
        [maxabsval,indx] = min(abs(allroots));
        optroot = allroots(indx); % the relevant one (sign change doesn't change abs val)
    end
end
% Now that we have the root where the optimal abscissa or radius is attained, 
% we can recover the coefficients of the polynomial with this multiple root
if nargout < 2
    % don't need coefficients, nothing to do here
elseif radius | ~radius & optn1 == n % includes complex case 
    v = convolvebinomials(optn1, n - optn1);
    v = v(2:n+1)'; % discard the leading 1 and transpose
    coeff = v .* (optroot .^ ((1:n)')); % don't need n1, n2 here
    coeff = [1 conj(coeff')]; % include monic entry and transpose without conjugating
    % equivalently, could use Matlab's poly
    % rts(1:optn1) = -optroot;
    % rts(optn1+1:n) = optroot;
    % coeff2 = poly(rts); 
    % [coeff; coeff2] % should be the same within rounding error
else % real abscissa case, where optimal value not attained
    % numerical trouble if eps is too small
    done = 0;
    if prtlevel > 0
        fprintf('affpolymin: real abscissa case, optimal value not attained\n')
        fprintf('affpolymin: first trying eps = %g\n', eps)
    end
    while ~done & eps < 1000 % never know how big might need eps to be!
        % if we put one root at -(optroot - eps), we can put
        % the other at -M << 0, M ~ 1/eps
        if prtlevel > 0
            fprintf('affpolymin: putting %d roots near optroot and the others O(1/eps) further left\n', optn1)
        end
        [coeff,M,QQ] = divergingpoly(optn1, n - optn1, optroot - eps, b, prtlevel);
        % check the result, since this process is numerically unstable
        if ~isnan(coeff)
            lambda = roots(coeff);
            [dif,indx] = sort(abs(real(lambda) - optroot)); % increasing order
            % one of the optn1 roots whose real part is closest to optroot
            % should also have the largest real part.  If not, it is probably
            % because one of the roots constructed to be a large negative number
            % has the wrong sign
        else
            lambda = nan;
        end
        if isnan(lambda) | max(real(lambda)) > max(real(lambda(indx(1:optn1))))
            % that did not work: either coeff has nans or the large root must have the wrong sign
            % try again decrementing n1 and incrementing n2: see comments in 
            % divergingpoly and proof of theorem in BGMO paper
            optn1save = optn1;
            optn1 = optn1 - 1;
            if prtlevel > 0
                fprintf('affpolymin: that did not work, instead try putting %d roots near optroot and the others O(1/eps) further left\n', optn1)
            end
            [coeff,M,QQ] = divergingpoly(optn1, n - optn1, optroot - eps, b, prtlevel);
            % check the result AGAIN, since this process is numerically unstable
            if ~isnan(coeff)
                lambda = roots(coeff);
                [dif,indx] = sort(abs(real(lambda) - optroot)); % increasing order
                % one of the optn1 roots whose real part is closest to optroot
                % should also have the largest real part.  If not, it is probably
                % because one of the roots constructed to be a large negative number
                % has the wrong sign
            else
                lambda = nan;
            end
            if isnan(lambda) | max(real(lambda)) > max(real(lambda(indx(1:optn1))))
            % that did not work: either coeff has nans or the large root
            % must have the wrong sign
                if prtlevel > 0
                    fprintf('affpolymin: AGAIN that did not work, increasing eps and trying again\n')
                end
                eps = 10*max([eps (1e-16)^(1/n)]); % in case eps was actually 0
                % restore optn1 in case it was decremented above
                optn1 = optn1save;  
            else
                done = 1;  % worked fine with the increased n1
            end
        else
            done = 1;  % worked fine with the first try of n1
        end
    end
end
optroot = -optroot; % applies to abscissa and complex radius, irrelevant for real radius
M = -M; % applies to real abscissa case, irrelevant otherwise
if radius & b_real
    if optroot < 0
        % change sign and also change n1 accordingly
        optroot = -optroot;
        optn1 = n - optn1;
    end
    if prtlevel > 0
        fprintf('affpolymin: real radius case, putting %d roots at optroot and %d roots at -optroot\n', optn1, n-optn1);
    end
end
if nargout < 2
    return
end
% do not reverse the sign of allroots, as did previously
k = n - optn1;
Q = Qsave;