function [dX,dy,dZ] = sqlsol(A,X,Z,Q,Evsum,Rd,rp,Rc,Lfac,Ufac,Perm,blk,n,qbs,qbi)
% SQLSOL  computes the solution of the 3x3 block linear system
%         for mixed conic programming using Schur complement.
%
% [dX,dy,dZ] = sqlsol(A,X,Z,Q,Evsum,Rd,rp,Rc,Lfac,Ufac,Perm,blk,n)
%
%  input variables
%     - A            structure of constraints data
%     -     A.s      constraint matrix for SD
%     -     A.q      constraint matrix for QC
%     -     A.l      constraint matrix for LP
%     - X            primal variables structure
%     -     X.s      SD primal variable
%     -     X.q      QC primal variable
%     -     X.l      LP primal variable
%     - Z            dual slacks structure
%     -     Z.s      SD dual slack
%     -     Z.q      QC dual slack
%     -     Z.l      LP dual slack
%     - Q            the orthogonal matrix of eigenvectors of the dual slack Z.s
%     - Evsum        the matrix of eigenvalue sums of the dual slack Z.s
%     - Rd           dual residual structure
%     -     Rd.s     SD dual residual
%     -     Rd.q     QC dual residual
%     -     Rd.l     LP dual residual
%     - rp           the primal residual
%     - Rc           centrality residual structure
%     -     Rc.s     SD centrality residual
%     -     Rc.q     QC centrality residual
%     -     Rc.l     LP centrality residual
%     - Lfac         the L factor from the LU factorization of the Schur complement
%     - Ufac         the U factor from the LU factorization of the Schur complement
%     - Perm         the permutation matrix for the LU factorization of the Schur complement
%     - blk          block info structure
%     -     blk.s    block info for SD
%     -     blk.q    block info for QC
%     -     blk.l    block info for LP
%     - n            number of variables structure
%     -     n.sm     SD: size of matrices
%     -     n.sv     SD: length of vector representations
%     -     n.q      QC: length of X.q
%     -     n.l      LP: length of X.l
%     - qbs          vector holding the starting indices of the QC blocks
%     - qbi          a matrix whose ith column is a vector of length n.q with ith
%                    QC block all ones and the other blocks all zeros.
%
%  output variables
%     - dX        primal step structure
%     - dy        y component of the dual step
%     - dZ        slack structure component of the dual step

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 4/20/97
%
 yrhs = rp;
 if n.sm > 0
    yrhs_s = X.s*Rd.s;
    yrhs_s = yrhs_s + yrhs_s' - Rc.s; % essential to compute right-hand side in this
                                      % way: cannot cancel terms (see AHO section 5)
    yrhs_s = Q'*yrhs_s*Q;
    yrhs_s = Q*evsumdiv(yrhs_s,Evsum,blk.s)*Q';
    yrhs = yrhs + A.s*svec(yrhs_s,blk.s);
 else
    dX.s = 0;
    dZ.s = 0;
 end
 if n.q > 0
    yrhs_q = arwmul(X.q,Rd.q,blk.q);
    yrhs_q = arwimul(Z.q,yrhs_q - Rc.q,blk.q);
    yrhs = yrhs + A.q*yrhs_q;
 else
    dX.q = 0;
    dZ.q = 0;
 end
 if n.l > 0
    yrhs_l = (X.l .* Rd.l - Rc.l) ./ Z.l;
    yrhs = yrhs + A.l*yrhs_l;
 else
    dX.l = 0;
    dZ.l = 0;
 end
 dy = Ufac\(Lfac\(Perm*yrhs));
 if n.sm > 0
    dZ.s = Rd.s - smat(A.s'*dy,blk.s);
    Xrhs = X.s*dZ.s;
    Xrhs = Rc.s - (Xrhs + Xrhs');
    Xrhs = Q'*Xrhs*Q;
    dX.s = Q*evsumdiv(Xrhs,Evsum,blk.s)*Q';
 end
 if n.q > 0
    dZ.q = Rd.q - A.q'*dy;
    Xrhs = Rc.q - arwmul(X.q,dZ.q,blk.q);
    dX.q = arwimul(Z.q,Xrhs,blk.q);
 end
 if n.l > 0
    dZ.l = Rd.l - A.l'*dy;
    Xrhs = Rc.l - X.l .* dZ.l;
    dX.l = Xrhs ./ Z.l;
 end
%
% END function
