% DSDP   driver for the function fdsdp
%
%   Semi Definite Programming Solver using either the XZ or the
%   XZ + ZX method for problems with diagonal constraints, namely
%   m = n = sum(blk) and the constraint matrices are of the form
%
%            Ak = Ek = ek ek^T,   1 <= k <= n
%
%   Note: dsdp calls either fdsdp.m or fsdp.m depending upon
%   the value of the flag useXZ; when calling fsdp.m, dsdp must
%   build the matrix A.
%   The following variables must be available in the Matlab workspace:
%      - C            cost matrix
%      - b            rhs of primal constarints
%      - blk          block structure vector
%      - X            initial guess for primal variable
%      - y            initial guess for dual variable
%      - Z            initial guess for dual slack variable
%      - maxit        maximum iterations allowed
%      - tau          fraction of step to boundary of psd cone
%      - steptol      tolerance for short steps
%      - abstol       absolute tolerance on gap
%      - reltol       relative tolerance on gap
%      - gapprogtol   tolerance on sufficient progress
%      - feasprogtol  tolerance on sufficient progress
%      - bndtol       upper bound on size of solution
%      - prtlevel     verbosity level (accessed only by fdsdp.m or fsdp.m)

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> new point rejected since X indefinite: chol(X) failed
%    termflag =  2  ==> new point rejected since Z indefinite:
%                       XZ method: chol(Z) failed or Z is singular
%                       XZ+ZX method: chol(Z) failed or Z has eigenvalues <= 0
%    termflag =  3  ==> XZ method: SchurComp is numerically indefinite or singular
%                       XZ+ZX method: SchurComp is numerically singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X is not positive definite
%    termflag =  2 and iter = 0 ==> initial Z is not positive definite
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 flops(0);
 tstart = clock;
 cputstart = cputime;
%
 tau_orig = tau;
 scalefac_orig = scalefac;
 if useXZ,
    if tau > 0.99
       fprintf('\ndsdp: tau is set to %7.5f\n',tau);
       fprintf('   the XZ method does not perform well with such a high ');
       fprintf('value of tau!\n');
       choice = input('   Do you wish to proceed? <y/n> ','s');
       if choice ~= 'y'
          break;
       end;
    end;
    fprintf('\ndsdp: using XZ method...\n\n');

    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',tau,scalefac);
       [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
           fdsdp(C,b,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
                gapprogtol,feasprogtol,bndtol,prtlevel);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\ndsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & maxit == 0,
             fprintf('\ndsdp: Data passed validation test!\n');
          elseif termflag == 7 & prtlevel == 0,
             fprintf('\ndsdp: Data failed validation tests\n');
             fprintf('For more information, rerun dsdp with prtlevel = 1\n');
          end;
          break;
       end;
       if autorestart & (termflag == 4 | termflag == 5)...
         & (gapval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          tau = min(tau,.9);
          scalefac = 100*scalefac;
          dinitvars;
       else,
          break;
       end;
    end;  % runs
 else,  % use XZ+ZX method: must construct the matrix A
    sparseblocks = 1;
    n = sum(blk);
    n2 = (sum(blk .* (blk+1)))/2;
    A = sparse(n,n2);
    for i = 1:n,
       Ai = sparse(n,n);
       Ai(i,i) = 1;
       A(i,:) = svec(Ai,blk,sparseblocks)';
    end;
    fprintf('\ndsdp: using XZ+ZX method...\n\n');

    for runs = 1:3,
       fprintf('\ntau = %8.4f,     scalefac = %8.0f\n\n',tau,scalefac);
       [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
           fsdp(A,b,C,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
                gapprogtol,feasprogtol,bndtol,prtlevel);
       if iter == 0,
          if termflag == 4 | termflag == 5,
             fprintf('\ndsdp: Initial point may be too close to boundary\n');
          elseif termflag == 6 & maxit == 0,
             fprintf('\ndsdp: Data passed validation test!\n');
          elseif termflag == 7 & prtlevel == 0,
             fprintf('\ndsdp: Data failed validation tests\n');
             fprintf('For more information, rerun dsdp with prtlevel = 1\n');
          end;
          break;
       end;
       if autorestart & (termflag == 4 | termflag == 5)...
         & (gapval(iter+1) + feasval(iter+1,1) + feasval(iter+1,2) > 1.0e-3),
          if runs < 3,
             fprintf('\nrestarting...\n');
          end;
          if runs == 1,
             tau = min(tau,.99);
          elseif runs == 2,
             tau = min(tau,.9);
          end;
          scalefac = 100*scalefac;
          dinitvars;
       else,
          break;
       end;
    end;  % runs
 end;
 tau = tau_orig;
 scalefac = scalefac_orig;
%
 if iter > 0,
    if termflag == 6,
       fprintf('\ndsdp: reached the maximum number of iteration\n');
    end;
    gapval = gapval(1:(iter+1));
    feasval = feasval(1:(iter+1),:);
    objval = objval(1:(iter+1),:);
    elapsedtime = etime(clock,tstart);
    cputend = cputime - cputstart;
    if runs > 1,
       fprintf('\nsdp: number of runs = %3.0f\n',min(runs,3));
    end;
    fprintf('\ndsdp: elapsed time               = %- 9.3f seconds\n',elapsedtime);
    fprintf('dsdp: elapsed cpu time           = %- 9.3f seconds\n',cputend);
    fprintf('dsdp: flops                      = %- 13.0f\n',flops);
    fprintf('dsdp: number of iterations       = %- 4.0f\n',iter);
    fprintf('dsdp: final value of X.Z         = %- 11.3e\n',gapval(iter+1));
    fprintf('dsdp: final primal infeasibility = %- 11.3e\n',feasval(iter+1,1));
    fprintf('dsdp: final dual infeasibility   = %- 11.3e\n',feasval(iter+1,2));
    fprintf('dsdp: primal objective value     = %- 24.16e\n',objval(iter+1,1));
    fprintf('dsdp: dual objective value       = %- 24.16e\n',objval(iter+1,2));
 end;
