function [cndjac,dgap,pinfeas,dinfeas,blockmat] =...
    sdpcond(A,b,C,blk,X,y,Z)
% SDPCOND   computes the condition number of the 3x3 block system
%           for SDP evaluated at the point (X,y,Z)
%
% [cndjac,dgap,pinfeas,dinfeas,blockmat] = sdpcond(A,b,C,blk,X,y,Z)
%
%   The routine also computes the primal and dual infeasibilities
%   at this point as well as the value of <X,Z> allowing the user
%   to "verify" the optimality conditions at (X,y,Z).
%
%   The 3x3 block matrix has the following form
%
%           [ zeros(n2,n2)      A'         eye(n2)   ]
%           [      A        zeros(m,m)   zeros(m,n2) ]
%           [ skron(Z,Id)   zeros(n2,m)  skron(X,Id) ]
%
%   where A is a matrix whose rows are svec of the constraint
%   matrices A1,...,Am, and skron(M,N) denotes the symmetric
%   Kronecker product of the two matrices M, N.
%   The point (X,y,Z) is a solution to the SDP satisfying strict
%   complementarity and primal and dual nondegeneracy if and only
%   if the 3x3 block matrix is nonsingular. Moreover, such a
%   solution is unique. Thus a very large condition number at a
%   point (X,y,Z) satisfying the optimality conditions is a
%   strong indication that the SDP is degenerate.
%   (See AHO for further details).
%
%  input variables
%     - A            matrix of primal constraints
%     - b            rhs of primal constraints
%     - C            cost matrix
%     - blk          block structure vector
%     - X            the computed primal solution
%     - y            the computed dual solution
%     - Z            the computed dual slack at solution
%
%  output variables
%     - cndjac       the condition number of the 3x3 block matrix
%     - dgap         the gap trace(X*Z)
%     - pinfeas      the norm of primal feasibility residual, ||b - A*svec(X)||
%     - dinfeas      the norm of dual feasibility residual, ||C - A*y - Z||
%     - blockmat     the 3x3 block matrix

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
 n = length(X);
 [m,n2] = size(A);
 Id = speye(n);
 vX = svec(X,blk);
 vZ = svec(Z,blk);
 vC = svec(C,blk);
 rp = b - A*vX;
 rd = vC - vZ - A'*y;
 pinfeas = norm(rp);
 dinfeas = norm(rd);
 dgap = full(vX'*vZ);
%
 s1 = 1; f1 = n2;
 s2 = n2+1; f2 = n2 + m;
 s3 = n2 + m + 1; f3 = 2*n2 + m;
 blockmat = sparse(f3,f3);
 idx2 = n2 + m;
 blockmat(s2:f2,s1:f1) = A;
 blockmat(s1:f1,s2:f2) = A';
 blockmat(s1:f1,s3:f3) = speye(n2);
 blockmat(s3:f3,s1:f1) = skron(Z,Id,blk);
 blockmat(s3:f3,s3:f3) = skron(X,Id,blk);
% cndjac = cond(full(blockmat));
% cheaper to use condest since it is not necessary to convert blockmat
% to a full matrix.
 cndjac = condest(blockmat);
%
 fprintf('\nsdpcond: gap                      = %11.3e\n',dgap);
 fprintf('sdpcond: primal infeasibility     = %11.3e\n',pinfeas);
 fprintf('sdpcond: dual infeasibility       = %11.3e\n',dinfeas);
 fprintf('sdpcond: cond estimate of 3x3 block matrix = %11.3e\n',cndjac);
%
% END function
