function [lam,Q,indef] = blkeig(X,blk)
% BLKEIG Computes eigenvalues and eigenvectors of a block diagonal
%        matrix X with block structure blk.
%
% [lam,Q,indef] = blkeig(X,blk)
%
%   nargout = 1 ==> compute eigenvalues lam only
%   nargout = 2 ==> compute eigenvalues lam and eigenvectors Q
%   nargout = 3 ==> compute eigenvalues, eigenvectors, and returns
%                   indef = 1 if X has a nonpositive eigenvalue,
%                   indef = 0 otherwise.
%
% Note: the eigenvalues are returned as a vector!
%
% input variables:
%     - X         a symmetric block diagonal matrix
%     - blk       block structure vector
%
% output variables:
%     - lam       the vector of eigenvalues
%     - Q         the orthogonal matrix of eigenvectors
%     - indef     flag: 0 if X is positive definite, 1 otherwise

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
 nblk = length(blk);
 n = sum(blk);
 lam = zeros(n,1);
 if nblk > 1,
    Q = sparse(n,n);
 else,
    Q = zeros(n);
 end;
 indef = 0;
 fin = 0;
%
 for i=1:nblk,
    start = fin + 1;
    fin = fin + blk(i);
    if nargout >= 2,  % compute eigenvectors
       Z = full(X(start:fin,start:fin));
       [Q(start:fin,start:fin), t] = eig(Z);
       lam(start:fin) = diag(t);
       if nargout == 3,
          min_lam = min(lam(start:fin));
          if min_lam <= 0,
             indef = 1;
             return;
          end;
       end;
    else,   % compute eigenvalues only
       lam(start:fin) = eig(X(start:fin,start:fin));
    end;
 end;   % i loop
%
% END function
