function [x,y,gt_mu,gt_sigma,gt_weight,BackgroundPoints]=CreateMixture4(nPoints, nComponents, mean_variance, sigma_variance, Background)
% Generates 2-D for from a mixture of Gaussians model
%
% Input:
% ------
% nPoints - approx. number of points to generate (1x1 int)
% nComponents - number of components in mixture model (1x1 int)
% mean_variance - randomness in location of component means (1x1 float)
% sigma_variance - size of component covariance matrices (1x1 float)
% Background - portion of points drawn from a uniform distribution (set to zero
% for pure Gaussian points
%
% Output:
% -------
% x - x-coordinate data (1 x nPoints float)
% y - y-coordinate data (1 x nPoints float)
% gt_mu - ground truth means (2 x nComponents float)
% gt_sigma - ground truth covariance matrix (2 x 2 x nComponents float)
% gt_weight - ground truth component weights (1 x nComponents float)
% BackgroundPoints - number of points drawn from uniform (1 x 1 int)
%
% Example:
% --------
% [x,y,gt_mu,gt_sigma,gt_weight,bg_points]=CreateMixture4(1000,3,10,1,0);
%
% Rob Fergus 24/5/05 for VGG reading group
%

DEBUG                  = 1; %% Plot data & distributions
IMAGE_CENTRE           = [5 5]; % Centre of image
RAND_COVARIANCE_MATRIX = 1; % Isotropic covariance matrices (0) or full (1)
SIGMA_DISTORTION       = 5; % Eccentricity of full covariance matrices (only matters if RAND_COVARIANCE_MATRIX = 1)
BG_MARGIN              = 4; % distance beyond extrema of Gaussian points for uniform distrubtion to lie.

cols={'r' 'g' 'b' 'c' 'm' 'y' 'k'};

%%%%% Setup distributions

for a=1:nComponents
   
   posx=((rand*mean_variance)-(mean_variance/2));
   posy=((rand*mean_variance)-(mean_variance/2));

   Mean(a,:)= IMAGE_CENTRE + [posx posy];
   
   Weight(a)=rand;
   
   if RAND_COVARIANCE_MATRIX
      Covariance(:,:,a)=eye(2)* sigma_variance + SIGMA_DISTORTION * RandCovarianceMatrix(2,sigma_variance);
   else
      Covariance(:,:,a)=eye(2) * sigma_variance;
   end
   
end

% Normalize weightition coeffs.
Weight=Weight./sum(Weight);

%%%% Generate data
Data = [];
for b=1:nComponents;
   
   %% How many points for this component?
   dist(b)=round(nPoints*Weight(b));
   %% Draw from Gaussian  
   Data_temp=randnND(Mean(b,:),Covariance(:,:,b),dist(b));
   %%% Concatenate with rest
   Data=[Data,Data_temp];
   
end

% Now do background, uniform distribution
if (Background>0)

   %%% Find edge of data and add in 
   xmin=min(Data(1,:))-BG_MARGIN;
   xmax=max(Data(1,:))+BG_MARGIN;
   ymin=min(Data(2,:))-BG_MARGIN;
   ymax=max(Data(2,:))+BG_MARGIN;
   
   Area=(xmax-xmin)*(ymax-ymin);
   BackgroundPoints=round(Area*Background);

   for c=1:BackgroundPoints
      
      rx=(rand*(xmax-xmin))+xmin;
      ry=(rand*(ymax-ymin))+ymin;
      Rdata=[rx;ry];
   
      Data=[Data,Rdata];
      
   end
else
    BackgroundPoints = 0;
end


%%% Break data down into x and y portions
x=Data(1,:);
y=Data(2,:);

%%% Set output variables
gt_weight = Weight;
gt_mu = Mean';
gt_sigma = Covariance;

if DEBUG
   
   figure(999); clf;
   plot(x,y,'b.');
   hold on;

   
   for c=1:nComponents
      lw = round( 10 * Weight(c) ) + 1;
      [ell_handle(c),xy,L,l,th] = draw_ellipse(Mean(c,:),Covariance(:,:,c),cols{rem(c-1,7)+1},lw,[]);
   end
 
   title('Data with ground truth distribution')
   
end
