import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.atomic.AtomicIntegerArray;
import java.util.concurrent.atomic.AtomicReferenceArray;

/* Fairness: the implementation guarantees fairness by making sure
 * that (1) each room is accessible infinitely often; and (2) each
 * thread that is waiting on a room when it is opened will be able
 * to enter the room before it is closed again.
 */

public class Rooms {
  final int numRooms;
  volatile int currRoom;
  volatile int occupants;
  volatile int waiting;
  volatile boolean sealed;
  AtomicReferenceArray<Handler> handler;
  AtomicIntegerArray waitCount;
  AtomicReferenceArray<Condition> waitRoom;
  Lock lock;
  Condition earlyExit;

  public interface Handler {
    void onEmpty();
  }
  
  public Rooms(int m) {
    numRooms = m + 1;
    // we need a fair lock to ensure that waiting threads will be able to enter a room
    lock = new ReentrantLock(true);
    earlyExit = lock.newCondition();
    currRoom = -1;
    occupants = 0;
    waiting = 0;
    sealed = false;
    handler = new AtomicReferenceArray<Handler>(numRooms);
    waitCount = new AtomicIntegerArray(numRooms);
    waitRoom = new AtomicReferenceArray<Condition>(numRooms);
    for (int i = 0; i < numRooms; i++) {
      waitRoom.set(i, lock.newCondition());
    }
  }

  void seal(int i) {
    waitRoom.set(i, lock.newCondition());
    sealed = true;
  }

  void enter(int i) throws InterruptedException {
    if (i < 0 || i >= numRooms) {
      throw new IllegalArgumentException("tried to enter nonexistant room");
    }
    lock.lock();
    try {
      // first call to enter?
      if (currRoom < 0) {
        currRoom = i;
      }

      // demand for other room?
      if (currRoom != i) {
        // then seal current room
        seal(currRoom);
      }

      if (sealed) {
        // wait in i-th waiting room
        waitCount.incrementAndGet(i);
        waitRoom.get(i).await();
        waiting--;
      }
      occupants++;
      // signal thread that is waiting on an early exit
      earlyExit.signal();
    } finally {
      lock.unlock();
    }
  }

  boolean exit() throws InterruptedException {
    lock.lock();
    try {
      // exiting before waiting threads had a chance to enter?
      if (occupants == 1 && waiting > 0) {
        // then wait for next thread so that room does not become empty
        earlyExit.await();
      }
      occupants--;
      // current room empty?
      if (occupants == 0) {
        // call exit handler
        Handler h = handler.get(currRoom);
        if (h != null) {
          h.onEmpty();
        }
        // open next larger room with waiting threads
        int i = currRoom;
        for (int j = 0; j < numRooms; j++) {
          int k = (i + j + 1) % numRooms;
          int waitingK = waitCount.get(k);
          // does room k have waiting threads?
          if (waitingK > 0) {
            // did we already open another room?
            if (currRoom == i) {
              // no, then open room k
              waiting = waitingK;
              currRoom = k;
              sealed = false;
              waitCount.set(k, 0);
              waitRoom.get(k).signalAll();
            } else {
              // yes, then immediately seal the room we just opened
              // this ensures that all rooms that are on demand will eventually be opened
              seal(currRoom);
              break;
            }
          }
        }
      }
    } finally {
      lock.unlock();
    }
    return true;
  }

  public void setExitHandler(int i, Rooms.Handler h) {
    if (i < 0 || i >= numRooms) {
      throw new IllegalArgumentException("tried to register exit handler for nonexistant room");
    }
    handler.set(i, h);
  }
}

