function [xbest, ybest, ierr] = pspa_real(A, E, y, imagtol, plotfig, xold, ...
                                        eigsolver)
% James Burke, Adrian Lewis, Emre Mengi and Michael Overton
% (Last updated on March 10th 2005)
%
% called by pspa.m
%
% For each component in y, search for rightmost intersection between
% horizontal line with given y component and the pseudospectrum.
% For each j, let us define B2 = A - y(j)*i*eye(n).
% There are 2 ways of posing this as an eigenvalue problem:
% the first: look for REAL eigenvalues of    (we DO NOT use this)
%    [ B2' -E       (this is NOT Hamiltonian)   (E = epsln*I)
%     -E   B2];
% the second: look for IMAGINARY eigenvalues of    (we DO use this)
%    [-(i*B2)'  E       =   [i*B2'   E       which IS Hamiltonian
%     -E      i*B2]           -E    i*B2]       (but complex)
% the two matrices are similar, with eigenvalues that are
% symmetric wrt the imaginary axis only.
% The difference: whether or not multiply by i before applying
% largest singular value operation.
% Return the best x value and a corresponding y value.
% Input parameter xold is used only for plotting, if plotfig > 0

n = length(A);
ierr = 0;

for j=1:length(y)
   B2 = A - y(j)*i*eye(n);

   % use built-in eig of matlab to compute Hamiltonian eigenvalues
   if eigsolver == 0       
       eM2 = eig([i*B2' E; -E   i*B2]);   
   else
       [Am,QG] = haconv(i*B2',E,-E);
       % Use HAPACK
       if eigsolver == 1
           eM2 = haeig(Am,QG);
       % Use SLICOT
       else
           eM2 = Hameig(Am,QG);
       end
   end
       
       

   if min(abs(real(eM2))) <= imagtol % check if M2 has a real eigenvalue
      indx = find(abs(real(eM2)) <= imagtol);  % extract such eigenvalues
      xnew(j) = max(imag(eM2(indx)));
      
      if plotfig > 0
          figure(plotfig)
          plot([xold xnew(j)], [y(j) y(j)], 'm-')  % plot horizontal line
          plot(xnew(j), y(j), 'b+')       % plot right end point
      
          if isreal(A)
              plot([xold xnew(j)], -[y(j) y(j)], 'm-')  % plot horizontal line
              plot(xnew(j), -y(j), 'b+')    % plot right end point
          end
      end
      
      
   else
       % Normally horizontal search should return an intersection point
       % for each midpoint, however because of rounding errors it failed
       % to do so.       
       xnew(j) = -inf;
   end % end of else
   
end % end of for

if ~isfinite(max(xnew))
    error('pspa_real: no intersection points found for one of the midpoints (please set epsln larger)')
end

[xbest,ind] = max(xnew);     % furthest to right, returns 1 index
ybest = y(ind);              % corresponding y value

% mark the rightmost intersection point
if plotfig > 0 & xbest > xold
   plot(xbest, ybest, 'b*')
end

return;