function [f, z, ierr, iwar] = pspa(A, epsln, prtlevel, plotfig, keyb, eigsolver)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  PSPA, pseudo spectral abscissa
%%  Copyright (C) 2005  James Burke, Adrian Lewis, Emre Mengi and Michael Overton
%%
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (Last updated on March 10 2005)
%
% call:  [f, z] = pspa(A, epsln, prtlevel, plotfig, keyb, eigsolver)
%  Quadratically convergent criss-cross method to compute
%  the eps-pseudospectral abscissa of A.
%  
% DEFINITION:
%   The epsln-pseudospectral abscissa of A is the globally optimal value of: 
%      max Re z
%      s.t. sigma_min(A - z I) = epsln     (smallest singular value)
%
% METHOD:
% The method in [2] is implemented. At each iteration given an estimate
% alpha first a vertical search finds the intersection points of the
% vertical line y=alpha and the epsln-pseudospetrum boundary. From the
% intersection points segments on the line lying inside epsln-pseudospectrum
% are determined. From the midpoint of each segment a horizontal search
% returns the rightmost intersection point of the pseudospectrum boundary and
% the horizontal line passing through the midpoint. The estimate alpha
% is refined to the largest value returned by the horizontal searches.
% Horizontal and vertical searches can be performed by solving Hamiltonian
% eigenvalue problems (for details see [2]).
%
% note: In windows user has the flexibility to choose the Hamiltonian
%       eigenvalue solver by setting the parameter eigsolver appropriately.
%       In all other platforms set eigsolver = 0, or don't specify it so
%       that Matlab's eig is used.
%
%  input
%     A         square matrix
%     epsln     real >= 0 (0 implies f is spectral abscissa)
%     prtlevel  1: chatty; 0: prints only if problems arise (default)
%     plotfig   0: no plot, otherwise figure number for plot
%     keyb      0: no keyboard mode(default), otherwise after each 
%		           iteration enter keyboard mode
%     eigsolver Hamiltonian eigenvalue solver to be used.
%               0: Matlab's eig (tolerances are needed to accept an 
%                  eigenvalue imaginary; Default)
%               1: HAPACK routine haeig (The eigenvalue symmetry preserving 
%                  method by Benner, Mehrmann and Xu in [1]; tolerances are 
%                  not needed).
%               2: SLICOT routine hameig (The square reduced method of Van
%                  Loan [2]; preserves the eigenvalue symmetry, but half of 
%                  the precision may be lost in the worst case; tolerances 
%                  are not needed).
%              
%  output
%     f         the epsln-pseudospectral abscissa of A
%     z         one of the global maximizers except when
%               the matrix is real. If the input matrix is real 
%               it contains either a global optimizer on the
%               real axis or a complex conjugate pair.
%     ierr      error flag. 0 : success.
%                           1 : Too many iterations (didn't converge).
%                           2 : Spectral abscissa cannot be improved.
%                           3 : Vertical search failed. (Odd vertical
%                               intersection points detected.)
%                           4 : Horizontal search failed. (No imaginary
%                               eigenvalue of the Hamiltonian matrix is
%                               extracted.)
%     iwar      a structure of warning flags.
%               iwar.eps : 1 if epsilon is too small
%                          0 otherwise
%               iwar.sval: 1 if singular value test removed all of the
%                            vertical intersection points
%                          0 otherwise
%   
%
%    [1] P. Benner, V. Mehrmann and H. Xu. A numerically stable, 
%        structure preserving method for computing the eigenvalues 
%        of real Hamiltonian or symplectic pencils. Numerische 
%        Mathematik, 78(3):329-358,1998.
%    [2] J.V. Burke, A.S. Lewis and M.L. Overton. Robust stability 
%        and a criss-cross algorithm for pseudospectra. IMA Journal
%        of Numerical Analysis, 23:359-375,2003.
%    [3] C.V. Loan A symplectic method for computing all eigenvalues
%        of a Hamiltonian matrix, Linear Algebra and Its Applications,
%        61, 1984, 233-251.


A = full(A);

if nargin <= 5
    eigsolver = 0;
end

if nargin <= 4
    keyb = 0; % default: no user intervention
end

if nargin <= 3 
   plotfig = 0;     % default: no plot
end

if nargin <= 2 
   prtlevel = 0;    % default: no printing
end






if isnaninf(A) | isnaninf([epsln prtlevel plotfig keyb])
   error('pspa: nan or inf arguments not allowed')
end
if epsln < 0 | imag(epsln) ~= 0    
   error('pspa: epsln must be nonnegative real')
end

n = length(A);
if size(A) ~= [n n]
   error('pspa: A must be square')
end

ierr = 0;
iwar.eps = 0;
iwar.sval = 0;

% draw the eigenvalues of A
eA = eig(A);
if plotfig > 0
   figure(plotfig);
   plot(real(eA), imag(eA), 'ko')
   hold on
end



smalltol = 1e-8*max(norm(A),epsln);

if epsln == 0  % pseudospectrum is just the spectrum
   [sortreal, indx] = sort(-real(eA));
   eA = eA(indx);

   % return pure spectral abscissa
   f = real(eA(1));
   
   % and the eigenvalues with real part equal to the spectral abscissa
   ind = find(real(eA) >= f - smalltol);
   z = eA(ind);
      
else   % much faster than bisection, but just as reliable        
        
   if (epsln < 10^-9)
       % epsilon is too small
       iwar.eps = 1;
   end
    
   xold = -inf;
   [x,ind] = max(real(eA));  % initial iterate
   y = imag(eA(ind));
   if prtlevel > 0
      fprintf('pspa: x = %22.15f \n', x);
      fprintf('initial estimate is calculated \n');
   end
   iter = 0;
   no_imageig = 0;
   ybest = [];
   E = epsln*eye(n);
   Areal = isreal(A);

   % imagtol is used to detect zero real parts
   if eigsolver == 0
       imagtol = smalltol; 
   else
       imagtol = 0;
   end

   while ~no_imageig & x > xold    % x is increasing in exact arithmetic
      iter = iter + 1;
      if iter > 20         
          % too many steps
          ierr = 1;
          f = -1;
          z = -1;
          return;
      end
      
      % given current x, look for all relevant intersections of vertical 
      % line with the pseudospectrum, process and return pair midpoints.
      % note: input x is a scalar, but output y is a vector.
      % the input ybest is a scalar.
      [y,imagtol,ierr,iwar] = pspa_imag(A, E, epsln, x, ybest, iter, imagtol, ...
          plotfig, prtlevel, eigsolver, iwar);
      
      
      if ierr 
          f = -1;
          z = -1;
          return;
      end
      
      
      if keyb ~= 0
          keyboard
      end
      
      no_imageig = isempty(y);
      if ~no_imageig
         if prtlevel > 0
            fprintf('pspa: y = %22.15f \n', y);    % y may be a vector
         end
         % given the resulting y values, look for rightmost intersection of
         % the corresponding horizontal lines with the pseudospectrum
         % note: input y is a vector, but output x is a scalar: the max
         % of the max values.  ybest is the corresponding y value and is
         % a scalar, even if there was a tie e.g. from a complex conjugate pair.
         xold = x;
         ybestt = ybest;
         [x, ybest, ierr] = pspa_real(A, E, y, imagtol, plotfig, xold, eigsolver);

         if ierr
             f = -1;
             z = -1;
             return;
         end

         if x < xold
            x = xold;    % causes while loop to terminate
            ybest = ybestt;

            if prtlevel > 0
               fprintf('pspa: could not find bigger x \n')
            end
         end % end of if x < xold
         
         
         if prtlevel > 0
            fprintf('pspa: x = %22.15f  \n', x);
            fprintf('pspa: iteration %d is completed \n', iter);
         end
      end % end of if ~no_imageig
   end % end of while
   
   
   if isempty(ybest)
       % failed at the first iteration
       ierr = 2;
       f = -1;
       z = -1;
       return;
   end


   f = x;
   z = x+i*ybest;
   if isreal(A) & ~isreal(z)
      z = [z; z'];
   end
   

   
end % end of else


return;