function [z, x, y, f] = geteigvecs(A, Atype, scalefac, v, u, mod, k, extra)
% get eigenvalue and right and left eigenvectors
% get kth largest (wrt real part if mod == 0, wrt modulus if mod == 1) 
% eigenvalue z and corresponding right and left eigenvectors x and y
% of B = A + scalefac*u*v', with ||x||=||y||=1, x'y > 0
% if Atype == 1, A is dense so use dense matrix operations
% if Atype == 2, A is sparse matrix
% if Atype == 3, A is function handle
% for last two cases, use function handle "aplusuvt" to compute products
%             (A + scalefac*u*v')*p  
% extra is the number of extra eigenvalues to compute in eigs, to help
% to ensure finding the desired eigenvalue.  This may be useful for example
% if imag(A) is small but nonzero so there are nearly conjugate eigenvalues
% as we want to ensure that the eigenvalues found for A and its transpose
% are nearly the same, not nearly conjugate
%
% note the order of the input arguments v, u.  
% this is because v is a starting estimate for x and u is the same for y
% also return f, spectral abscissa or radius (wrt real part if mod==0,
% wrt to modulus if mod == 1): this is not the same as real(z) or abs(z)
% when k > 1

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  PSAPSR 1.2 Copyright (C) 2010 Nicola Guglielmi, Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nargin < 6
    mod = 0; % default is max real part
end
% normally k=1, except that it may be desirable to initiate iteration at
% more than one eigenvalue
if nargin < 7
    k = 1;
end
n = length(u); % length(A) does not work if A is a function handle
% eigenvalues of A+scalefac*u*v'
if Atype == 1 % A is dense, so compute all the eigenvalues
    B = A + scalefac*u*v';
    [X, Lambda] = eig(B);
    lambda = diag(Lambda);
    % Y=inv(X) bad numerically for e.g. original A=triu(rand(75))
    % use transpose, not conjugate transpose, to be consistent with next part 
    [Y, Lambdaleft] = eig(B.'); % transpose, not conjugate transpose
    lambdaleft = diag(Lambdaleft);
else
    opts.disp = 0; % for eigs
    opts.isreal = isreal(A) & isreal(scalefac) & isreal(u) & isreal(v); % for eigs, when Atype== 3
    if ~mod 
        eigscode = 'LR'; % want eigenvalue(s) with largest real parts
    else 
        eigscode = 'LM'; % want eigenvalue(s) with largest magnitudes
    end
    % experiments show that the cpu time for the right eigenvectors is
    % substantially more than for the left eigenvectors, and that this is
    % true even if the original A is replaced by A'
    % is this because the data structures for representing A favor
    % multiplying by A' not by A?
    %
    % first the right eigenvectors
    Afun = @(p)aplusuvt(A, Atype, scalefac, u, v, 0, p);
    if norm(v,1) > 0 
        opts.v0 = v;  % v is the previous x, except at initial iterate where it may be 0
    end
    % [X, Lambda] = eigs(Afun, n, k + extra, eigscode, opts); 
    % Tim Mitchell's modification to eigs eliminates the horrible
    % possibility of eigs returning a vector whose largest real part is
    % zero which is not genuine but the result of failure to converge!
    [X, lambda] = eigsplus(Afun, n, k + extra, eigscode, opts); 
    if isempty(lambda)
        error('geteigvecs: eigs found no eigenvalues to default accuracy')
    end
    % then the left eigenvectors: right eigenvectors of TRANSPOSED matrix
    % cannot use conjugate transposed matrix and conjugate eigenvalues later,
    % because then "largest" eigenvalue of a conjugate pair will be the wrong one!!!
    Atransfun = @(p)aplusuvt(A, Atype, scalefac, u, v, 1, p);
    if norm(u,1) > 0
        opts.v0 = conj(u);  % conj(u) is the previous y, see below
    end
    % [Y, lambdaleft] = eigs(Atransfun, n, k + extra, eigscode, opts); 
    [Y, lambdaleft] = eigsplus(Atransfun, n, k + extra, eigscode, opts); 
    if isempty(lambdaleft)
        error('geteigvecs: eigs found no eigenvalues to default accuracy')
    end
end
% lambda = diag(Lambda); (for eigs; eigsplus returns a vector of eigenvalues)
% sort eigenvalues: they are not sorted by eig, and although they are by eigs,
% conjugate pairs are sorted inconsistently so resort
[lamsort, indx] = sortcomplex(lambda, mod);
z = lamsort(k); % same as lambda(indx(k))
x = X(:,indx(k)); % corresponding eigenvector
% lambdaleft = diag(lambdaleft); (for eigs, not eigsplus)
% look for entry of lambdaleft that is closest to z
[eigdif,j] = min(abs(lambdaleft - z));
% the standard notation is that a left eigenvector of a matrix M corresponding to
% eigenvalue lambda is is the right eigenvector of the complex conjugate transpose 
% matrix M' corresponding to the conjugate eigenvalue lambda'.  See, for example,
% Horn and Johnson, and Stewart (but not Wilkinson, who uses transpose
% instead of complex conjugate transpose).  It would have been convenient
% to follow this convention in the computation above, but when eigs is used
% this results in computation of the wrong eigenvalue in a conjugate pair.
% Hence, conjugation was avoided up until this point, but now we must
% conjugate the left eigenvector to comply with standard notation.
y = conj(Y(:,j));
if eigdif > 1e-8*max(abs(lambda))
    warning(sprintf('geteigvecs: eigenvalues of B and its transpose differ by %g', eigdif))
end
if ~mod
    theta = 0;  % for defining RPZ-compatible
else
    theta = angle(z);
end
[x,y] = rpcompatible(x,y,-theta); % so x,y are RPZ-compatible wrt exp(i*theta)
% the following ensures monotonicity when options.monotone is 1
% no change from version 1.01 except to add this comment
xtv = x'*v;
ytu = y'*u;
psi = (1-ytu*real(ytu))*(xtv)' + (1-(xtv)'*real(xtv))*ytu;
% fprintf(' real(psi) is %g\n', real(psi))
if real(psi) < 0
    x = -x;
    y = -y;
end
% finally, also return spectral abscissa or radius (same as real(z) or abs(z) if k == 1)
zbig = lamsort(1);
if ~mod
    f = real(zbig);
else
    f = abs(zbig);
end