function [f, x, y, z, info, xrec, yrec, zrec, trec, cpurec] = psapsr(A, epsln, mod, options)
% PSAPSR: Pseudospectral Abscissa or Pseudospectral Radius 
%
%  f = PSAPSR(A, epsln) normally returns
%    the epsln-pseudospectral ABSCISSA of the matrix A defined by
%      max {real(z): z is an eigenvalue of B with ||A-B|| <= epsln}
%    and is guaranteed to be a lower bound.  A can be dense or sparse.
%
%  f = PSASR(Afun, epsln) uses a function handle Afun instead of A. 
%    Afun must do matrix vector products with both A and A', as follows:
%    the function call Afun(p,t) returns A*p if t==0, A'*p if t==1.
%    For example, set Afun=@(p,t)amult(p,t) where the routine amult
%    returns A*p if t==0, A'*p if t==1. In all usages below, Afun
%    can be substituted for A.
%
%  f = PSAPSR(A, epsln, 1) normally returns
%    the epsln-pseudospectral RADIUS of the matrix A defined by
%      max {abs(z): z is an eigenvalue of B with ||A-B|| <= epsln}
%    and is guaranteed to be a lower bound.
%    A is either a matrix (usually sparse), or a function handle as
%    described above.
%
%  [f, x, y, z] = PSAPSR(A, epsln, mod) also returns column vectors x, y 
%   and scalar z satisfying: ||x||=||y||=1, 0 < y'x <= 1, 
%   epsln is the smallest singular value of A - zI, with corresponding 
%   right and left singular vectors x and -y, so (A - zI)x = -epsln y,
%   and x,y are right, left eigenvectors of B = A + epsln y*x', 
%   so Bx = zx,  y'B = zy'. 
%   When mod=0 (abscissa case, default), z is the rightmost point found 
%    in the epsln-pseudospectrum, and p = real(z).
%   When mod=1, (radius case), z is the largest modulus point found 
%    in the epsln-pseudospectrum, and p = abs(z).
% 
%  [f, x, y, z, info, xrec, yrec, zrec, trec, cpurec] = PSAPSR(A, epsln, mod) 
%  (abscissa case) also returns info:
%    0: normal return; 1: max iterations exceeded; 2: line search failure
%   as well as the record of all computed x, y and z iterates, the final 
%   values of t, and the CPU time/iteration.

%  [f, x, y, z] = PSAPSR(A, epsln, options) or 
%  [f, x, y, z] = PSAPSR(A, epsln, mod, options)
%   allows specification of the following options:
%  
%   options.maxit: max number of iterations (default 100)
%   options.ftol:   termination tolerance based on f (default 1e-8)
%   options.vectol: termination tolerance based on eigenvectors (default 0)
%   options.kstart: start iteration at kth rightmost or 
%     kth largest-modulus eigenvalue (default 1)
%   options.x0, options.y0: when A is a sparse matrix, or when A is
%     a function handle for computing matrix vector products A*v:
%     initial guesses for right and left eigenvectors for rightmost 
%     or largest-modulus eigenvalue of A (must be column vectors with
%     length the dimension of A).  These will be used for the first
%     call to "eigs".  At least one must be provided when  A is a 
%     function handle in order to specify the dimension of A, 
%     but it may be the zero vector if no estimate is known
%     (default: 0, which will be replaced by random starting vectors)
%   options.extra: number of additional eigenvalues to compute in "eigs", 
%     to help ensure finding the rightmost or largest-modulus eigenvalue.
%     May be good to set this to 1 if the imaginary part of the matrix
%     is tiny but not zero, since then there will be nearly conjugate 
%     eigenvalues, and want to ensure the eigenvalues found for A and its 
%     transpose are nearly the same, not nearly conjugates. 
%     Ignored if A is dense as then all eigenvalues are computed.  
%     (default: 0, for efficiency)
%   options.monotone: 1 for monotonically increasing iterates (default)
%                        (Algorithms PSA1 and PSR1 of paper)
%                     0 for basic iteration: no guarantees
%                        (Algorithms PSA0 and PSR0 of paper)
%   options.prtlevel: print level 
%     0: no output; 1: minimal output (default); 2: verbose output
%
%   Primarily intended for large sparse A or other matrix A for which
%   matrix-vector products A*v, A'*v are cheap.  Can also be used for 
%   small dense A, but in that case a more reliable code can be found at
%   http://www.cs.nyu.edu/mengi/robuststability.html
%
%   Reference: Nicola Guglielmi and Michael Overton, Fast Algorithms 
%    for the Approximation of the Pseudospectral Abscissa and 
%    Pseudospectral Radius of a Matrix, submitted to SIAM J Mat Anal Appl
%   Send comments/bug reports to Michael Overton, overton@cs.nyu.edu,
%   with a subject header containing the string "psapsr".
%   Version 1.01, 2011, see GPL license information below.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  PSAPSR 1.01 Copyright (C) 2011 Nicola Guglielmi, Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if nargin < 2
    error('both A and epsln are required input parameters')
end
if strcmp(class(A), 'double')
    n = size(A,1);
    if size(A,2) ~= n
        error('the matrix A must be square')
    end
    if ~issparse(A)
        Atype = 1; % A is dense
    else
        Atype = 2; % A is sparse
    end
elseif strcmp(class(A), 'function_handle')
    Atype = 3; % A is a function handle for computing products A*w
else
    error('A must either be a matrix or a function handle')
end
if ~strcmp(class(epsln), 'double') || epsln < 0 || imag(epsln) ~= 0
    error('epsln must be nonnegative')
end
if nargin < 3
    mod = 0; % no 3rd arg: default objective is spectral abscissa
elseif isstruct(mod)
    options = mod;  % 3rd arg can be used for options if mod omitted
    mod = 0;
elseif isempty(mod); % means options = []
    options = [];
    mod = 0;
elseif (mod ~=0 && mod ~= 1)
        error('mod must be 0 or 1')
end
if nargin < 4  && ~exist('options','var') % options not assigned to 3rd arg
    options = [];  % note: mod cannot be 4th argument
end
if isfield(options, 'maxit')
    maxit = options.maxit;
else
    maxit = 100;
end
if isfield(options, 'ftol')
    ftol = options.ftol;
else
    ftol = 1e-8; % absolute and relative tolerance on f 
end
if isfield(options, 'vectol')
    vectol = options.ftol;
else
    vectol = 0; % 0 means turned off
end
if isfield(options, 'kstart')
    kstart = options.kstart;
else
    kstart = 1;
end
if isfield(options, 'extra')
    extra = options.extra;
else
    extra = 0;
end
if isfield(options, 'monotone')
    monotone = options.monotone;
else
    monotone = 1; % guarantees monotonicity
end
if isfield(options, 'x0')
    x0 = options.x0;
    if size(x0,2) ~= 1
        error('options.x0 must be a column vector')
    end
    if exist('n','var')
        if length(x0) ~= n
            error('length of options.x0 is not compatible with size of A')
        end
    else
        n = length(x0);
    end
else
    if exist('n','var')
        x0 = zeros(n, 1); % if not, hold off until we see if options.y0 specified
    end
end
if isfield(options, 'y0')
    y0 = options.y0;
    if size(y0,2) ~= 1
        error('options.y0 must be a column vector')
    end
    if exist('n','var')
        if length(y0) ~= n
            error('length of options.y0 is not compatible with size of A or length of options.x0')
        end
    else
        n = length(y0);
        if ~exist('x0','var')
            x0 = zeros(n,1); % now that we know what n is
        end
    end
else
    if exist('n','var')
        y0 = zeros(n, 1);
    else
        error('options.x0 or options.y0 must be provided when A is a function handle')
    end
end
if isfield(options, 'prtlevel')
    prtlevel = options.prtlevel;
else
    prtlevel = 1;
end
if prtlevel > 0
    if ~mod
        fprintf('Computing pseudospectral abscissa: n = %d, epsln = %g, Atype = %d, monotone = %d\n', n, epsln, Atype, monotone)
    else
        fprintf('Computing pseudospectral radius: n = %d, epsln = %g, Atype = %d, monotone = %d\n', n, epsln, Atype, monotone)
    end
end
if extra > n-3
    extra = max([0, n-3]); % otherwise get warning from eigs
end
%
% compute right and left eigenvectors x0, y0 
% for kstart-th largest real part or largest modulus eigenvalue z0 of A,
% normalized so ||x0||=||y0||=1 and y'x is real and positive
% setting scalefac to 0 means that epsln*y0*x0' is not added to A, 
% but x0, y0 are still used as initial guesses for eigenvectors of A if Atype 
% is 2 or 3, unless they are zero.
scalefac = 0;
% we also need spectral abscissa or radius to check if first step is any good
cputime0 = cputime; % cputime is built in function
[z, x, y, f] = geteigvecs(A, Atype, scalefac, x0, y0, mod, kstart, extra); 
% from now on we will not use kstart: we will only want the largest
% eigenvalue wrt to real part or modulus
t = 0; % there is no t at initial iterate
for iter = 1:maxit
    if prtlevel > 1
        fprintf('iter %d, f = %g   \n', iter, f)
    end
    if nargout > 4 % don't preallocate since don't expect many iterations
        zrec(iter) = z; 
        xrec(:,iter) = x;
        yrec(:,iter) = y;
        cputime1 = cputime;  
        trec(iter) = t;
        cpurec(iter) = cputime1 - cputime0;
        cputime0 = cputime1;
    end
    if epsln == 0 % cannot move before loop in case nargout > 4
        % pseudospectrum is just the spectrum
        info = 0;
        return
    end
    % choose u, v to maximize derivative of spectral absicssa or spectral radius 
    %  of B + t(u*v' - y*x')
    scalefac = epsln;
    [znew,xnew,ynew,fnew] = geteigvecs(A, Atype, scalefac, x, y, mod, 1, extra); 
    % termination check based on the eigenvector
    if iter > 1 && vectol > 0 
        if Atype == 1 
            yxdif = max(max(abs(ynew*xnew' - y*x')));
        else
            xrat = xnew./x;
            yrat = ynew./y;
            xrat = xrat/xrat(1);
            yrat = yrat/yrat(1);
            one = ones(n,1);
            yxdif = norm(xrat - one) + norm(yrat - one);
        end
        if yxdif < vectol
            if prtlevel > 0
                fprintf('terminating: yxdif = %g at iteration %d\n', yxdif, iter)
            end
            info = 0; % normal termination
            break % out of iteration
        end
    end
    % termination check based on the function value
    if iter > 1 && abs(fnew - f) < ftol*max([1 abs(f)])
        if prtlevel > 0
            fprintf('terminating: fnew - f = %g at iteration %d\n', fnew - f, iter)
        end
        info = 0; % normal termination
        break % out of iteration
    end
    % if there is no monotonicity check, do special check for first iteration
    if ~monotone && iter == 1
        ytx = y'*x; 
        if fnew < f + epsln*ytx
            % initial step is no good: swap x and y so eigenvalue variation is  linear
            % then the abscissa increases by at least epsln on the first step
            if prtlevel > 0
                fprintf('insufficient increase at first step: swapping yx* for xy*\n')
            end
            % in the next call, y and x are swapped
            [znew,xnew,ynew,fnew] = geteigvecs(A, Atype, scalefac, y, x, mod, 1, extra); 
            if fnew < f + epsln*ytx
                error('something went wrong with the swap')
            end
        end
    elseif monotone % bisect t if necessary
        t = 1;
        while fnew < f
            t = t/2;
            if prtlevel > 0
                fprintf('f decreased by %g so reduce t to %g\n', f-fnew, t)
            end
            if t < 0.001
                if prtlevel > 0
                    fprintf('terminating since t is small: may have reached limits of precision\n')
                end
                info = 2;
                return
            end
            x = t*xnew + (1-t)*x;
            x = x/norm(x);
            y = t*ynew + (1-t)*y;
            y = y/norm(y);
            [znew,xnew,ynew,fnew] = geteigvecs(A, Atype, epsln, x, y, mod, 1, extra);
        end
    end  
    % check monotonicity
    if fnew < f
        if iter == 1
            error('decrease in f should not happen since swap should have taken place, see above')
        elseif monotone
            warning('f decreased, but there should be a monotone increase in f')
        end
        if prtlevel > 0 % could happen otherwise
            fprintf('f decreased by %g at iteration %d\n', f-fnew, iter)
        end
    end
    % at next step we will compute eigenvalues of A + epsln*ynew*xnew'
    x = xnew;
    y = ynew;
    z = znew;
    f = fnew;
end
if iter == maxit
    info = 1;
    if prtlevel > 0
        fprintf('reached %d iterations, stop, f = %g\n',maxit, fnew)
    end
end
if nargout > 4
    zrec = [zrec znew];
    xrec = [xrec xnew];
    yrec = [yrec ynew];
    cpurec = [cpurec cputime - cputime0];
end
if ~mod
    f = real(znew); % not z, if tolerance was satisfied
else
    f = abs(znew);
end