function [mu, g, J] = getmugj(method, U, S, V, options)
% called from newtonsaddle, compute target function g for Newton's method,
% and its Jacobian J (although this is not needed when called from line
% search, it does not pay to take advantage of this, since computation of
% J is trival compared to cost of SVD). 
% There are 3 different functions for the 3 different methods;
% see comments in newtonsaddle.m.  Also get mu, the least squares
% estimate for the multiplier (mu = 0 if method == 2).
%
% The formulas for the gradients of the singular values are trivial but
% the Hessians are more complicated, and there are two different versions
% (the first version has two different files for smallest and second smallest
% singular values, but the second one handles both cases). 
% This option is not documented in neardefmat comments.
% The default is to use the second version, and so the other codes
% fgh_sigma_smallest and fgh_sigma_secondsmallest neardefmat are not
% distributed with neardefmat.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  NEARDEFMAT 1.0 Copyright (C) 2010  Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isfield(options,'fghversion')
    fghversion = 2;
else
    fghversion = options.fghversion;
end

tinytol = options.tinytol;
if method == 2 % g is in R^2
    mu = 0; % no multiplier required
    % only need to look at smallest singular value gradient and Hessian
    if fghversion == 1
        [fN, gN, HN] = fgh_sigma_smallest(U, S, V);
    else
        n = length(S);
        [fN, gN, HN] = fgh_sigma_general(U, S, V, n);
    end
    g = gN;
    eigHN = eig(HN);
    J = HN; % Hessian of smallest singular value
    % considered shifting HN by a multiple of the identity to ensure
    % that it is indefinite, since we want to find a saddle point, not
    % a min or a max.  Trouble with this is that the modified Newton
    % direction is no longer guaranteed to be a descent direction for g'g
    return
end
% methods 1 and 3 need the gradient and Hessian of smallest two singular values
if fghversion == 1
    [fN, gN, HN] = fgh_sigma_smallest(U, S, V);
    [fNm1, gNm1, HNm1] = fgh_sigma_secondsmallest(U, S, V); 
else
    n = length(S);
    [fN, gN, HN] = fgh_sigma_general(U, S, V, n);
    [fNm1, gNm1, HNm1] = fgh_sigma_general(U, S, V, n-1);
end
% for methods 1 and 3, g is in R^3 and differ from each other only in 
% the 3rd equation
fdif = fNm1 - fN;
gdif = gNm1 - gN;
% compute least squares multiplier estimate for mu in the overdetermined
% linear system mu*gNm1 + (1-mu)*gN = 0 (a complex equation with one real
% variable)
if norm(gdif) < tinytol % happens in case n=2: gNm1 = gN = [0 0]'
    mu = 0;
else
    mu = -gdif\gN; % avoid normal equations -gdif'*gN/(gdif'*gdif);
end
if mu < 0  % project onto [0,1]
    mu = 0;
elseif mu > 1
    mu = 1;
end
if method == 1
    g = [mu*gNm1 + (1-mu)*gN;      % first two equations
         mu*fdif];                 % third equation
    J = [mu*HNm1 + (1-mu)*HN   gdif  % 3 by 3 Jacobian matrix
         mu*gdif'              fdif];
else % method 3
    g = [mu*gNm1 + (1-mu)*gN;      % first two equations
         fdif];                 % third equation
    J = [mu*HNm1 + (1-mu)*HN   gdif  % 3 by 3 Jacobian matrix
         gdif'                   0];
    % if norm(HN) + norm(HNm1) > 1e8*norm(J)
    %    fprintf('getmugJ: big cancellation in J!\n')
    % end
end
% svals = diag(S)'
% J
