function [B, z, u, v, dist, other] = neardefmat(A, options)
%  NEARDEFMAT  Nearest defective matrix.
%   B = NEARDEFMAT(A) returns a defective matrix that is near A in the
%   spectral and Frobenius norms: hopefully, a nearest defective matrix,
%   which is also a nearest matrix with multiple eigenvalues.
%
%   [B, z, u, v, dist, other] = NEARDEFMAT(A, options) takes input options
%   and returns quantities used to construct B = A - dist*u*v', where u, v
%   are unit left and right singular vectors for A - zI, with u'*v = 0.
%   By construction z is a double eigenvalue of B corresponding to left
%   and right eigenvectors u and v.  This is because z is a point 
%   where two of A's pseudospectral components, that is connected 
%   components of the pseudospectral set
%         {w: smallest singular value of A-wI <= dist}, 
%   coalesce; equivalently z = x+iy, where (x,y) is a saddle point of 
%          f(Re w,Im w) = smallest singular value of A - wI.
%   This saddle point is a smooth saddle point in the generic case but a 
%   nonsmooth saddle point in special cases. B is the nearest defective
%   matrix to A when dist is the smallest value such that coalescence
%   occurs; equivalently, z is a lowest saddle point.
%   The final output argument provides additional quantities.
%
%   input:
%     A: square matrix
%     options: structure (optional):
%       options.nsearch: number of saddle point searches 
%         (between 1 and n(n+1)/2, default is n, any number > n(n+1)/2
%         (including inf) is interpreted to mean n(n+1)/2), 
%         ignored if options.zinit is provided)
%       options.zinit: vector of user supplied starting points (complex 
%         numbers) for saddle point searches (if not provided, searches 
%         are initiated at options.nsearch weighted averages of eigenvalue 
%         pairs, in increasing order of likely success based on eigenvalue
%         conditioning measures)
%       options.stoptol: stopping tolerance for saddle point searches
%         (default: 1e-10*||A||). 
%       options.tinytol: tiny tolerance used for multiple purposes
%         (less than options.stoptol, default: 1e-14*||A||)
%       options.bigtol: used to decide whether saddle point search failed;
%          if so, try again with two searches explicitly specifying smooth
%          and nonsmooth cases in advance
%          (bigger than options.stoptol, default 1e-6*||A||) 
%       options.maxit: maximum number of iterates per saddle point search
%         (default: 30)
%       options.minstep: minimum step allowed in line search used for 
%         every iterate of saddle point searches before reporting failure 
%         (default: 1e-3)
%       options.prtlevel: print level: one of
%         0 (none), 
%         1 (basic info printed, default), 
%         2 (saddle point info printed), 
%         3 (all iterates printed),
%         4 (saddle points are also plotted in active window), or 
%         5 (all iterates are also plotted in active window)
%          (in these cases, user may wish to call EigTool in advance)
%   output: 
%     B: hopefully, the defective matrix that is nearest to A, though 
%        this is not guaranteed.  Unless something goes wrong, B should
%        be a defective matrix, modulo rounding errors.
%     z: double eigenvalue of B
%     u, v: corresponding left and right eigenvectors of B with u'*v = 0
%     dist: norm(A-B) (spectral and Frobenius norm); dist is the smallest
%        singular value of A - zI, and B is constructed as A-dist*u*v',
%        where u and v are corresponding unit singular vectors of A - zI
%     other: additional output quantities:
%       other.upperbnd: known upper bound on distance, possibly less
%         than dist in which case B cannot be nearest defective matrix
%       other.utv:  u'*v, 0 should be close to 0
%       other.uresidual: ||u'*B - z*u'||, should be close to 0
%       other.vresidual: ||Bv - z*v||, should be close to 0
%       other.mu: multiplier that is 0 if z is a smooth saddle point 
%         of smallest singular value (the generic case), in [0,1] if it
%           is a nonsmooth saddle point, and equal to 0.5 if A is normal
%       other.gnorm: norm of system of equations defining saddle point
%       other.Jcond: condition number of Jacobian of this nonlinear system
%
%   Reference: R.Alam, S. Bora, R.Byers, M.L. Overton, 
%    Characterization of and Construction of the Nearest Defective 
%    Matrix via Coalescence of Pseudospectra,  submitted to LAA,
%    June 2009
%   Send bug reports and comments to overton@cs.nyu.edu with subject 
%   header "neardefmat".
%   Version: 1.0, June 2010

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  NEARDEFMAT 1.0 Copyright (C) 2010  Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Goal is to return the nearest defective matrix to A.  Alam and Bora
% showed that this distance is the smallest real number eps for which two 
% components of the eps-pseudospectrum coalesce.  If z is a point of
% coalescence, then the smallest singular value of A-zI equals eps.
% There are two cases.  In the generic case, the smallest singular 
% value is simple, so its left and right singular vectors u and v
% are unique up to signs and the nearest defective matrix is 
% A - eps uv'.  In this case z is a saddle point of the locally
% smooth function f(Re w, Im w) = sigma_min(A - wI), and u'*v is zero.  
% In the nongeneric case, in particular when A is normal, the smallest
% singular value is double, and the nearest defective matrix 
% is again A - eps uv', where u and v are now chosen so that u'*v = 0.
% There are two possible choices of u and v; only one is returned.
% In this case, we call z a nonsmooth saddle point. 

% Algorithm used:
%  Compute eigenvalues and their condition numbers and use 
%  these to get initial estimates, one for each eigenvalue pair. 
%  Use these to construct options.nsearch initial points, unless
%  these are provide by user in options.zinit. Search for saddle points 
%  of sigma_min(A - . I), using a Newton iteration that combines both
%  smooth and nonsmooth cases. Construct the (hopefully) nearest
%  defective matrix from the lowest saddle point that was found.
%  There is no guarantee that the algorithm will find the lowest
%  saddle point, but any z that it returns IS an approximate
%  saddle point (smooth or nonsmooth), and consequently any B that it 
%  returns is nearly defective, though not necessarily close to the 
%  nearest defective matrix.
%
% R. Alam, B. Bora, R. Byers and M. Overton 
% Work began in Berlin, February 2004, thanks to V. Mehrmann.

[n, m] = size(A);
if n ~=m 
    error('neardefmat: input A is not square')
end
if any(any(isnan(A) | isinf(A)))
    error('neardefmat: input A contains a nan or inf')
end
% used for setting options
normA = norm(A);  
if nargin < 2 
    options = [];
end
if ~isfield(options, 'nsearch')
    options.nsearch = n;
elseif options.nsearch < 1
    options.nsearch = 1;
end 
nsearch = options.nsearch; % inf is OK, will not give infinite loop
if ~isfield(options, 'stoptol')
    options.stoptol = 1e-8*normA; % setting too small risks a decent answer being rejected
end
stoptol = options.stoptol;
if ~isfield(options, 'tinytol')
    options.tinytol = 1e-14*normA;
end
tinytol = options.tinytol;
if ~isfield(options, 'bigtol')
    options.bigtol = 1e-6*normA;
end
bigtol = options.bigtol;
if ~isfield(options, 'maxit')
    options.maxit = 30;
end 
% maxit not used in this routine
if ~isfield(options, 'minstep')
    options.minstep = 1e-3;
end
% minstep not used in this routine
if ~isfield(options, 'prtlevel')
    options.prtlevel = 1;
end
prtlevel = options.prtlevel;
% default output
u = nan*ones(n,1);
v = nan*ones(n,1);
other.upperbnd = nan;
other.utv = nan;
other.uresidual = nan; 
other.vresidual = nan;
other.mu = nan;
other.gnorm = nan; 
other.Jcond = nan;
if n == 0 % empty case: no nearest defective matrix
    if prtlevel > 0
        fprintf('neardefmat: A is empty, distance is infinite\n')
    end
    B = A; dist = inf; z = nan;   
    return
end
if n == 1 % scalar case: no nearest defective matrix
    if prtlevel > 0
        fprintf('neardefmat: A is a scalar, distance is infinite\n')
    end
    B = nan; dist = inf; z = nan;   
    return
end
%
% first compute eigenvalues and their condition numbers
[evec, Lambda, p] = condeig(A); % built in function 
indx = find(isnan(p));  % nan condition numbers should be inf
p(indx) = inf;          % otherwise ratios will be nan instead of inf
lambda = diag(Lambda);
% compute ratios of pairwise eigenvalue separation to associated
% sum of eigenvalue condition numbers
k = 0;
for i=1:n 
    for j=i+1:n
        k = k+1;
        eigdif(k) = abs(lambda(i) - lambda(j)); 
        ratio(k) = eigdif(k)/(p(i) + p(j));
        rowind(k) = i;
        colind(k) = j;
    end
end
% half the smallest distance between eigenvalue pairs 
[halfeigsep, kmin] = min(eigdif/2);  % note the factor of 2 
% check the normal case BEFORE checking the multiple eigenvalue case,
% because when a matrix satisfies both, the former check, which gives
% mu = 0.5, is more satisfactory
if max(p) < 1 + tinytol    % the normal case
    dist = ratio(kmin);  % same as half eigenvalue distance except that tinytol > 0
    if prtlevel > 0
        fprintf('neardefmat: A is normal within tolerance\n')
        fprintf(' distance to nearest defective matrix B is %g\n', dist)
    end
    imin = rowind(kmin);
    jmin = colind(kmin);
    z = (lambda(imin) + lambda(jmin))/2;
    % the following is overkill in the normal case since we already 
    % computed the eigenvalue decomposition, but it works
    % I = eye(n); [U, S, V] = svd(A - z*I);
    % [u,v]=numrange(U(:,n-1),U(:,n),V(:,n-1),V(:,n),prtlevel);
    % instead, compute u, v directly from eigenvectors
    if dist == 0 % else clause divides by dist
        if prtlevel > 0
            fprintf('neardefmat: A has multiple eigenvalues, returning B = A\n')
        end
        B = A;
        % not sure how to set u and v in this case, default is nans which
        % is OK
    else
        edif = lambda(imin) - lambda(jmin);
        theta = edif'/abs(edif); % note the conjugate
        u = sqrt(0.5)*evec(:,imin) + sqrt(0.5)*evec(:,jmin); % eigenvectors were computed above
        v = theta*(sqrt(0.5)*evec(:,imin) - sqrt(0.5)*evec(:,jmin));
        B = A - dist*u*v';
        other.utv = u'*v;
        other.uresidual = norm(u'*B - z*u');
        other.vresidual = norm(B*v - z*v);
        other.mu = 0.5;  % gradients of the two smallest singular vectors are equal and opposite
        if prtlevel > 0
            neardefmatfinish(B);
        end
    end
    if dist > 0 & prtlevel > 3
        plot(real(z),imag(z),'m*')
    end
    return
end
% compute lower and upper bounds on the distance
[lowerbnd,upperbnd] = wilkbds(evec, Lambda, p, normA);
other.lowerbnd = lowerbnd; % for returning values
other.upperbnd = upperbnd;  
% check for nearly multiple eigenvalues or very badly ill conditioned
% eigenvalues - in this case rounding errors may prevent the Newton
% approach from working so we just use the average of the closest eigenvalues
if upperbnd <= tinytol % multiple eigenvalue case
    if halfeigsep <= tinytol & prtlevel > 0
        fprintf('neardefmat: A has multiple eigenvalues within tolerance\n')
    elseif upperbnd <= tinytol & prtlevel > 0
        fprintf('neardefmat: upper bound is less than tolerance: A probably has an eigenvalue with huge condition number\n')
    end
    % set z to midpoint of the two very close eigenvalues and get dist, u and v
    % from SVD
    imin = rowind(kmin);
    jmin = colind(kmin);
    z = (lambda(imin) + lambda(jmin))/2; % these differ by at most 2*tinytol
    [U,S,V] = svd(A - z*eye(n));
    dist = min(S(n,n));
    u = U(:,n);
    v = V(:,n);
    B = A - dist*u*v';  % should be very close to A
    other.utv = u'*v;
    other.uresidual = norm(u'*B - z*u');
    other.vresidual = norm(B*v - z*v);
    return
end
if prtlevel > 0
    fprintf('neardefmat: lower bound on distance is %g and upper bound is %g\n', lowerbnd, upperbnd)
end    
if isfield(options,'zinit')
    zinit = options.zinit;
    if any(isnan(zinit) | isinf(zinit))
        error('neardefmat: options.zinit contains a nan or inf')
    end
    nsearch = length(zinit); % instead of options.nsearch
    if prtlevel > 0 
        fprintf('neardefmat: using user-provided initial points for saddle point searches\n')
    end
else
    % smallest ratios yield best starting estimates for saddle point searches
    [ratiosort,indx] = sort(ratio); 
    Areal = isreal(A);    
    % run through nsearch eigenvalue pairs i,j with the smallest ratios
    k = 0; % indexes eigenvalue pairs
    kk = 0; % indexes starting points
    while k < n*(n-1)/2 & kk < nsearch
        k = k + 1;
        i = rowind(indx(k));
        j = colind(indx(k));
        % get starting estimate: condition numbers are all finite and > 1
        z0 = (p(j)*lambda(i) + lambda(j)*p(i))/(p(i) + p(j));
        % if A is real do not use starting points in lower half plane 
        if ~Areal | Areal & imag(z0) > -tinytol % need tolerance since p(i) and p(j) may
        % differ by rounding error even if eigenvalues are conjugate
            kk = kk + 1;
            zinit(kk) = z0;
        end
    end
end
nsearch = length(zinit); % after potentially discarding some if A is real
% saddle point searches
for k=1:nsearch % now k indexes starting points
    z0 = zinit(k);
    if prtlevel > 2
        fprintf('neardefmat: saddle point search #%g, z0 = %g + %g i\n', k, real(z0), imag(z0))
    end
    [z, f, u, v, opt, mu, gnorm, Jcond, localmax] = newtonsaddle(1, A, z0, options);
%   do not discard saddle point just because gnorm is not sufficiently
%   small: this can be a bad idea, e.g. the example foo2 with maxit=15 and
%   stoptol=1e-7*norm(A).  Instead, return the final residual to the user
%   as suggested by Ralph and described in the paper: this is the minimum
%   of two quantities, a smooth and a nonsmooth optimality measure computed
%   in newtonsaddle
% if newtonsaddle failed with method 1 (that is, the better residual is bigger than bigtol)
% try methods 2 AND 3
    if ~(opt < bigtol) % checking if opt > bigtol fails if opt == nan
%      method 1 is designed to work in both the smooth AND nonsmooth cases,
%      but if it fails, better try the methods that are explicitly intended
%      for each of these cases.  Not enough to try, say, just the method for the smooth
%      case (method 2), because in the nonsmooth case, this may go off to a
%      smooth saddle point that is far away even though there is a
%      nonsmooth saddle point nearby
        if prtlevel > 2
            fprintf('neardefmat: newtonsaddle with method 1 failed, so try methods 2 AND 3\n')
        end
        [z1, f1, u1, v1, opt1, mu1, gnorm1, Jcond1, localmax1] = newtonsaddle(2, A, z0, options);
        if ~(opt1 < bigtol)
            f1 = inf;
            if prtlevel > 2
                fprintf('neardefmat: rejecting method 2 as opt > bigtol or opt is nan\n')
            end
        end
        [z2, f2, u2, v2, opt2, mu2, gnorm2, Jcond2, localmax2] = newtonsaddle(3, A, z0, options);
        if ~(opt2 < bigtol)
            f2 = inf;
            if prtlevel > 2
                fprintf('neardefmat: rejecting method 3 as opt > bigtol or opt is nan\n')
            end
        end
        if f1 < f2
            z = z1; f = f1; u = u1; v = v1; opt = opt1; mu = mu1; gnorm = gnorm1; Jcond = Jcond1; localmax = localmax1;
        else 
            z = z2; f = f2; u = u2; v = v2; opt = opt2; mu = mu2; gnorm = gnorm2; Jcond = Jcond2; localmax = localmax2;
        end
    end
    if prtlevel > 1
        if f < inf
            fprintf('neardefmat: saddle found at z = %g + %g i with f = %g, ||g|| = %g, mu = %g\n', ...
              real(z), imag(z), f, gnorm, mu)
        else
            fprintf('neardefmat: no saddle point found from this starting point\n')
        end
    end
    if prtlevel > 3 & f < inf
        plot(real(z),imag(z),'ro')
    end
    saddle(k) = z; sigmin(k) = f; mult(k) = mu; gnm(k) = gnorm; Jcd(k) = Jcond;
    leftvec{k} = u; rightvec{k} = v; locmax{k} = localmax; optmeas{k} = opt;
end % kth starting point for saddle point searches
% now look at all the saddle points found
[dist,kmin] = min(sigmin);   % minimum of all saddle values
if dist == inf
    if options.nsearch < n*(n-1)/2 % not nsearch, which may have been reduced to eliminate conjugate starting points
        error('neardefmat: failed to find a saddle point:\n try increasing options.nsearch')
    else
        error('neardefmat: failed to find a saddle point:\n please email matrix to overton@cs.nyu.edu with subject line "neardefmat failed"')
    end
else
    z = saddle(kmin); 
    u = leftvec{kmin}; v = rightvec{kmin};
    localmax = locmax{kmin};
    opt = optmeas{kmin};
    B = A - dist*u*v';
    Ashift = A - z*eye(size(A));
    other.utv = u'*v;
    other.uresidual = norm(u'*Ashift- dist*v'); % singular vector residuals
    other.vresidual = norm(Ashift*v - dist*u);
    other.optmeasure = opt; % this is the sum of the previous 3
    other.mu = mult(kmin); 
    other.gnorm = gnm(kmin); 
    other.Jcond = Jcd(kmin);
    if prtlevel > 0
        fprintf('neardefmat: finished, lowest saddle found is %g + %g i with\n', real(z), imag(z))
        fprintf(' dist = %g, mu = %g, |u''*v| = %g\n',  dist, other.mu, abs(other.utv))
        fprintf(' singular vector residual norms are %g, %g\n',other.uresidual, other.vresidual)
        fprintf(' this saddle point was found in search # %d\n', kmin); 
        if localmax
            fprintf(' WARNING: apparently this is a local max, not a saddle point\n')
        end
        fprintf('distance to nearest defective matrix B found is %g\n', dist) 
        if dist > upperbnd*(1 + tinytol) + tinytol;  % do not print warning if margin is negligible
            fprintf('*** since this is greater than upper bound %g, B cannot be the nearest defective matrix\n', upperbnd)
            fprintf('try increasing options.nsearch and/or reducing options.stoptol\n')
        end
        if dist < lowerbnd*(1 - tinytol) - tinytol;  % do not print warning if margin is negligible
            fprintf('since this is less than lower bound %g, B cannot be the nearest defective matrix\n', lowerbnd)
            fprintf('***** something must be wrong!  is this just rounding error?\n')
        end
    end
end
if prtlevel > 0
    neardefmatfinish(B);
end
if prtlevel > 3
    plot(real(z),imag(z),'bo')
end