% GNSD computes the generalized null space decomposition of a matrix A.
% 
% The generalized null space (GNS) decomposition of a matrix A of order
% n is a decomposition of a form that is sufficiently well illustrated
% by the following special case
% 
%    B = V'*A*V =  0  B12  B13  B14
%                  0   0   B23  B24
%                  0   0    0   B34
%                  0   0    0   B44
% 
% where
% 
%    1. V is unitary,
% 
%    2. B12 and B23 are of full column rank,
% 
%    3. B44, if it is not null, is nonsingular,
% 
%    4. The diagonal block submatrices of B are square, and, with
%       with the exception of B44, their orders are nonincreasing.
% 
% If V = (V1, V2, V3, V4) is partitioned conformally with B, then
% V1, V2, and V3 span the generalized null spaces of A, that is
% 
%         A^{i-1}*Vi is of full rank and A^i*Vi = 0.
% 
% The matrix B (and hence A) has the property that 3 is the smallest
% integer k for which rank(A^k) = rank(A^(k+1)) and is called the index
% of A.  The sequence gamma1, gamma2, gamma3 of the dimensions of the
% generalized null spaces is nonincreasing.
% 
% The calling sequence for gnsd is 
% 
%    function [B, V, index, gamma, bs, enrms] = gnsd(A, tol)
% 
%    Input:
% 
%       A    The matrix whose GNS decomposition is to be computed.
% 
%       tol  A tolerance used to determine in the course of
%            the decomposition 
% 
%               1. when a normalized v of a submatrix C of B can be
%                  regarded as an approximate null vector (specifically
%                  when norm(Cv) <= tol) or
% 
%               2. when an element of B can be regarded as zero.
% 
%    Output:
% 
%       B, V  The matrices of the decomposition.
% 
%       index The index of B.    
% 
%       gamma An row vector of length index containing the dimensions of
%             the generalized null spaces of B.
% 
%       bs    A block structure row vector of size nbs whose i-th
%             entry is the index of first column B1i and hence the first
%             row of Bj1.  The last entry of bs is n+1.
%             The nbs is index+1 if A is nilpotent and index+2
%             if not.
%
%       enrms An optional a row vector containing norm(A - V*B*V'),
%             norm(A - V*BB*V'), norm(eye(n) - V'*V), where BB
%             is B before the block lower triangular part is set
%             zero.
%            
% This function was used to run the test cases in the paper 
% AN EFFICIENT ALGORITHM FOR COMPUTING THE GENERALIZED NULL SPACE 
% DECOMPOSITION by NICOLA GUGLIELMI, MICHAEL L. OVERTON, AND G. W. STEWART
% SIAM J. MATRIX ANAL. APPL. 36, No. 1, pp. 38-54, 2017
% There are two minor differences between the paper and the code.  
% First, the array gamma in the code corresponds to mu in the
% paper.  Second, in the paper the reduction is described in terms of
% the matrix B, which is initialized to A.  In the code the reduction is
% performed in the array A and the result transferred to the array B at
% the end.
%
% NOTE: This version has been recoded to handle complex matrices.  The
% major difference is new versions of routines to generate and apply
% complex plane rotations.  See the functions grotgen and grotapp.
% The previous 2014 version, which was posted in the supplementary 
% materials tacitly assumed that A was real.
%
% Coded by Pete Stewart, 2014, 2017

function [B, V, index, gamma, bs, enrms] = gnsd(A, tol)

% Error checking.     

if any(any(isnan(A))) || any(any(isinf(A))) 
    error('A contains a NaN or Inf')
end
n = size(A, 1); 
if n ~= size(A, 2)    
    error('A must be square')
end
if ~isreal(tol) || tol < 0
    error('tol must be nonnegative')
end

% Initialize.

gamma = [];
[Q, R] = qr(A);
V = eye(n); 
if nargout == 6, AA = A; end


k = 1;

while(true)

%  Start reducing R(k:n,k:n) to determine a generalized null space.

   for j = k:n

%     Try to find a null vector and quit if not found.

      [v, norm_rv] = approx_nvlp(R(j:n,j:n));
      if norm_rv > tol, break; end; % no null vector found

%     Reduce v to e1 preserving the upper triangularity of R
%     and accumulating the transformations in Q and A.

      v = [zeros(j-1,1); v];
      for i = n:-1:j+1
         [v(i-1), v(i), VV] = grotgen(v(i-1), v(i));
         [R(1:i,i-1), R(1:i,i)] = grotapp(VV, R(1:i,i-1), R(1:i,i), 11);
         [V(:,i-1), V(:,i)] = grotapp(VV, V(:,i-1), V(:,i), 11);
         [A(:,i-1), A(:,i)] = grotapp(VV,A(:,i-1), A(:,i), 11);
         [A(i-1,:), A(i,:)] = grotapp(VV, A(i-1,:), A(i,:), 00);
         [Q(i-1,:), Q(i,:)] = grotapp(VV, Q(i-1,:), Q(i,:), 00);
         [R(i-1,i-1), R(i,i-1),U] = grotgen(R(i-1,i-1), R(i,i-1));
         [R(i-1,i:n), R(i,i:n)] = grotapp(U, R(i-1,i:n), R(i,i:n), 00);
         [Q(k:n,i-1),Q(k:n,i)] = grotapp(U,  Q(k:n,i-1),Q(k:n,i), 11);
      end

%     Zero the first row of R and accumulate the transformations
%     in Q.
     
      for i=j+1:n
         [R(i,i), R(j,i), W] = grotgen(R(i,i), R(j,i));
         [R(i,i+1:n), R(j,i+1:n)] = grotapp(W, R(i,i+1:n), R(j,i+1:n), 00);
         [Q(k:n,i),Q(k:n,j)] = grotapp(W, Q(k:n,i),Q(k:n,j), 11);
      end
   end

%  No more null vectors decide what to do next.

   if j==n && norm_rv < tol    % A is nilpotent.  Quit.
      gamma = [gamma, j-k+1];
      break;

   elseif k==j  % Finished with gns determination.  Quit.
      break;

   else    % Start determining the next gns by downdating
           % the QR factorization.

      gamma = [gamma, j-k];
      for i=k:j-1
         [Q(i+1:n,j:n), R(j:n,j:n)] = ddqrrow(Q(i:n,j:n), R(j:n,j:n));
      end
      k=j;
   end
end

% Finished.  Set up the output.

index = max(size(gamma));
B = A;
bs(1) = 1;
for j = 1:index
   bs(j+1) = bs(j) + gamma(j);
   B(bs(j):n, bs(j):bs(j+1)-1) = 0;
end
if bs(index+1) ~= n+1, bs(index+2) = n+1; end
nbs = max(size(bs));

if nargout == 6
   enrms = [norm(AA - V*A*V'),norm(AA - V*B*V'), norm(eye(n) - V'*V)];
end

