function [tolall, index_all, gamma_all, jbnum_all] = gnsdplot(A,allblocks,fig)
%
% [tolall, index_all, gamma_all, jbnum_all] = gnsdplot(A,allblocks,fig)
% compute the generalized null space info for many different tolerances and
% plot the computed index. 
% if allblocks is 1, extend plot to indicate presence of Jordan blocks of
% size less than the index too
%
% Don't forget to subtract off lambda*I from A before calling this if
% interested in the structure for a nonzero eigenvalue!
%
% This was used to generate Figure 4 in:
%
% AN EFFICIENT ALGORITHM FOR COMPUTING THE GENERALIZED NULL SPACE DECOMPOSITION
% NICOLA GUGLIELMI, MICHAEL L. OVERTON, AND G. W. STEWART
% SIAM J. MATRIX ANAL. APPL. 36, No. 1, pp. 38-54, 2015
%
% Written by Michael Overton, November 2014
% Calls GNSD, written by Pete Stewart.
%
if nargin < 2
    allblocks = 0;
end
if nargin < 3
    fig = 1;
end
fsize = 18; % font size for plots
warning off % otherwise lots of warnings re nearly singular matrix
tolall = logspace(-17,0,200); % 10^(-17) to 10^(0)
for j=1:length(tolall)
    % tol_tri(j)
    [B, V, index, gamma, bs, enrms] = gnsd(A,tolall(j));
    index_all(j) = index; % estimated index (max Jordan block size) using tolerance j
    % kth entry of gamma_all{j} is estimated number of Jordan blocks of size >= k using tolerance j
    gamma_all{j} = gamma; % dims of generalized null spaces
    % kth entry of jbnum_all{j} is estimated number of Jordan blocks of size == k using tolerance j
    jbnum_all{j} = -diff([gamma,0]); 
end
figure(fig)
clf
if ~allblocks % just plot the index as a function of the tolerance
    semilogx(tolall,index_all,'*')
    title('GNSD: index','FontSize',fsize)
    xlabel('tolerance \tau','FontSize',fsize)
    ylabel('computed index \nu','FontSize',fsize)
else % plot all the blocks that appear as a function of the tolerance
    biggest_block = max(index_all);
    blk = zeros(length(tolall),biggest_block);
    % extend plot to indicate presence of Jordan blocks of size less than the index too
    for j = 1:length(tolall)
        q = 0;
        for k = index_all(j) : -1 : 1;
            if jbnum_all{j}(k) > 0
                q = q + 1; % a new block size to record
                blk(j,q) = k;
            end
        end
    end
    % colors are descending from index: blue for index, green for next largest size Jordan block present, etc
    semilogx(tolall,blk,'o') 
    title('GNSD: show all block sizes that are present','FontSize',fsize)
    xlabel('tolerance \tau','FontSize',fsize)
    ylabel('sizes of all Jordan blocks that are present','FontSize',fsize)
end
ax = axis;
ax(1) = tolall(1);
ax(2) = tolall(end); % change x axis end points
set(gca,'YTick',[ax(3):ax(4)]) % don't change y axis end points, but make ticks only at integer values
set(gca,'FontSize',fsize)
axis(ax)