function [D, nu, gamma, resid_norms] = drazin(A,tol)
% 
% Compute the Drazin inverse of A using the algorithm in Appendix A of
% AN EFFICIENT ALGORITHM FOR COMPUTING THE GENERALIZED NULL SPACE 
% DECOMPOSITION by NICOLA GUGLIELMI, MICHAEL L. OVERTON, AND G. W. STEWART
% SIAM J. MATRIX ANAL. APPL. 36, No. 1, pp. 38-54, 2015
%
% The Drazin inverse is the unique matrix D satisfying
%   D*A^{nu+1} = A^nu, DAD = D, AD - DA
% where nu is the index of A, which is one of the outputs of the 
% Generalized Null Space Decomposition (GNSD). Appendix A explains how
% to compute A by solving a Sylvester equation after computing the GNSD
% to within the tolerance requested by tol.
%
% inputs:
%   A: input matrix 
%   tol: tolerance for computing the GNSD (default 1e-8)
% outputs:
%   D: the approximate Drazin inverse of A, given the input tolerance
%   nu: the estimated index of A, given the input tolerance
%   gamma: the estimated dimensions of the generalized null spaces of A, given the input tolerance
%   resid_norms: ||D*A^{nu+1} - A^nu||,||DAD - D||, ||AD - DA||
%
% Written by Michael Overton in Oct 2017. Calls the gnsd code written
% by Pete Stewart in 2013 and revised in 2017 (to handle complex matrices).

[n,ncols] = size(A);
if n ~= ncols
    error('A must be square')
end
if nargin < 2
    tol = 1e-8;
end
if tol <= 0
    error('tol must be positive')
end
%
% compute the GNSD, for which if there was no tolerance the outputs would satisfy:
%   B is the GNSD satisfying V'*A*V = B where V is unitary
%   nu is the index of A and of B (dimension of its largest Jordan block)
%   gamma is the vector of dimensions of the generalized null spaces of A and of B
% see the gnsd mfile and the paper mentioned above for more info
%
[B, V, nu, gamma, bs] = gnsd(A,tol); 
%
% let q be the sum of the dimensions of the generalized null spaces
%
q = sum(gamma); % this is the same as bs(end - 1) - 1 if A has been determined by the GNSD
                % to be nilpotent and otherwise it is bs(end) - 1
%
% partition the GNSD as B = [N  L] 
%                           [0  M]
% where N has order q by q, L is q by n-q, and M is n-q by n-q.
% 
N = B(1:q,1:q); 
L = B(1:q,q+1:n);
M = B(q+1:n,q+1:n);
%
% Solve the Sylvester equation KM - NK = L using a built-in Matlab code
%
K = sylvester(-N,M,L);
%
% compute the Drazin inverse of [N  L]
%                               [0  M]
Minv = inv(M);
NLM_drazin = [zeros(q,q),  K*Minv
              zeros(n-q,q),Minv];
%
% finally, compute the Drazin inverse of A
%
D = V*NLM_drazin*V';      % V is unitary
resid_norms = [norm(D*A^(nu+1) - A^nu),norm(D*A*D - D), norm(A*D - D*A)];     
