function runExamplesFromPaper
% run affpolymin on Examples 1 to 9 of 
%   V.D. Blondel, M. Gurbuzbalaban, A. Megretski and M.L. Overton,
%    Explicit Solutions for Root Optimization of a Polynomial Family with
%    One Affine Constraint, submitted to IEEE Trans Auto Control, 2010
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  AFFPOLYMIN 1.0 Copyright (C) 2010  Mert Gurbuzbalaban, Michael Overton
%%  This program is free software: you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation, either version 3 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program.  If not, see <http://www.gnu.org/licenses/>.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
close all
%%
%Example 1
fprintf('Example 1: Find the infimal abscissa for \n');
fprintf('P = {z^6 + a_1 z^5 + a_2 z^4 + a_3 z^3 + a_4 z^4 + a_5 z^5 | a_1 + a_2 = 0, a_i complex}\n')
% use i instead of 1 in order to optimize over complex coefficients
[optroot, coeff] = affpolymin([0 i i 0 0 0 0],0); 
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('Coefficients of the optimal polynomial [1,a_1,a_2,...] are: \n'))
coeff
fprintf('Computed roots are\n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 2
display('Example 2: Find the infimal abscissa for');
fprintf('P = {(s^4 + 2 s^2)(x_0 + x_1 s + s^2 ) + y_0 + y_1 s + y_2 s^2 | x_0, x_1, y_0, y_1, y_2 reals }\n')
fprintf('This problem comes from fixed-order linear controller design, see D. Henrion and M. L. Overton \nMaximizing the closed loop asymptotic decay rate for the two-mass-spring control problem. \nTechnical Report 06342, LAAS-CNRS, March 2006. http://homepages.laas.fr/henrion/Papers/massspring.pdf\n')
fprintf('By a change of variables, the problem may be rewritten as :\n')
fprintf('P = {s^6 + a_1 s^5 + a_2 s^4 + a_3 s^3 + a_4 s^4 + a_5 s^5 + a_6 | 2 a_1 - a_3 = 0, a_i real }\n')
[optroot, coeff] = affpolymin([0 2 0 -1 0 0 0],0);
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('Coefficients of the optimal polynomial [1,a_1,a_2,...] are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 3
display('Example 3: Find the infimal abscissa for')
fprintf('P = {(s^2 -2 delta s + 1)(s^3 + w_2 s^2 + w_1 s^1 + w_0 s^0) + (s^2-1)w_3 | w_0,w_1,w_2,w_3 complex}\n')
fprintf('This problem is derived from a Belgian chocolate stabilization problem.\n')
fprintf('If delta is < (sqrt{2+sqrt{2}})/2 (approx 0.924), optimal abscissa lies on the left hand plane and stabilization is possible.\nFor details, see Burke,Henrion,Lewis,Overton Stabilization via nonsmooth, nonconvex optimization. IEEE Trans Auto Control, 2000 \n')
delta = [];
while(length(delta)==0)
    delta = input('Enter a value for delta: ')
end
%    w0 w1 w2 w3
T = [0 0 1 0 ;       % s^4
    0 1 -2*delta 0;  % s^3
    1 -2*delta 1 1;  % s^2
    -2*delta 1 0 0;  % s
    1 0 0 -1];       % 1
v=null(T');
b = [ -[-2*delta 1 0 0 0]*v, v'];
[optroot, coeff] = affpolymin(b,0);
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('Coefficients of the optimal polynomial [1,a_1,a_2,...] are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 4
display('Example 4: Find the optimal root radius for')
fprintf('P = {z^2 + a_1 z + a_2 | 1 + a_1 + a_2 = 0, a_i real}\n')
[optroot, coeff] = affpolymin([1 1 1 ],1);
fprintf('affpolymin computes infimal root radius = rho^* = optroot where')
optroot
display(sprintf('Coefficients of the optimal quadratic [1,a_1,a_2] are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 5
display('Example 5: Find the infimal root abscissa for')
fprintf('P = {z^2 + a_1 z + a_2 | a_1 = 0, a_2 real}\n')
[optroot, coeff] = affpolymin([0 1 0 ],0);
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('Coefficients of the optimal quadratic [1,a_1,a_2] are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 6
display('Example 6: Find the infimal root abscissa for')
fprintf('P = {z^2 + a_1 z + a_2 | a_2 = -1, a_1 real}\n')
[optroot, coeff, k] = affpolymin([1 0 1 ],0);
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('This is not attained, but coefficients of a quadratic [1,a_1,a_2] with 1 root near optroot and one a large negative number are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 7
display('Example 7: Find the infimal root abscissa for')
fprintf('P = {z^3 + a_1 z^2 + a_2 z + a_3 | a_3=-1, a_i reals }\n')
[optroot, coeff] = affpolymin([1 0 0 1],0);
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('This is not attained, but coefficients of a cubic [1,a_1,a_2,a_3] with 1 root near optroot and two large negative numbers are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 8
display('Example 8: SIMO static output feedback example from Anderson, Bose and Jury')

fprintf('P= {(z^3-13z) + (z^2 - 5z)w_1 + (z+1)w_2 ~|~ w_1,w_2 real or complex }\n')

%    w1 w2
T = [1  0;       % z^2
    -5  1;       % z
     0  1];      % 1
v=null(T');
b = [ -[0 -13 0]*v, v'];
[optroot, coeff] = affpolymin(b,0);
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('Coefficients of the optimal polynomial [1,a_1,a_2,...] are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 9
display('Example 9: SISO frequency domain example from Spanos, Milman and Mingori')
fprintf('P = {(s^4+5s^3+33s^2+79s+50)(s^2 + w_1 s + w_2) + (s^2+15s+50)(w_3 s^2 +w_4 s +w_5): w in R^5\n')
%    w1 w2 w3 w4 w5
T = [ 1  0  0  0  0 ;    % s^5
      5  1  1  0  0 ;    % s^4
     33  5 15  1  0 ;    % s^3
     79 33 50 15  1;     % s^2
     50 79  0 50 15 ;    % s
      0 50  0  0 50 ];   % 1
v=null(T');
b = [ -[5 33 79 50 0 0]*v, v'];
[optroot, coeff] = affpolymin(b,0);
fprintf('affpolymin computes infimal root abscissa = alpha^* = optroot where')
optroot
display(sprintf('Coefficients of the optimal polynomial [1,a_1,a_2,...] are: \n'))
coeff
fprintf('Computed roots are \n')
roots(coeff)
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
%%
%Example 8 from shorter version of paper published in CDC 2010 
display('Example 8 (CDC 2010): an example showing a global and local minimizer')
% random data was previously generated for several trials until an example of this kind was found, 
% showing by means of a clear contour plot that both local and global minimizers may exist
%
% plot is contours of the radius of the cubics parameterized by\n')
% z^3 + [F(1) + G(1,1) x(1) + G(1,2) x(2)] z^2 + [F(2) + G(2,1) x(1) +  G(2,2) x(2)] z + [F(3) + G(3,1) x(1) + G(3,2) x(2)]' 
F = [-0.049761722906494
      0.719300581221887
     -0.283063409581732]; 
G = [-1.141419279710520  -0.168117876318615
      0.061068871317613  -0.687348838217945  
      0.012317687245050  -0.990650043494419];
% make the contour plot
nx = 200;
ny = 200;
ax = [-2 2 -2 2];
xmin = ax(1); xmax = ax(2); ymin = ax(3); ymax = ax(4);
fgrid = zeros(nx,ny);
xstep = (xmax - xmin)/(nx-1);
ystep = (ymax - ymin)/(ny-1);
xgrid = zeros(1,nx);
ygrid = zeros(1,ny);
for j = 1:nx 
   xgrid(j) = xmin + (j-1)*xstep;
   for k = 1:ny 
      ygrid(k) = ymin + (k-1)*ystep;
      x = [xgrid(j); ygrid(k)];
      % compute coefficients of polynomial at this grid point
      coeff = [1; F + G*x]; 
      % compute radius of polynomial
      fgrid(k,j) = max(abs(roots(coeff)));
   end
end
[c,h] = contour(xgrid, ygrid, fgrid, 50);
hold on;
title('Contour Plot of Radius of a Two-Parameter family of Monic Cubics')
xlabel('x_1');
ylabel('x_2');
fprintf('The contour plot shows the contours of the radius of a two-parameter affine family of monic cubics\n')
% since this is an affine parameterization of two variables, there is one
% implicit affine constraint on the coefficients of the monic cubic family
v = null(G')';  % so v*G = 0, hence v*(F + G*x) = v*F
beta = -v*F;
b = [beta v];  % the affine constraint which must be input to affpolymin,
               % since b*(coeff vector) = beta + v*(F + G*x) = 0
% now find the globally optimal coefficients
[optroot, coeffglobalmin, k, M, allroots] = affpolymin(b,1);
% note that -allroots (minus all the roots of the polynomials g_k of the paper) 
% includes candidates for local minimizers: as we can see from the plot, 
% -allroots(2) is the radius value for a local minimizer
fprintf('|allroots(3)| is the globally minimal value and |allroots(2)| is a locally minimal value\n')
allroots
% extract the parameter values from the coefficients of the globally optimal polynomial, discarding the initial 1 
xglobalmin = G\(coeffglobalmin(2:4)' - F);
% plot the global minimizer on the contour plot
plot(xglobalmin(1),xglobalmin(2),'r*')
fprintf('The red asterisk shows the global minimizer\n')
% construct the local minimizer too: it turns out to be (z+allroots(2)^3
coefflocalmin = poly(-allroots(2)*ones(1,3));  % Matlab's built-in poly
xlocalmin = G\(coefflocalmin(2:4)' - F);
plot(xlocalmin(1),xlocalmin(2),'ro')
fprintf('The red circle shows the local minimizer\n')
hold off
fprintf('hit any key to continue\n\n')
fprintf('-----------------------------------------------------------------------------------------------------\n')
pause;
%%
%Example 9 from shorter version of paper published in CDC 2010 
display('Example 9 (CDC 2010): Consider the monic quintic polynomials subject to the affine constraint B_0 + a_1 + a_2 + a_3 + a_4 + a_5 = 0, i.e')
fprintf('P = {z^5 + a_1 z^4 + a_2 z^3 + a_3 z^2 + a_4 z + a_5 | B_0 + a_1 + a_2 + a_3 + a_4 + a_5 = 0 }\n')
fprintf('Figure plots the optimal abscissa for a_i real and a_i complex respectively, as a function of B_0 in [-2,4]:\n')
%PARAMETERS
n = 5; %dimension
delta_coarse = 0.1; %number of discretisation for [b1min,b1max].
delta_fine = 0.01; %number of discretisation for the finer grid
left1 = 0.95; % [left1, right1] is to be sampled finer.
right1 =1.05;
b1max = 4;
b1min = -2; 
   
%INITIALIZE VECTORS
b = ones(1,n+1)*i;
b1 = b1min : delta_coarse : left1;
b1 = [b1 , left1: delta_fine : right1];
b1 = [b1 , right1 :delta_coarse: b1max ];
nsim = length(b1);
optroot_complex = zeros(1,nsim); %contains opt abscissa with complex coef
optroot_real = zeros(1,nsim); %contains opt abscissa with real coef


for j = 1:nsim
    %Change b(1) at each iteration
    b(1)= b1(j)*i; % i is to make b(1) complex
    [optroot_complex(j)] = affpolymin(b,0,1e-6,0); % turn off output
end

%print opt abscissa with complex monic polynomials
grid on
plot(b1,real(optroot_complex),'-.r','MarkerSize',8,'LineWidth',6)
hold on


%INITIALIZE VECTORS
b = ones(1,n+1);

%solve inf abscissa problem with monic real polynomials
for j=1:nsim
    b(1)= b1(j);
    [optroot_real(j)] = affpolymin(b,0,1e-6,0);
end

%print opt abscissa with real monic polynomials
plot(b1,optroot_real,'c','MarkerSize',8,'LineWidth',5)   
 axis([b1min b1max -0.4 1.2])
 xlabel('B_0')
 ylabel('optimal abscissa')
 % xlabel('B_0','fontsize',16)
 % ylabel('optimal abscissa','fontsize',16)
 legend('complex case','real case',2)
 grid on