//
//  Real/Expr Package Ver. 1.0
//    Copywrite (c) 1995, 1996 Exact Computation Project
//    written by Koji Ouchi (ouchi@simulation.nyu.edu)
//
//  File: extLong.cc
//    class extLong
//

#include <iostream.h>
#include <limits.h>
#include "extLong.h"

//  infty and tiny

long inftyLong = LONG_MAX;
long tinyLong  = LONG_MIN;

//  overflow-free long int arithmetics

static long nearHalfInftyLong = inftyLong / 2;
static long farHalfInftyLong  = inftyLong - nearHalfInftyLong;

static long nearHalfTinyLong = tinyLong / 2;
static long farHalfTinyLong  = tinyLong - nearHalfTinyLong;

long add4Long(long x, long y)
{
  long z;
  
  if (x >= farHalfInftyLong)
    //  x is large
  {
    long dx = x - farHalfInftyLong;
    
    if (y >= nearHalfInftyLong - dx)
      z = inftyLong;
    else
      z = x + y;
  }
  else if (x <= farHalfTinyLong)
    //  x is small
  {
    long dx = farHalfTinyLong - x;
    
    if (y <= nearHalfTinyLong + dx)
      z = tinyLong;
    else
      z = x + y;
  }
  else
    //  x is not small nor large
    if (y >= farHalfInftyLong)
      //  y is large
    {
      long dy = y - farHalfInftyLong;
      
      if (x >= nearHalfInftyLong - dy)
	z = inftyLong;
      else
	z = x + y;
    }
    else if (y <= farHalfTinyLong)
      //  y is small
    {
      long dy = farHalfTinyLong - y;
      
      if (x <= nearHalfTinyLong + dy)
	z = tinyLong;
      else
	z = x + y;
    }
    else
      //  both x and y are not small nor large
      z = x + y;
  
  return z;
}

long sub4Long(long x, long y)
{
  long z;
  
  if (x >= farHalfInftyLong)
    //  x is large
  {
    long dx = x - farHalfInftyLong;
    
    if (y <= - nearHalfInftyLong + dx)
      z = inftyLong;
    else
      z = x - y;
  }
  else if (x <= farHalfTinyLong)
    //  x is small
  {
    long dx = farHalfTinyLong - x;
    
    if (y >= - nearHalfTinyLong - dx)
      z = tinyLong;
    else
      z = x - y;
  }
  else
    //  x is not small nor large
    if (y <= - farHalfInftyLong)
      //  y is small
    {
      long dy = - farHalfInftyLong - y;
      
      if (x >= nearHalfInftyLong - dy)
	z = inftyLong;
      else
	z = x - y;
    }
    else if (y >= - farHalfTinyLong)
      //  y is large
    {
      long dy = y + farHalfTinyLong;
      
      if (x <= nearHalfTinyLong + dy)
	z = tinyLong;
      else
	z = x - y;
    }
    else
      //  both x and y are not small nor large
      z = x - y;
  
  return z;
}

//  class extLong

//  constructors

extLong :: extLong() : val(0), isOverFlow(0)
{}

extLong :: extLong(int i) : val(i)
{
  if (val == inftyLong)
    isOverFlow = 1;
  else if (val == tinyLong)
    isOverFlow = - 1;
  else
    isOverFlow = 0;
}

extLong :: extLong(long l) : val(l)
{
  if (val == inftyLong)
    isOverFlow = 1;
  else if (val == tinyLong)
    isOverFlow = - 1;
  else
    isOverFlow = 0;
}

extLong :: extLong(unsigned long u)
{
  if (u >= (unsigned long)(inftyLong))
  {
    val        = inftyLong;
    isOverFlow = 1;
  }
  else
  {
    val        = u;
    isOverFlow = 0;
  }
}

extLong :: extLong(const extLong& e)
{
  val        = e.val;
  isOverFlow = e.isOverFlow;
}

//  the destructor

extLong :: ~extLong()
{}

//  assignment operator

extLong extLong :: operator =(const extLong& e)
{
  val        = e.val;
  isOverFlow = e.isOverFlow;
  
  return *this;
}

//  comparison operators

int extLong :: compare(const extLong& x) const
{
  if (val > x.val)
    return 1;
  else if (val == x.val)
    return 0;
  else
    //  val < x.val;
    return - 1;
}

int extLong :: operator ==(const extLong& x) const
{
  return compare(x) == 0;
}

int operator ==(long l, const extLong& x)
{
  return extLong(l).operator ==(x);
}

int extLong :: operator !=(const extLong& x) const
{
  return compare(x) != 0;
}

int operator !=(long l, const extLong& x)
{
  return extLong(l).operator !=(x);
}

int extLong :: operator <(const extLong& x) const
{
  return compare(x) < 0;
}

int operator <(long l, const extLong& x)
{
  return extLong(l).operator <(x);
}

int extLong :: operator <=(const extLong& x) const
{
  return compare(x) <= 0;
}

int operator <=(long l, const extLong& x)
{
  return extLong(l).operator <=(x);
}

int extLong :: operator >(const extLong& x) const
{
  return compare(x) > 0;
}

int operator >(long l, const extLong& x)
{
  return extLong(l).operator >(x);
}

int extLong :: operator >=(const extLong& x) const
{
  return compare(x) >= 0;
}

int operator >=(long l, const extLong& x)
{
  return extLong(l).operator >=(x);
}

//  arithmetic operators

extLong extLong :: operator +(const extLong& x) const
{
  extLong y;
  
  if (isOverFlow == 1 || x.isOverFlow == 1)
    //  *this or x is infty
  {
    y.val        = inftyLong;
    y.isOverFlow = 1;
  }
  else
  {
    if ((y.val = add4Long(val, x.val)) == inftyLong)
      y.isOverFlow = 1;
    else if (y.val == tinyLong)
      y.isOverFlow = - 1;
    else
      y.isOverFlow = 0;
  }
  
  return y;
}

extLong operator +(long l, const extLong& x)
{
  return extLong(l).operator +(x);
}

extLong extLong :: operator -(const extLong& x) const
{
  extLong y;
  
  if (isOverFlow == 1)
    //  *this is infty
  {
    y.val        = inftyLong;
    y.isOverFlow = 1;
  }
  else
  {
    if ((y.val = sub4Long(val, x.val)) == inftyLong)
      y.isOverFlow = 1;
    else if (y.val == tinyLong)
      y.isOverFlow = - 1;
    else
      y.isOverFlow = 0;
  }
  
  return y;
}

extLong operator -(long l, const extLong& x)
{
  return extLong(l).operator -(x);
}

//  arithmetic and assignment operators

extLong extLong :: operator +=(const extLong& x)
{
  extLong t = operator +(x);
  
  val        = t.val;
  isOverFlow = t.isOverFlow;
}

extLong extLong :: operator -=(const extLong& x)
{
  extLong t = operator -(x);
  
  val        = t.val;
  isOverFlow = t.isOverFlow;
}

//  builtin functions

long extLong :: asLong() const
{
  return val;
}

int extLong :: isInfty() const
{
  return isOverFlow == 1;
}

int extLong :: isTiny() const
{
  return isOverFlow == - 1;
}

//  stream operators

ostream& extLong :: operator <<(ostream& o) const
{
  if (isOverFlow == 1)
    o << " infty ";
  else if (isOverFlow == - 1)
    o << " tiny ";
  else
    o << val;
  
  return o;
}

ostream& operator <<(ostream& o, const extLong& x)
{
  x.operator <<(o);
  
  return o;
}

//  unary minus

extLong extLong :: operator -() const
{
  if (isOverFlow == 1)
    return extLong(tinyLong);
  if (!isOverFlow)
    return extLong(- val);
  else
    //  isOverFlow == - 1
    return extLong(inftyLong);
}

//  class extULong

//  constructors

extULong :: extULong() : extLong()
{}

extULong :: extULong(int i) : extLong()
{
  if ((val = long(i)) == inftyLong)
    isOverFlow = 1;
  else if (val < 0)
  {
    if (val == tinyLong)
      isOverFlow = 0;
    
    val = 0;
  }
}

extULong :: extULong(long l) : extLong()
{
  if ((val = l) == inftyLong)
    isOverFlow = 1;
  else if (val < 0)
  {
    if (val == tinyLong)
      isOverFlow = 0;
    
    val = 0;
  }
}

extULong :: extULong(unsigned long u) : extLong(u)
{}

extULong :: extULong(const extLong& L) : extLong(L)
{
  if (val < 0)
    val = 0;
  
  if (isOverFlow < 0)
    isOverFlow = 0;
}

//  assignment operator

extULong extULong :: operator =(const extLong& L)
{
  this->extLong :: operator =(L);
  
  if (val < 0)
    val = 0;
  
  if (isOverFlow < 0)
    isOverFlow = 0;
}

