//
//  Real/Expr Package Ver. 1.0
//    Copywrite (c) 1995, 1996 Exact Computation Project
//    written by Koji Ouchi (ouchi@simulation.nyu.edu)
//
//  File: BigFloat.cc
//    class BigFloat
//

#include <float.h>
#include <limits.h>
#include <math.h>
#include "BigFloat.h"

static const BigInt anonymBigInt(0);

//  floor log base 2 of abs(x)
//  convention lg(0) = 0

long flrLg(long x)
{
  if (x)
    if (x == LONG_MIN)
      return LONG_BIT - 1;
    else
      //  1 <= |x| <= LONG_MAX
    {
      long l;
      
      if (x < 0)
	x = - x;
      
      l = - 1;
      
      while (x)
      {
	x >>= 1;
	++l;
      }
      
      return l;
    }
  else
    //  x == 0
    return 0;
}

long flrLg(unsigned long x)
{
  if (x)
  {
    long l;
    
    l = - 1;
    
    while (x)
    {
      x >>= 1;
      ++l;
    }
    
    return l;
  }
  else
    //  x == 0
    return 0;
}

long flrLg(const BigInt& x)
{
  return lg(x);
}

long clLg(unsigned long x)
{
  if (x >= LONG_MAX + 2)
    return LONG_BIT;
  else if (x >= 2)
    //  2 <= x <= LOGG_MAX + 1
    return flrLg((x << 1) - 1);
//  else if (x)
//    // x == 1
//    return 0;
  else
//    // x == 0
    //  x == 1 or 0
    return 0;
}

//  chunks

static long CHUNK_BIT = (long)(LONG_BIT / 2 - 2);

long chunkCeil(long bits)
{
  if (bits > 0)
    return (bits - 1) / CHUNK_BIT + 1;
  else
    return - (- bits) / CHUNK_BIT;
}

long chunkFloor(long bits)
{
  if (bits >= 0)
    return bits / CHUNK_BIT;
  else
    return - (- bits - 1) / CHUNK_BIT - 1;
}

long bits(long chunks)
{
  return CHUNK_BIT * chunks;
}

BigInt chunkShift(const BigInt& x, long s)
{
  if (!s || sign(x) == 0)
    return x;
  else if (s > 0)
    //  shift left
    if (sign(x) > 0)
      return x << bits(s);
    else
      //  x < 0
      return - (- x << bits(s));
  else
    //  shift right
    if (sign(x) > 0)
      return x >> bits(- s);
    else
      //  x < 0
      return - (- x >> bits(- s));
}

//  class BigFloatRep

//  constructor

BigFloatRep :: BigFloatRep(int i)
: m(i), err(0), exp(0), refCount(0)
{}

BigFloatRep :: BigFloatRep(long l)
: m(l), err(0), exp(0), refCount(0)
{}

long DBL_MAX_CHUNK = chunkCeil(DBL_MAX_EXP);

BigFloatRep :: BigFloatRep(double d) : err(0), refCount(0)
{
  if (d == 0.0)
  {
    m   = 0;
    exp = 0;
  }
  else
    //  d != 0.0
  {
    int isNegative = 0;
    
    if (d < 0.0)
    {
      isNegative = 1;
      d          = - d;
    }
    
    m = 0;
    
    int    binExp;
    double f = frexp(d, &binExp);
    
    exp = chunkFloor(binExp);
    
    long s = binExp - bits(exp);
    
    long   stop = 0;
    double intPart;
    
    while (f != 0.0 && stop < DBL_MAX_CHUNK)
    {
      f =   ldexp(f, (int)CHUNK_BIT);
      f =   modf(f, &intPart);
      m <<= CHUNK_BIT;
      m +=  (long)intPart;
      exp--;
    }
    
    if (s)
      m <<= s;
    if (isNegative)
      m.negate();
    
    stop++;
  }
}

BigFloatRep :: BigFloatRep(const BigInt& I, unsigned long u, long l)
: m(I), err(u), exp(l), refCount(0)
{}

Rational BigFloatRep :: Rationalize() const
{
  if (!err)
    if (exp >= 0)
      return Rational(chunkShift(m, exp), 1);
    else
      return Rational(m, chunkShift(1, - exp));
  else
    //  err != 0
    error("rationalize is called with inexact operand.");
}

//  the destructor

BigFloatRep :: ~BigFloatRep()
{}

//  approximation

void BigFloatRep :: trunc(const BigInt& I,
			  const extULong& r, const extLong& a)
{
  if (sign(I))
  {
    long tr = chunkFloor((- r + lg(I)).asLong());
    long ta = chunkFloor(- a.asLong());
    long t;
    
    if (r.isInfty() || a.isTiny())
      t = ta;
    else if (a.isInfty())
      t = tr;
    else
      t = ta < tr ? tr : ta;
    
    if (t > 0)
    {
      BigInt remainder;
      
      m   = chunkShift(I, - t);
      err = 1;
      exp = t;
    }
    else
      //  t <= 0
    {
      m   = I;
      err = 0;
      exp = 0;
    }
  }
  else
    //  I == 0
  {
    m   = 0;
    err = 0;
    exp = 0;
  }
}

void BigFloatRep :: truncM(const BigFloatRep& B,
			  const extULong& r, const extLong& a)
{
  if (sign(B.m))
  {
    long tr = chunkFloor((- 1 - r + lg(B.m)).asLong());
    long ta = chunkFloor(- 1 - a.asLong()) - B.exp;
    long t;
    
    if (r.isInfty() || a.isTiny())
      t = ta;
    else if (a.isInfty())
      t = tr;
    else
      t = ta < tr ? tr : ta;
    
    if (t >= chunkCeil(clLg(B.err)))
    {
      m   = chunkShift(B.m, - t);
      err = 2;
      exp = B.exp + t;
    }
    else
      //  t < chunkCeil(clLg(B.err))
      error("truncM is called with stricter precision than current error.");
  }
  else
    //  B.m == 0
  {
    long t = chunkFloor(- a.asLong()) - B.exp;
    
    if (t >= chunkCeil(clLg(B.err)))
    {
      m   = 0;
      err = 1;
      exp = B.exp + t;
    }
    else
      //  t < chunkCeil(clLg(B.err))
      error("truncM is called with stricter precision than current error.");
  }
}

void BigFloatRep :: approx(const BigFloatRep& B,
			   const extULong& r, const extLong& a)
{
  if (B,err)
    if (1 + clLg(B.err) <= lg(B.m))
      truncM(B, r + 1, a);
    else
      //  1 + clLg(B.err) > lg(B.m)
      truncM(B, inftyLong, a);
  else
    //  B.err == 0
  {
    trunc(B.m, r, a - bits(B.exp));
    exp += B.exp;
  }
}

void BigFloatRep :: div(const BigInt& N, const BigInt& D,
			const extULong& r, const extLong& a)
{
  if (sign(D))
    if (sign(N))
    {
      long tr = chunkFloor((- r + lg(N) - lg(D) - 1).asLong());
      long ta = chunkFloor(- a.asLong());
      
      if (r.isInfty() || a.isTiny())
	exp = ta;
      else if (a.isInfty())
	exp = tr;
      else
	exp = ta < tr ? tr : ta;
      
      BigInt remainder;
      
      divide(chunkShift(N, - exp), D, m, remainder);
      
      if (exp <= 0 && sign(remainder) == 0)
	err = 0;
      else
	err = 1;
    }
    else
      //  N == 0
    {
      m   = 0;
      err = 0;
      exp = 0;
    }
  else
    //  N == 0
    error("zero divisor.");
}

void BigFloatRep :: approx(const Rational& R,
			   const extULong& r, const extLong& a)
{
  div(R.numerator(), R.denominator(), r, a);
}

//  error-normalization

void BigFloatRep :: normal()
{
  long le = flrLg(err);
  
  if (le >= CHUNK_BIT + 2)
  {
    long f = chunkFloor(--le);
    
    m   >>= bits(f);
    err >>= bits(f);
    err +=  2;
    exp +=  f;
  }
}

void BigFloatRep :: bigNormal(BigInt& bigErr)
{
  long le = lg(bigErr);
  
  if (le < CHUNK_BIT + 2)
    err = bigErr.as_long();
  else
  {
    long f = chunkFloor(--le);
    
    m      >>= bits(f);
    bigErr >>= bits(f);
    err    = bigErr.as_long() + 2;
    exp    += f;
  }
}

//  arithmetics

void BigFloatRep :: add(const BigFloatRep& x, const BigFloatRep& y)
{
  long expDiff = x.exp - y.exp;
  
  if (expDiff > 0)
    //  x.exp > y.exp
    if (!x.err)
    {
      m   = chunkShift(x.m, expDiff) + y.m;
      err = y.err;
      exp = y.exp;
    }
    else
      //  x.err > 0
    {
      m   = x.m + chunkShift(y.m, - expDiff);
      err = x.err + 5;
      exp = x.exp;
      
      normal();
    }
  else if (!expDiff)
    //  x.exp == y.exp
  {
    m   = x.m + y.m;
    err = x.err + y.err;
    exp = x.exp;
    
    normal();
  }
  else
    //  x.exp < y.exp
    if (!y.err)
    {
      m   = x.m + chunkShift(y.m, - expDiff);
      err = x.err;
      exp = x.exp;
    }
    else
      //  y.err > 0
    {
      m   = chunkShift(x.m, expDiff) + y.m;
      err = y.err + 5;
      exp = y.exp;
      
      normal();
    }
}

void BigFloatRep :: sub(const BigFloatRep& x, const BigFloatRep& y)
{
  long expDiff = x.exp - y.exp;
  
  if (expDiff > 0)
    //  x.exp > y.exp
    if (!x.err)
    {
      m   = chunkShift(x.m, expDiff) - y.m;
      err = y.err;
      exp = y.exp;
    }
    else
      //  x.err > 0
    {
      m   = x.m - chunkShift(y.m, - expDiff);
      err = x.err + 5;
      exp = x.exp;
      
      normal();
    }
  else if (!expDiff)
  {
    m   = x.m - y.m;
    err = x.err + y.err;
    exp = x.exp;
    
    normal();
  }
  else
    //  x.exp < y.exp
    if (!y.err)
    {
      m   = x.m - chunkShift(y.m, - expDiff);
      err = x.err;
      exp = x.exp;
    }
    else
      //  y.err > 0
    {
      m   = chunkShift(x.m, expDiff) - y.m;
      err = y.err + 5;
      exp = y.exp;
      
      normal();
    }
}

void BigFloatRep :: mul(const BigFloatRep& x, const BigFloatRep& y)
{
  m = x.m * y.m;
  
  BigInt bigErr = abs(x.m) * y.err + x.err * abs(y.m) + x.err * y.err;
  
  exp = x.exp + y.exp;
  
  bigNormal(bigErr);
}

void BigFloatRep :: div(const BigFloatRep& x, const BigFloatRep& y,
			const extULong& R)
{
  if (!y.isZeroIn())
    //  y.m > y.err
    if (!x.err && !y.err)
    {
      div(x.m, y.m, R, inftyLong);
      exp += x.exp - y.exp;
    }
    else
      //  x.err > 0 or y.err > 0
    {
      BigInt bigErr, errRemainder;
      
      if (x.isZeroIn())
	//  x.m <= x.err
      {
	m   = 0;
	exp = x.exp - y.exp;
	
	divide(abs(x.m) + x.err, abs(y.m) - y.err, bigErr, errRemainder);
      }
      else
	//  x.m > x.err
      {
	long lx = flrLg(x.m);
	long ly = flrLg(y.m);
	long r;
	
	if (!x.err)
	  //  x.err == 0 and y.err > 0
	  r = ly + 2;
	else if(!y.err)
	  //  x.err > 0 and y.err == 0
	  r = lx + 2;
	else
	  //  x.err > 0 and y.err > 0
	  r = lx < ly ? lx + 2: ly + 2;
	
	long   t = chunkFloor(- r + lx - ly - 1);
	BigInt remainder;
	
	divide(chunkShift(x.m, - t), y.m, m, remainder);
	exp = t + x.exp - y.exp;
	
	divide(abs(remainder) + (x.err >> bits(t))
	       + 2 * (t > 0) + y.err * abs(m),
	       abs(y.m) - y.err,
	       bigErr,
	       errRemainder);
      }
      
      if (sign(errRemainder))
	++bigErr;
      
      bigNormal(bigErr);
    }
  else
    //  y.m <= y.err
    error("zero dvisor.");
}

//  squareroot

void BigFloatRep :: sqrt(const BigInt& x, const extLong& a)
{
  if (sign(x) == 0)
  {
    m   = 0;
    err = 0;
    exp = 0;
  }
  else if (x == 1)
  {
    m   = 1;
    err = 0;
    exp = 0;
  }
  else
  {
    m   = x;
    err = 0;
    exp = 0;
    
    BigFloatRep q(anonymBigInt, 0, 0);
    BigFloatRep z(anonymBigInt, 0, 0);
    extLong     aa;
    
    for (;;)
    {
      aa    = a - bits(exp);
      q.div(x, m, inftyLong, aa);
      q.err = 0;
      q.exp -= exp;
      
      z.sub(*this, q);
      
      if (sign(z.m) <= 0 || z.MSB() < - a)
	break;
      
      z.add(*this, q);
      m   = z.m >> 1;
      err = 0;
      exp = z.exp;
    }
  }
}

static long HALF_CHUNK_BIT = (CHUNK_BIT + 1) / 2;

void BigFloatRep :: sqrt(const BigFloatRep& x, const extLong& a)
//void BigFloatRep :: sqrt(const BigFloatRep& w, const extLong& a)
{
//  BigFloatRep x(m, err, exp);
//  x.m   = 3;
//  x.err = 6;
//  x.exp = 0;
//  
  if (sign(x.m) >= 0)
    //  x.m >= 0
  {
//    cerr << " debug : x = " << x.m << " " << x.err << " " << x.exp << endl;
//    
    int delta = x.exp & 1;
    
    if (x.isZeroIn())
      //  x.m <= x.err
    {
      m = 0;
      
      if (!x.err)
	err = 0;
      else
	//  x.err > 0
      {
	err = (long)(:: sqrt((double)x.err));
	err++;
	err <<= 1;
	
	if (delta)
	  err <<= HALF_CHUNK_BIT;
      }
      
      exp = x.exp >> 1;
      
      normal();
    }
    else if (!x.err)
      //  x.m > x.err = 0
    {
      BigFloatRep z(anonymBigInt ,0 ,0);
      extLong     ppp = a + 1;
      extLong     pp  = ppp + bits(x.exp >> 1);
      
      z.sqrt(chunkShift(x.m, delta), pp);
      
      long p = (pp + bits(z.exp)).asLong();
      
//      cerr << " debug : p = " << ppp << " " << pp << " " << p << endl;
//      cerr << " debug : z = " << z.m << " " << z.err << " " << z.exp << endl;
//      cerr << " debug : z = ";
//      z.operator <<(cerr);
//      cerr << endl;
//      
      if (p <= 0)
      {
	m = z.m;
	
	BigInt bigErr = 1 << - p;
	
	exp = z.exp + (x.exp >> 1);
	
//	cerr << " debug : y = " << m << " " << bigErr << " " << exp << endl;
//	cerr << " debug : y = ";
//	operator <<(cerr);
//	cerr << endl;
//	
	bigNormal(bigErr);
      }
      else
	//  p > 0
      {
	m = chunkShift(z.m, chunkCeil(p));
	
	long r = CHUNK_BIT - 1 - (p + CHUNK_BIT - 1) % CHUNK_BIT;
	
	err = 1 >> r;
	exp = - chunkCeil(ppp.asLong());
	
//	cerr << " debug : y = " << m << " " << err << " " << exp << endl;
//	cerr << " debug : y = ";
//	operator <<(cerr);
//	cerr << endl;
//	
	normal();
      }
    }
    else
      //  x.m > x.err > 0
    {
      BigFloatRep z(anonymBigInt,0 ,0);
      extLong     A = - flrLg(x.err) + (lg(x.m) + 1 - bits(delta) >> 1) + 3;
      
      z.sqrt(chunkShift(x.m, delta), A);
      
      long qqq = - 1 + (lg(x.m) >> 1) - delta * HALF_CHUNK_BIT;
      long qq  = qqq - clLg(x.err);
      long q   = qq + bits(z.exp);
      
//      cerr << " debug : q = " << A << " " << qqq << " " << qq << " " << q << endl;
//      cerr << " debug : z = " << z.m << " " << z.err << " " << z.exp << endl;
//      cerr << " debug z = : ";
//      z.operator <<(cerr);
//      cerr << endl;
//      
      if (q <= 0)
      {
	m = z.m;
	
	long   qqqq   = - qqq - bits(z.exp);
	BigInt bigErr = x.err << - qqqq;
	
	exp = z.exp + (x.exp >> 1);
	
//	cerr << " debug : y = " << m << " " << err << " " << exp << endl;
//	cerr << " debug : y = ";
//	operator <<(cerr);
//	cerr << endl;
//	
	bigNormal(bigErr);
      }
      else
	//  q > 0
      {
	m = chunkShift(z.m, chunkCeil(q));
	
	long r = CHUNK_BIT - 1 - (q + CHUNK_BIT - 1) % CHUNK_BIT;
	
	err = 1 >> r;
	exp = (x.exp >> 1) - chunkCeil(qq);
	
//	cerr << " debug : y = " << m << " " << err << " " << exp << endl;
//	cerr << " debug : y = ";
//	operator <<(cerr);
//	cerr << endl;
//	
	normal();
      }
    }
  }
  else
    error("squareroot is called with negative operand.");
}

//  comparison

int BigFloatRep :: compareMExp(const BigFloatRep& x) const
{
  if (!err && !x.err)
  {
    int st = sign(m);
    int sx = sign(x.m);
    
    if (st > sx)
      return 1;
    else if (st == 0 && sx == 0)
      return 0;
    else if (st < sx)
      return - 1;
    else
      //  need to compare m && exp
    {
      long expDiff = exp - x.exp;
      
      if (expDiff > 0)
	//  exp > x.exp
	return compare(chunkShift(m, expDiff), x.m);
      else if (!expDiff)
	return compare(m, x.m);
      else
	//  exp < x.exp
	return compare(m, chunkShift(x.m, - expDiff));
    }
  }
  else
    error("comparison is called with inexact operands.");
}

//  bultin functions

extLong BigFloatRep :: lMSB() const
{
  if (!isZeroIn())
    return extLong(lg(abs(m) - err)) + bits(exp);
  else
    return extLong(tinyLong);
}

extLong BigFloatRep :: MSB() const
{
  if (sign(m))
    return extLong(lg(m)) + bits(exp);
  else
    return extLong(tinyLong);
}

extLong BigFloatRep :: flrLgErr() const
{
  if (err)
    return extLong(flrLg(err)) + bits(exp);
  else
    return extLong(tinyLong);
}

extLong BigFloatRep :: clLgErr() const
{
  if (err)
    return extLong(clLg(err)) + bits(exp);
  else
    return extLong(tinyLong);
}

int BigFloatRep :: isZeroIn() const
{
  long lm = lg(m);
  long le = flrLg(err);
  
  if (lm < le)
    return 1;
  else if (lm > le)
    return 0;
  else
    return sign(abs(m) - err) <= 0;
}

int BigFloatRep :: signM() const
{
  return sign(m);
}

//  stream operator

//ostream& BigFloatRep :: operator <<(ostream& o) const
//{
//  o << " < " << m << " , " << err << " , " << exp << " > ";
//  
//  return o;
//}
//
BigInt FiveTo(unsigned long exp)
{
  if (exp == 0)
    return BigInt(1);
  else if (exp == 1)
    return BigInt(5);
  else
  {
    BigInt x = FiveTo(exp / 2);
    
    x = x * x;
    
    if (exp & 1)
      x *= 5;
    
    return x;
  }
}

static double lgTenM = 3.321928094887362;

ostream& BigFloatRep :: operator <<(ostream& o) const
{
  long exp4Five =
    (long)ceil(ceil(double(clLg(err) + bits(exp)) / lgTenM));
  long exp4Two = exp4Five - bits(exp);
  
  BigInt M, R;
  
  if (exp4Five > 0)
    if (exp4Two > 0)
      //  exp4Five > 0 && exp4Two > 0
      divide(m, FiveTo(exp4Five) << exp4Two, M, R);
    else
      //  exp4Five > 0 && exp4Two <= 0
      divide(m << - exp4Two, FiveTo(exp4Five), M, R);
  else
    if (exp4Two > 0)
      //  exp4Five <= 0 && exp4Two > 0
      divide(m * FiveTo(- exp4Five), BigInt(1) << exp4Two, M, R);
    else
      //  exp4Five <= 0 && exp4Two <= 0
      M = m * FiveTo(- exp4Five) >> - exp4Two;
  
  //  construct string to output
  
  char* decimalM = dec(M);
  char* p        = decimalM;
  
  //  isNagetive == 1 if M < 0
  
  int isNegative = 0;
  
  if (*p == '-')
  {
    isNegative = 1;
    p++;
  }
  
  //  lenDecimalM = # of digits that will be put
  //  p[lastDigit] will be rounded
  
  long lenDecimalM = strlen(p) - 1;
  
  //  check how many digits user want to see
  
  if (lenDecimalM > defPrtDgt)
  {
    exp4Five    += lenDecimalM - defPrtDgt;
    lenDecimalM = defPrtDgt;
  }
  
  if (p[lenDecimalM] >= '5')
    //  round up last digit
  {
    long l = lenDecimalM - 1;
    
    while (p[l] == '9' && l >= 0)
    {
      p[l] = '0';
      l--;
    }
    
    if (l < 0)
    {
      *p = '1';
      exp4Five++;
    }
    else
      p[l]++;
  }
  
  if (lenDecimalM == 0)
//    o << "NA";
    o << "Small";
  else
    //  lenDecimalM > 0
  {
    //  put mantissa
    
    if (isNegative)
      o.put('-');
    
    o << *p++ << '.';
    exp4Five += lenDecimalM--;
    
    if (lenDecimalM == 0)
      o.put('0');  //  to enphasize this is BigFloat, not integer
    else
      while (lenDecimalM)
      {
	o.put(*p++);
	lenDecimalM--;
      }
    
    //  put exponent
    
    if (exp4Five > 0)
      o << "e+" << exp4Five;
    else if (exp4Five < 0)
      o << 'e' << exp4Five;
  }
  
  return o;
}

//  error detection

void BigFloatRep :: error(const char* msg) const
{
  //  output error message to stderr
  
  cerr << "BigFloat Error : " << msg << endl;
  
  //  unusual termination
  
  exit(1);
}

//  class BigFloat

//  constructors

BigFloat :: BigFloat()
{
  rep = new BigFloatRep(anonymBigInt, 0, 0);
  rep->refCount++;
}

BigFloat :: BigFloat(int i)
{
  rep = new BigFloatRep(i);
  rep->refCount++;
}

BigFloat :: BigFloat(long l)
{
  rep = new BigFloatRep(l);
  rep->refCount++;
}

BigFloat :: BigFloat(double d)
{
  rep = new BigFloatRep(d);
  rep->refCount++;
}

BigFloat :: BigFloat(const BigInt& I, unsigned long u, long l)
{
  rep = new BigFloatRep(I, u, l);
  rep->refCount++;
}

BigFloat :: BigFloat(const BigFloat& x)
{
  rep = x.rep;
  rep->refCount++;
}

BigFloat :: operator Rational() const
{
  rep->Rationalize();
}

//  the destructor

BigFloat :: ~BigFloat()
{
  if (--rep->refCount == 0)
    delete rep;
}

//  assignment operator

BigFloat& BigFloat :: operator =(const BigFloat& x)
{
  if (--rep->refCount == 0)
    delete rep;
  
  rep = x.rep;
  rep->refCount++;
  
  return *this;
}

//  approximation

void BigFloat :: approx(const BigInt& I, const extULong& r, const extLong& a)
{
  if (rep->refCount > 1)
    //  *rep is shared
  {
    --rep->refCount;
    
    rep = new BigFloatRep(rep->m, rep->err, rep->exp);
    rep->refCount++;
  }
  
  rep->trunc(I, r, a);
}

void BigFloat :: approx(const BigFloat& B,
			const extULong& r, const extLong& a)
{
  if (rep->refCount > 1)
    //  *rep is shared
  {
    --rep->refCount;
    
    rep = new BigFloatRep(rep->m, rep->err, rep->exp);
    rep->refCount++;
  }
  
  rep->approx(*B.rep, r, a);
}

void BigFloat :: approx(const Rational& R,
			const extULong& r, const extLong& a)
{
  if (rep->refCount > 1)
    //  *rep is shared
  {
    --rep->refCount;
    
    rep = new BigFloatRep(rep->m, rep->err, rep->exp);
    rep->refCount++;
  }
  
  rep->approx(R, r, a);
}

//  unary minus

BigFloat BigFloat :: operator -() const
{
  return BigFloat(- rep->m, rep->err, rep->exp);
}

BigFloat operator -(const BigFloat& x)
{
  return x.operator -();
}

//  arithmetics

BigFloat BigFloat :: operator +(const BigFloat& x) const
{
  BigFloat y;
  
  y.rep->add(*rep, *x.rep);
  
  return y;
}

BigFloat operator +(const BigFloat& x, const BigFloat& y)
{
  return x.operator +(y);
}

BigFloat BigFloat :: operator -(const BigFloat& x) const
{
  BigFloat y;
  
  y.rep->sub(*rep, *x.rep);
  
  return y;
}

BigFloat operator -(const BigFloat& x, const BigFloat& y)
{
  return x.operator -(y);
}

BigFloat BigFloat :: operator *(const BigFloat& x) const
{
  BigFloat y;
  
  y.rep->mul(*rep, *x.rep);
  
  return y;
}

BigFloat operator *(const BigFloat& x, const BigFloat& y)
{
  return x.operator *(y);
}

BigFloat BigFloat :: div(const BigFloat& x, const extULong& r) const
{
  BigFloat y;
  
  y.rep->div(*rep, *x.rep, r);
  
  return y;
}

BigFloat operator /(const BigFloat& x, const BigFloat& y)
{
  return x.div(y, defRelPrec);
}

//  squareroot

BigFloat BigFloat :: sqrt(const extLong& a) const
{
  BigFloat x;
  
  x.rep->sqrt(*rep, a);
  
  return x;
}

BigFloat sqrt(const BigFloat& x)
{
  return x.sqrt(defAbsPrec);
}

//  comparisons

int BigFloat :: compare(const BigFloat& x) const
{
  return rep->compareMExp(*x.rep);
}

int operator ==(const BigFloat& x, const BigFloat& y)
{
  return x.compare(y) == 0;
}

int operator !=(const BigFloat& x, const BigFloat& y)
{
  return x.compare(y) != 0;
}

int operator  <(const BigFloat& x, const BigFloat& y)
{
  return x.compare(y) < 0;
}

int operator <=(const BigFloat& x, const BigFloat& y)
{
  return x.compare(y) <= 0;
}

int operator  >(const BigFloat& x, const BigFloat& y)
{
  return x.compare(y) > 0;
}

int operator >=(const BigFloat& x, const BigFloat& y)
{
  return x.compare(y) >= 0;
}

//  arithmetic and assignment opeartors

BigFloat& BigFloat :: operator +=(const BigFloat& x)
{
  BigFloat t = operator +(x);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

BigFloat& BigFloat :: operator -=(const BigFloat& x)
{
  BigFloat t = operator -(x);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

BigFloat& BigFloat :: operator *=(const BigFloat& x)
{
  BigFloat t = operator *(x);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

BigFloat& BigFloat :: operator /=(const BigFloat& x)
{
  BigFloat t = div(x, defRelPrec);
  
  if (--rep->refCount == 0)
    delete rep;
  
  rep = t.rep;
  rep->refCount++;
  
  return *this;
}

//  builtin function

int BigFloat :: isExact() const
{
  return rep->err == 0;
}

extLong BigFloat :: lMSB() const
{
  return rep->lMSB();
}

extLong BigFloat :: MSB() const
{
  return rep->MSB();
}

extLong BigFloat :: flrLgErr() const
{
  return rep->flrLgErr();
}

extLong BigFloat :: clLgErr() const
{
  return rep->clLgErr();
}

int BigFloat :: isZeroIn() const
{
  return rep->isZeroIn();
}

int BigFloat :: sign() const
{
  return rep->signM();
}

int sign(const BigFloat& x)
{
  return x.sign();
}

BigFloat BigFloat :: abs() const
{
  if (rep->signM() >= 0)
    return BigFloat(*this);
  else
    return BigFloat(- rep->m, rep->err, rep->exp);
}

BigFloat abs(const BigFloat& x)
{
  return x.abs();
}

//  stream operator

ostream& BigFloat :: operator <<(ostream& o) const
{
  rep->operator <<(o);
  
  return o;
}

ostream& operator <<(ostream& o, const BigFloat& x)
{
  x.operator <<(o);
  
  return o;
}

