classdef local_predicate < handle
    
    %Assuming the following indexing for inherited test results:
    %1:C0(f),2:C0(g),3:C1(f),4:C1(g),5:JC,6:MK,7:MK_face,8:C0_face

    properties (Constant)
        JC_scale = 4;
        MK_scale = 2;
        %For the test used on 6 sides:
        MK_box_scale = 1.3; %Scales each box before determining the faces, must be above 1 for current implementation
        MK_face_scale = 1.3; %Scales each face of the box, must be above 1 for current implementation
        MK_jac_ind = 6;
        C0_depth = 0; %0 means just evalaute on the main box, and gives the best performence by far (for C0_depth)
        C0_face_depth = 0; %0 means just evalaute on the main box, and seems to give the best performence (for  C0_face_depth)
    end
    
    methods
        function this = local_predicate()
            local_tracecurve;
        end
    end
    methods(Static)
        % ##### C0 #####
        function bool = C0(B,f,ind)
            if nargin > 2
                if isempty(B.testresults{ind})
                    bool = local_predicate.C0core(B,f);
                    B.testresults{ind} = bool;
                else
                    bool = B.testresults{ind};
                end
            else
                bool = local_predicate.C0core(B,f);
            end
        end
        function bool = C0core(B,f)
            %Returns true if the surface f=0 surely does not pass through B
            %bool = 0 \notin f(B)
            bool = funexcludes0(f,B,local_predicate.C0_depth); %~interval.zeros(1).subset(f(B));
        end
        % ####################
        


        % ##### C1 #####
        function bool = C1(B,df,ind)
            if nargin > 2
                if isempty(B.testresults{ind})
                    bool = local_predicate.C1core(B,df);
                    B.testresults{ind} = bool;
                else
                    bool = B.testresults{ind};
                end
            else
                bool = local_predicate.C1core(B,df);
            end
        end
        function bool = C1core(B,df)
            %C1(B,df, IND)
            %Returns true if the surface f=0 satisfies the C1 condition 
            %bool = 0 \notin < df(B) , df(B) >
                Bint = B.interval;
                bool = ~interval.zeros(3,1).subset(df(Bint)'*df(Bint));
        end
        function bool = C1cross(B,df,dg,ind)
            if nargin > 2
                if isempty(B.testresults{ind})
                    bool = local_predicate.C1crosscore(B,df,dg);
                    B.testresults{ind} = bool;
                else
                    bool = B.testresults{ind};
                end
            else
                bool = local_predicate.C1crosscore(B,df);
            end
        end
        function bool = C1crosscore(B,df,dg)
            %C1cross(B,df,dg,ind)
            %Returns true if the curve defined by f=0 and g=0 satisfies the C1 condition 
            %bool = 0 \notin < df(B) x dg(B) , df(B) x dg(B)>
                Bint = B.interval;
                cross_fg = cross(df(Bint),dg(Bint));
                bool = ~interval.zeros(3,1).subset(cross_fg'*cross_fg);
        end
        % ####################



        % ##### JC #####
        function [bool,v] = Jaccobian(B,df,dg,ind)
            if nargin > 3
                if isempty(B.testresults{ind})
                    [bool,v] = local_predicate.Jaccobiancore(B,df,dg);
                    B.testresults{ind} = v;
                else
                    v = B.testresults{ind};
                    bool = any(v);
                end
            else
                [bool,v] = local_predicate.Jaccobiancore(B,df,dg);
            end
            B.passesJaccobian = B.passesJaccobian || bool; %for function markcurve
        end
        function [bool,v] = Jaccobiancore(B,df,dg)
            B = B.scale(local_predicate.JC_scale);
            v = zeros(3,1);
            dB = [df(B),dg(B)];
            for i = 1:3
                testinterval = -(-1)^i*det(dB(setdiff([1,2,3],i),:));
                if 0 < testinterval
                    v(i) = 1;
                elseif testinterval < 0
                    v(i) = -1;
                else 
                    v(i) = 0; %Jaccobian did not succeed
                end
            end
            bool = any(v);
        end
        % ####################

        % ##### MK #####
        function bool = MK(B,f,df,g,dg,ind,ax)
            if nargin > 5
                if isempty(B.testresults{ind})
                    if nargin > 6
                        bool = local_predicate.MKcore(B,f,df,g,dg,ax);
                    else
                        bool = local_predicate.MKcore(B,f,df,g,dg);
                    end
                    B.testresults{ind} = bool;
                else
                    bool = B.testresults{ind};
                end
            else
                bool = local_predicate.MKcore(B,f,df,g,dg);
            end
        end
        function bool = MKcore(B,f,df,g,dg,ax)
            %Returns true if there are two pairs of opposite faces of B 
            %with f having opposite sign on one pair, and g on the other.
            %Includes preconditioning around the box center
            B = B.scale(local_predicate.MK_scale);
            m = B.center;
            dfm = df(m); dgm = dg(m);

	        Jm = [dfm,dgm,cross(dfm,dgm)]';
            invJm = Jm^(-1);
            h = @(varargin) cross(dfm,dgm)'*(funmanipulation.convert2vec(varargin{:})-m); %varargin can be x,y,z components or box or interval vector
            fprime = @(varargin) invJm(1,:) * [f(varargin{:});g(varargin{:});h(varargin{:})];
            gprime = @(varargin) invJm(2,:) * [f(varargin{:});g(varargin{:});h(varargin{:})];
            hprime = @(varargin) invJm(3,:) * [f(varargin{:});g(varargin{:});h(varargin{:})];

            
            faces = B.facets;
            if ~funsmaller0(fprime,faces(1,1)) || ~funsmaller0(@(B) -fprime(B),faces(1,2)) || ...
               ~funsmaller0(gprime,faces(2,1)) || ~funsmaller0(@(B) -gprime(B),faces(2,2)) || ...
               ~funsmaller0(hprime,faces(3,1)) || ~funsmaller0(@(B) -hprime(B),faces(3,2))
                bool = false;
            else
                bool = true;
            end
            

            if nargin>5
                Bint = B.interval;
                plothandle = B.plotbox(ax,'b');
                if ~exist('axorig','var') || isempty(axorig) || ~isgraphics(axorig) %#ok<NODEF> 
                    subplot(1,3,1,ax);
                    axorig = subplot(1,3,2);
                    r = groot;
                    Monitors = r.MonitorPositions;
                    [~,M] = max(Monitors(:,3));
                    fig = ax.Parent;
                    sizex = Monitors(M,3);
                    originalPos = fig.Position;
                    newPos = originalPos;

                    set(fig, 'units', 'pixels', 'position',newPos);
                    prepareaxes(axorig);
                end
                if ~exist('axtrans','var') || isempty(axtrans) || ~isgraphics(axtrans) %#ok<NODEF> 
                    axtrans = subplot(1,3,3);
                    prepareaxes(axtrans);
                end

                plothoriginal = local_predicate.plotfunctions(axorig,f,g,h,Bint);
                plothtransformed = local_predicate.plotfunctions(axtrans,fprime,gprime,hprime,Bint);
                ploth = [plothoriginal,plothtransformed];
            end
            
            if nargin>5
                if bool
                    titletext = 'success';
                else
                    titletext = [...
                         '$ f^\sim(B_x^-)$ = [',num2str(fprime(faces(1,1)).bounds),']',newline,...
                         '$-f^\sim(B_x^+)$ = [',num2str(uminus(fprime(faces(1,2))).bounds),']',newline,...
                         '$ g^\sim(B_y^-)$ = [',num2str(gprime(faces(2,1)).bounds),']',newline,...
                         '$-g^\sim(B_y^+)$ = [',num2str(uminus(gprime(faces(2,2))).bounds),']',newline,...
                         '$ h^\sim(B_y^-)$ = [',num2str(hprime(faces(3,1)).bounds),']',newline,...
                         '$-h^\sim(B_y^+)$ = [',num2str(uminus(hprime(faces(3,2))).bounds),']',newline];
                end
                title(axtrans,titletext,'interpreter','latex','FontSize',16);

                for i = 1:length(ploth)
                    delete(ploth{i});
                end
                delete(plothandle);
            end
        end
        % ####################

        % ##### Plot #####
        function ploth = plotfunctions(ax,fun1,fun2,fun3,B)
        %plot the functions within the box for testing reasons
            if nargin == 5
                plotinterval = [B(1).bounds,B(2).bounds,B(3).bounds];
                axis(ax,plotinterval);
            else
                plotinterval = axis(ax);
            end
            %Use the function arrayfun for the next 3 lines possibly
            fun1_element = @(x,y,z) elementwisefunction(fun1,x,y,z);
            fun2_element = @(x,y,z) elementwisefunction(fun2,x,y,z);
            fun3_element = @(x,y,z) elementwisefunction(fun3,x,y,z);
            ploth = cell(1,3);
            ploth{1} = fimplicit3(ax,fun1_element,plotinterval,'b');
            ploth{2} = fimplicit3(ax,fun2_element,plotinterval,'r');
            ploth{3} = fimplicit3(ax,fun3_element,plotinterval,'g');
            alpha(ploth{1},0.5);
            alpha(ploth{2},0.5);
            alpha(ploth{3},0.5);
        end
        function ploth = plotfunctions2(ax,fun1,fun2,B)
        %plot the functions within the box for testing reasons
            if nargin == 5
                plotinterval = [B(1).bounds,B(2).bounds,B(3).bounds];
                axis(ax,plotinterval);
            else
                plotinterval = axis(ax);
            end
            %Use the function arrayfun for the next 3 lines possibly
            fun1_element = @(x,y,z) elementwisefunction(fun1,x,y,z);
            fun2_element = @(x,y,z) elementwisefunction(fun2,x,y,z);
            ploth = cell(1,2);
            ploth{1} = fimplicit3(ax,fun1_element,plotinterval,'b');
            ploth{2} = fimplicit3(ax,fun2_element,plotinterval,'r');
            alpha(ploth{1},0.5);
            alpha(ploth{2},0.5);
        end
        % ####################
                
        % ##### MK_face #####
        function [bool,v] = MK_face(B,f,df,g,dg,ind,ax)
            if nargin > 5
                if isempty(B.testresults{ind})
                    if nargin > 6
                        [bool,v] = local_predicate.MK_face_core(B,f,df,g,dg,ax);
                    else
                        [bool,v] = local_predicate.MK_face_core(B,f,df,g,dg);
                    end
                    B.testresults{ind} = v;
                else
                    v = B.testresults{ind};
                    bool = any(v);
                end
            else
                [bool,v] = local_predicate.MK_face_core(B,f,df,g,dg);
            end
        end
        function [bool,v] = MK_face_core(B,f,df,g,dg)
            %Matrix showing which surfaces intersect the curve (populated below):
            res = zeros(3,2); 
            %(Does not consider intersections in non-parameterizable direction's faces)
            
            spandir = zeros(3,1);

            %C0 on the faces of the initial box
            [C0_face_small, ~] = local_predicate.C0_face_core(B.scale(1),f,g);
            if C0_face_small %MK test terminates true but no faces are found, will not be considered during reconstruction later
                bool = true;
                v = reshape(res.',1,[]);
                return
            end

            Bscaled = B.scale(local_predicate.MK_box_scale);
            
            %C0 on the faces of the scaled box
            [C0_face, exclusions] = local_predicate.C0_face_core(Bscaled,f,g);
            if C0_face %MK test terminates true but no faces are found, will not be considered during reconstruction later
                bool = true;
                v = reshape(res.',1,[]);
                return
            end

            [jacbool,jac] = local_predicate.Jaccobian(B,df,dg, local_predicate.MK_jac_ind);
            if ~jacbool %Jaccobian test fails, need to split
                bool = false;
                v = reshape(res.',1,[]);
                return
            end

            faces = Bscaled.facets;

            for i = 1:3
                if jac(i) ~= 0 %Parameterizable in this direction
                    for j = 1:2
                        if exclusions(2*(i-1)+j) ~= 1 %do not bother if face would be excluded
                            face = faces(i,j);
                            m = face.center;
                            dfm = df(m); dgm = dg(m);
    
                            %Remove the i'th dimension from dfm/dfg to get 2d versions
                            dfm(i,:) = [];
                            dgm(i,:) = [];
    
                            Jm = [dfm,dgm]';
                            invJm = Jm^(-1);
            
                            fprime = @(varargin) invJm(1,:) * [f(varargin{:});g(varargin{:})];
                            gprime = @(varargin) invJm(2,:) * [f(varargin{:});g(varargin{:})];
                            
                            edges = face.scale(local_predicate.MK_face_scale).facets;
                            
                            %facebool
                            if ~funsmaller0(fprime,edges(1,1)) || ~funsmaller0(@(B) -fprime(B),edges(1,2)) || ...
                               ~funsmaller0(gprime,edges(2,1)) || ~funsmaller0(@(B) -gprime(B),edges(2,2))
                                facebool = false;
                            else
                                facebool = true;
                            end
    
                            res(i,j) = facebool;
                        end
                    end
                    dirbool = res(i,1) && res(i,2);
                    spandir(i) = dirbool;
                end
            end
            
            %Flatten result for output
            v = reshape(res.',1,[]);
            bool = any(v);
            
        end
        %####################

        % ##### C0_face #####
        function [bool,v] = C0_face(B,f,g,ind) %Returns true iff all surfaces can be excluded
            if nargin > 3
                if isempty(B.testresults{ind})
                    [bool,v] = local_predicate.C0_face_core(B,f,g);
                    B.testresults{ind} = v;
                else
                    v = B.testresults{ind};
                    bool = all(v);
                end
            else
                [bool,v] = local_predicate.C0_face_core(B,f,g);
            end
        end
        function [bool,v] = C0_face_core(B,f,g)
            %Matrix showing which surfaces can be excluded
            res = zeros(3,2); 

            faces = B.facets;

            for i = 1:3
                for j = 1:2
                    face = faces(i,j);

                    res(i,j) = funexcludes0(f,face,local_predicate.C0_face_depth) || funexcludes0(g,face,local_predicate.C0_face_depth); %Can replace by calls to C0_core
                end
            end
            
            %Flatten result for output
            v = reshape(res.',1,[]);
            bool = all(v);
        end
        %####################
    end
end