# -*- coding: utf-8 -*-
#
# Compiles a program description composed of the sequence of instructions on the
# stack: e.g.
#
# a
# b
# b
# a
# 
# into sequence design files in the paren-plus format of Nupack, generates
# sequences from them using Nupack, and generates some data files and plots.
#
# Author: Nimrod Priell
#
import sys
import argparse # To parse command line arguments
import numpy as np # For extensive numerical computing
import shutil # To copy files around
from nupack_ops import *
from r_plotter_ops import *
from dna_program import *
from html_graphics import *

def write_temperature_analysis_file(temperature_mfe_desired_comparison_filename,
                                    temperatures, mfe_energies, mfe_probs, 
                                    desired_energies, desired_probs):
  """ Writes a file with the energies and probabilities of the MFE and the
      desired design for every temperature. """

  temperature_mfe_desired_comparison_file = open(
      temperature_mfe_desired_comparison_filename, "w")

  temperature_mfe_desired_comparison_file.write(
      "temp,mfe_energy,mfe_prob,desired_energy,desired_prob\n")

  for temperature in temperatures:
    temperature_mfe_desired_comparison_file.write("{},{},{},{},{}\n".format(temperature,
      mfe_energies[temperature], mfe_probs[temperature],
      desired_energies[temperature], desired_probs[temperature]))

  temperature_mfe_desired_comparison_file.close()


def find_delta_indices(desired_structure, mfe_structure):
  """ Returns a list with the indices of differing structure between the two
      specified structures. They are assumed to contain same number of same
      length strands ordered the same way (so just the connectivity differs) """
  desired_mfe_differences = []
  for i in range(0, len(desired_structure)):
    if desired_structure[i] != mfe_structure[i]:
      desired_mfe_differences.append(i)

  return desired_mfe_differences


def create_mfe_design_file(design_filename, mfe_design_filename, mfe_structure):
  """ Creates an MFE design file off of the original design file by adding the
      specified MFE structure line """
  shutil.copy(design_filename + ".in", mfe_design_filename + ".in")
  analysis_file = open(mfe_design_filename + '.in', 'a')
  analysis_file.write("\n" + mfe_structure + "\n")
  analysis_file.close()


def run_mfe_temperature_analysis(program):
  # Generate the structure file with desired bindings:
  desired_design_filename = program.design_filename + "_desired"
  shutil.copy(program.design_filename + ".in", desired_design_filename + ".in")
  analysis_file = open(desired_design_filename + '.in', 'a')
  analysis_file.write("\n" + program.desired_structure + "\n")
  analysis_file.close()
  print "Wrote the desired design analysis file ", desired_design_filename, ".in"

  nupack = NupackOps()
  mfe_structure = ""
  mfe_energies = {}
  mfe_probs = {}
  desired_energies = {}
  desired_probs = {}

  start_temperature = 18.0
  end_temperature = 52.0
  temperature_step = 0.2
  temperatures = np.arange(start_temperature, end_temperature, temperature_step)

  for temperature in temperatures:
    # Now let's compute how the minimum-free-energy complex compares with our
    # desired complex
    mfe_structure = nupack.mfe_for_temperature(temperature,
        program.design_filename, mfe_energies)
   
    desired_energies[temperature] = nupack.run_energy_analysis(
        temperature, desired_design_filename)

    desired_probs[temperature] = nupack.run_complex_prob_analysis(
        temperature, desired_design_filename)

    mfe_design_filename = program.design_filename + "_mfe"
    try: 
      create_mfe_design_file(program.design_filename, mfe_design_filename, mfe_structure)
    except IOError:
      sys.stderr.write('Could not create MFE design file ' + mfe_design_filename
          + '\n')

    mfe_probs[temperature] = nupack.run_complex_prob_analysis(
        temperature, mfe_design_filename)
    # End of temperature loop

  desired_mfe_differences = find_delta_indices(program.desired_structure, mfe_structure)

  print "The desired structure is ", program.desired_structure, " while the MFE structure is ", mfe_structure 
  print "", len(desired_mfe_differences), " differences at ", desired_mfe_differences

  temperature_mfe_desired_comparison_filename = program.design_filename + ".analysis"
  try: 
    write_temperature_analysis_file(temperature_mfe_desired_comparison_filename,
        temperatures, mfe_energies, mfe_probs, desired_energies, desired_probs)
  except IOError:
    sys.stderr.write('Could not write temperature analysis file to ' +
        temperature_mfe_desired_comparison_filename + '\n')

  sys.stdout.write('Wrote analysis file to ' + 
      temperature_mfe_desired_comparison_filename + '\n')
  
  rplotter = RPlotterOps()
  rplotter.plot_temperature(program.design_filename)


def parse_command_line_opts():
  parser = argparse.ArgumentParser(description='Compile DNA programs')
  parser.add_argument('filename', metavar='FILE', type=str, 
                      help='read the program from in/FILE.stack')
  parser.add_argument('-t', '--temperature', dest='temperature', 
                      type=float, default=37.0, nargs=1,
                      help='Temperature for the design')

  return parser.parse_args()


def main():
  """ Main entry point to the compiler """

  options = parse_command_line_opts()

  # Name of the stack input file
  out_dir = 'in'
  program_filename = options.filename
  design_filename = out_dir + '/' + program_filename

  temperature = options.temperature
  fstop = 0.001

  # Requested size of the instruction
  instruction_size = 20

  # Requested size of the tick/tock strands. It is recommended this will be even
  clock_size = 16
 
  program = DNAProgram(out_dir, program_filename, temperature)
  program.read_program()
  program.design_sequences()

  sys.stdout.write('Sequences are:\n' + program.pretty_sequences() + '\n')
  run_mfe_temperature_analysis(program)

  sys.stdout.write('Steps are:\n\t' + 
      "\n\t".join(program.desired_structures()) + '\n')

  for i in xrange(0, 7):
    program.run_design_analysis()
    program.render(HTMLGraphics())
    program.next_state()

# Main entry point
if __name__ == '__main__':
  main()

