package superply;

import java.io.IOException;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Vector;

import superply.ui.SPanel;
import superply.math.HintChecker;

import heurgame.Game;
import heurgame.GameServer;
import heurgame.HuClient;
import heurgame.PlayerProxy;
import heurgame.PlayerToken;
import heurgame.Referee;
import heurgame.analysis.SystemAnalyzer;
import heurgame.event.GameBroadcaster;
import heurgame.event.GameEvent;
import heurgame.event.GameListener;
import heurgame.event.PlayerBroadcaster;
import heurgame.event.PlayerEvent;
import heurgame.event.PlayerListener;
import heurgame.event.TimeBroadcaster;
import heurgame.event.TurnListener;
import heurgame.event.turn.NoLimitRRIterator;
import heurgame.logging.LogBox;
/*
 * Created on Nov 21, 2004
 */

/**
 * @author David Kaplin and Chris Quackenbush
 */
public class SuperPlyGame implements Game {
    private GameServer gameServer;
    private Vector huClients = new Vector();
    private long playerTimeMax = 0;
    private NoLimitRRIterator turnMaster;
    private SPanel visualization;
    private SPSystemAnalyzer judge;
    private PlayerToken[] playerOrder;
    private Referee official;
    private GameBroadcaster gameBroadcaster;//simultaneous
    private PlayerBroadcaster playerBroadcaster;//simultaneous
    private TimeBroadcaster timeBroadcaster;
	private HintChecker hintChecker;
    
    public void setup(GameServer s, SPanel v){
        gameServer = s;
        visualization = v;
        hintChecker = visualization.getHintChecker();
		final GameEvent e = new GameEvent();
		e.gameName = getName();
		e.context = this;
		e.currentState = "Lobby";
		gameBroadcaster.announceSetup(e);
    }
    public void newGame(long turnMax){
        System.out.println("NEW GAME");
        if (playerOrder.length==2){
            turnMaster = new NoLimitRRIterator();
        }else{
			System.out.println("Invalid Player Number: "+playerOrder.length);
			System.out.println("2 player limit");
			System.exit(1);
        }
        System.out.println("    -    NEW GAME");
        judge = new SPSystemAnalyzer(this,hintChecker,turnMaster);
        Vector order = new Vector();
        for(int i=0;i<playerOrder.length;i++){
        	order.add(playerOrder[i]);
        }
        turnMaster.setup(order);
        judge.newGame();
        playerTimeMax = turnMax;
		final GameEvent e = new GameEvent();
		e.gameName = getName();
		e.context = this;
		e.currentState = "The Game";
		final boolean done = false;
		gameBroadcaster.announceStart(e);
    }
    public long getMaximumNormalTime() { return playerTimeMax;}
    public long getMaximumWarningTime() { return playerTimeMax/10;}
    /** 
     * @see heurgame.Game#getName()
     */
    public String getName() { return "SuperPly v1.0"; }

    /** 
     * Current Minimum is 2
     * @see heurgame.Game#getMinimumPlayers()
     */
    public int getMinimumPlayers() { return 2;   }

    /**
     * All are integer values 
     * @return "boardSize hintType"
     */
    public String getGreeting() {
        return ""+ hintChecker.getBoardSize() + " " + hintChecker.getHintMethod();
    }

    /** 
     * @see heurgame.Game#getSystemAnalyzer()
     */
    public SystemAnalyzer getSystemAnalyzer() {
        return judge;
    }

    /** 
     * @see heurgame.Game#buildPlayerProxy(int, java.net.Socket)
     */
    public PlayerProxy buildPlayerProxy(Socket connection) {
        PlayerProxy proxy = new SuperPlyPlayerProxy(hintChecker);
        try {
            proxy.setup(proxy,
                    connection.getInetAddress().toString()+":"+
                    connection.getPort(),connection);
            proxy.setupLogging(new LogBox("PlayerProxy-"+proxy.getToken().getName()+"@"+proxy.getToken().getOrigin()
                    , new heurgame.ui.GraphicalLogInterface()));
            playerAdded(proxy.getToken());
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return proxy;
    }

    /** 
     * @see heurgame.Game#buildHumanClient(java.lang.String)
     */
    public void buildHumanClient(String arg0) {
        HuClient hclient = new SuperPlyHuClient();
        String saddress = gameServer.getServerAddress();
        String[] parts = saddress.split(":");
        
        try {
            int port = Integer.parseInt(parts[1]);
            if (parts[0].indexOf("/") !=-1){
                parts[0] =parts[0].substring(0,parts[0].indexOf("/"));
            }
            Thread h = new Thread(hclient);
            h.start();
            Socket out = new Socket(parts[0],port);
            hclient.connect(out,arg0);
            huClients.add(hclient);
        } catch (NumberFormatException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (UnknownHostException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    /** 
     * @see heurgame.Game#getHumanClients()
     */
    public HuClient[] getHumanClients() {
    	HuClient[] ret = new HuClient[huClients.size()];
        for(int i=0;i<ret.length;i++){
        	ret[i] = (HuClient)huClients.get(i);
        }
        return ret;
    }

    /** 
     * @see heurgame.Game#getState()
     */
    public String getState() {
        return judge.getState();
    }

    /** 
     * @see heurgame.Game#endTurn()
     */
    public void endTurn() {
       /* this.announcePlayerStatus();
        if (judge.isGameOver()){
            endGame();
        }else{
            startNextTurn();
        }*/
    }

    private GameEvent makeGameEvent(String state){
        GameEvent e = new GameEvent();
        e.gameName = getName();
        e.context = this;
        e.currentState = state;
        e.numberOfPlayers = playerOrder.length;
        return e;
    }
    /** 
     * @see heurgame.Game#endGame()
     */
    public void endGame() {
        announcePlayerStatus();
        final GameEvent e = makeGameEvent("GAMEOVER");
        turnMaster.removeAllTurnListeners();
        gameBroadcaster.announceEnd(e);
    }

    /** 
     * @see heurgame.Game#playerAdded(heurgame.PlayerProxy.PlayerInfo)
     */
    public void playerAdded(PlayerToken newPlayer) {
        final PlayerEvent e = new PlayerEvent();
        e.player = newPlayer;
        playerBroadcaster.announcePlayerJoined(e);
    }

    /** 
     * @see heurgame.Game#start()
     */
    public synchronized void start() {
         //official.reportStatus(turnMaster.getCurrentToken());
         //turnMaster.next();
         //official.reportIncrementalStatus();
         //this.announcePlayerStatus();
         while(!judge.isGameOver()){
             try {
                 Thread.sleep(200);
                 } catch (Exception e){
                     
                 }
             turnMaster.next();
             official.reportStatus(turnMaster.getCurrentToken());
             official.reportIncrementalStatus();
             this.announcePlayerStatus();
         }
         endGame();
        /*if (judge.isGameOver()){
            endGame();
        }else{
            
            startNextTurn();
        }*/
    }

    /** 
     * @see heurgame.Game#undoLastMove()
     */
    public boolean undoLastMove() {
        // TODO Auto-generated method stub
        return false;
    }

    /** 
     * @see heurgame.Game#addGameListener(heurgame.event.GameListener)
     */
    public void addGameListener(GameListener arg0) {
        gameBroadcaster.addGameListener(arg0);
    }

    /** 
     * @see heurgame.Game#addPlayerListener(heurgame.event.PlayerListener)
     */
    public void addPlayerListener(PlayerListener arg0) {
        playerBroadcaster.addPlayerListener(arg0);
    }
    public void addTurnListener(TurnListener t){
    	turnMaster.addTurnListener(t);
    }
    /** 
     * @see heurgame.Game#announceDisqualification(heurgame.PlayerProxy.PlayerInfo)
     */
    public synchronized void announceDisqualification(PlayerToken player) {
        if (playerOrder.length == 1){
            return;
        }
        final PlayerEvent e = new PlayerEvent();
        e.player = player;
        e.disqualified = true;
        playerBroadcaster.announcePlayerLeft(e);
        PlayerToken[] reducedPlayers = new PlayerToken[playerOrder.length - 1];
        Vector playersLeft = new Vector();
        for(int i=0;i<playerOrder.length;i++){
            if (playerOrder[i]==player){
                continue;   
            }
            playersLeft.add(playerOrder[i]);
        }
        for(int i = 0;i<reducedPlayers.length;i++){
            reducedPlayers[i] = (PlayerToken)playersLeft.get(i);
        }
        
        reorderPlayers(reducedPlayers);
        turnMaster.updateActiveTokens(playersLeft);
        this.judge.removePlayer(player);
        if (judge.isGameOver()){
            endGame();
        }
        
        
    }

    public synchronized void announcePlayerMoved(PlayerToken player) {
        final PlayerEvent e = new PlayerEvent();
        PlayerEvent[] ebatch = judge.computeScores(playerOrder);
        long timeLeft = 0;
        double score = 0;
        for(int i=0;i<ebatch.length;i++){
            if (ebatch[i].player == player){
                timeLeft = ebatch[i].timeLeft;
                score = ebatch[i].score;
                break;
            }
        }
        e.player = player;
        e.disqualified = false;
        e.validMove = true;
        e.timeLeft = timeLeft;
        e.score = score;
        playerBroadcaster.announcePlayerMoved(e);
    }
    
    /** 
     * @see heurgame.Game#announcePlayerLeft(heurgame.PlayerProxy.PlayerInfo)
     */
    public void announcePlayerLeft(PlayerToken player) {
        final PlayerEvent e = new PlayerEvent();
        e.player = player;
        e.disqualified = false;
        playerBroadcaster.announcePlayerLeft(e);
    }
    /** 
     * @see heurgame.Game#announcePlayerStatus()
     */
    public synchronized void announcePlayerStatus() {
        if (playerOrder.length == 0){
            return;
        }
        final PlayerEvent[] pe = judge.computeScores(playerOrder);
        for(int i=0;i<pe.length;i++){
            playerBroadcaster.announcePlayerStatusChanged(pe[i]);
        }
    }
    /** 
     * @see heurgame.Game#reorderPlayers(heurgame.PlayerToken[])
     */
    public void reorderPlayers(PlayerToken[] arg0) {
        playerOrder = arg0;
    }
    /** 
     * @see heurgame.Game#setReferee(heurgame.Referee)
     */
    public void setReferee(Referee ref) {
        official = ref;
        gameBroadcaster.addGameListener(official);
    }
    /** 
     * @see heurgame.Game#getIncrementalState()
     */
    public String getIncrementalState() {
        return this.judge.getIncrementalState();
    }
    /** 
     * @see heurgame.Game#setGameBroadcaster(heurgame.event.GameBroadcaster)
     */
    public void setGameBroadcaster(GameBroadcaster b) {
        gameBroadcaster = b;
    }
    /** 
     * @see heurgame.Game#setPlayerBroadcaster(heurgame.event.PlayerBroadcaster)
     */
    public void setPlayerBroadcaster(PlayerBroadcaster p) {
        playerBroadcaster = p;        
    }
    /** 
     * @see heurgame.Game#getTimeBroadcaster()
     */
    public TimeBroadcaster getTimeBroadcaster() {
        return timeBroadcaster;
    }
    /** 
     * @see heurgame.Game#setTimeBroadcaster(heurgame.event.TimeBroadcaster)
     */
    public void setTimeBroadcaster(TimeBroadcaster t) {
        timeBroadcaster = t;
    }
}
