package heurgame;

import heurgame.analysis.*;
import heurgame.event.*;

import java.util.Vector;
import java.util.ListIterator;

/**
 * @author David Kaplin
 * 
 * This referee is abstract enough to work with most turn-based games.
 * Alone, it knows nothing of the rules of the game.  To decide such 
 * manners it negotiates with the System and Move Analyzers.  It also
 * notifies the Game itself of certain events so that they can be 
 * broadcast to all listensers.  This Referee works with the players
 * directly and acts as a facade to keep them from interacting with 
 * the raw game logic.
 */
public class TurnBasedReferee extends DefaultReferee implements TurnListener{
    protected SystemAnalyzer finalJudge;
    protected MoveAnalyzer moveJudge;
    
    /**
     * Allows access to the game before the game starts
     * @see heurgame.event.GameListener#gameSetup(heurgame.event.GameEvent)
     */
    public synchronized void gameSetup(GameEvent e) {
        super.setup(e);
        System.out.println(timeMap.size());
    }

    /** 
     * Sends all Players the intial information.
     * 
     * @see heurgame.event.GameListener#gameStarted(heurgame.event.GameEvent)
     */
    public synchronized void gameStarted(GameEvent e) {
        super.setup(e);
        finalJudge = game.getSystemAnalyzer();
        moveJudge = finalJudge.getMoveAnalyzer();
        
        ListIterator p = playerProxies.listIterator();
        debugging.addEntry("Sending Initial Status",game.getGreeting());
        while(p.hasNext()){
            PlayerProxy proxy = (PlayerProxy)p.next();
            proxy.sendInitialStatus(game.getGreeting());
        }
    }

    /** 
     * In the case of a decisive victory informs the players if 
     * they have won.  Othewise it informs the players who have tied.
     * Finally it informs the players that have lost.
     * 
     * @see heurgame.event.GameListener#gameEnded(heurgame.event.GameEvent)
     */
    public synchronized void gameEnded(GameEvent e) {
        String goodString = "TIE";
        String badString= "LOSE";
        Vector loosers = new Vector();
        loosers.addAll(playerProxies);
        if (finalJudge.isDecisive()){
            goodString = "WIN";
            PlayerToken winToken = finalJudge.getWinningPlayer();
            PlayerProxy winner = (PlayerProxy)playerMap.get(winToken);
            winner.sendFinalStatus(goodString);
            debugging.addEntry("WINNER ANNOUNCED",winToken.getName()+" of "+winToken.getOrigin());
            loosers.remove(winner);//Remove the Proxy not the token!
        }else{
            PlayerToken[] pts = finalJudge.getWinningPlayers();
            for(int i=0;i<pts.length;i++){
                PlayerToken tieToken = pts[i];
                PlayerProxy tied = (PlayerProxy)playerMap.get(tieToken);
                tied.sendFinalStatus(goodString);
                debugging.addEntry("TIE ANNOUNCED",tieToken.getName()+" of "+tieToken.getOrigin());
                loosers.remove(tied);//Remove the Proxy, not the token!
            }
        }
        for(int i=0;i<loosers.size();i++){
            PlayerProxy lost = (PlayerProxy)loosers.get(i);
            lost.sendFinalStatus(badString);
            debugging.addEntry("Loser ANNOUNCED",lost.getToken().getName()+" of "+lost.getToken().getOrigin());
        }
    }

 
    /** 
     * Where we look at the moves 
     * 
     * @see heurgame.event.TurnListener#turnChanged(heurgame.event.TurnEvent)
     */
    public synchronized void turnChanged(TurnEvent e) {
        PlayerProxy chosen = (PlayerProxy)playerMap.get(e.player);
        PlayerToken tok = e.player;
        TimeKeeper timeKeeper = (TimeKeeper)timeMap.get(e.player);
		debugging.addEntry("Turn Change","Now at Round : "+ e.roundNumber+" of "+ e.totalRounds +" Player: "+tok.getName());
        long then = System.currentTimeMillis();
        String gameState = game.getState();
        timeKeeper.unPause();
        String playerMove = chosen.getMove(gameState);
        timeKeeper.pause();
        long diff =System.currentTimeMillis() - then;
        String response = moveJudge.evaluateMove(playerMove,tok,diff);
        
        while (response.startsWith(MoveAnalyzer.MOVE_INVALID)){
            if (timeKeeper.expired)
                break;
            debugging.addEntry("INVALID MOVE","Player: "+tok.getName()+ playerMove);
            String information = response.substring(MoveAnalyzer.MOVE_INVALID.length(),response.length());
            then = System.currentTimeMillis();
            timeKeeper.unPause();
            playerMove = chosen.sendInvalidMove(information);
            timeKeeper.pause();
            diff = System.currentTimeMillis() - then;
            response = moveJudge.evaluateMove(playerMove,tok,diff);
        }
        if (response.startsWith(MoveAnalyzer.MOVE_DISQUALIFIED)){
            String information = response.substring(MoveAnalyzer.MOVE_DISQUALIFIED.length(),response.length());
            disqualifyPlayer(response,e,chosen);
        }
    }
    
    private void disqualifyPlayer(String reason,TurnEvent timeDisqualified, PlayerProxy toRemove){
        toRemove.sendDisqualify(reason);
        playerProxies.remove(toRemove);
        game.announceDisqualification(toRemove.getToken());
        debugging.addEntry("Disqualify",toRemove.getToken().getName()+" from "+ toRemove.getToken().getOrigin()+" has been taken out of the game.");
    }
}
