package heurgame.ui;

import java.awt.Color;
import java.util.HashMap;

import heurgame.PlayerToken;
import heurgame.event.PlayerEvent;
import heurgame.event.PlayerListener;
/*
 * Created on Oct 12, 2004
 */

/**
 * @author David Kaplin
 *
 * Many games use Colors to represent a player. 
 * This class attempts to assign unique colors from a random pool of 
 * 32-bit colors to each player as they enter the system.
 */
public class PlayerColors implements PlayerListener {
    private HashMap colorMap = new HashMap();
    /**
     * Retrieves the color associated with a Token
     * @param t The Token to be represented
     * @return null, if the player hasn't joined, which is a 
     * highly unlikely scenareo because of the PlayerListener
     * Interface.
     */
    public Color getColor(PlayerToken t){
        return (Color)colorMap.get(t);
    }
    /** 
     * Adds a new color to represent the player
     * @see heurgame.event.PlayerListener#playerJoined(heurgame.event.PlayerEvent)
     */
    public void playerJoined(PlayerEvent e) {
        if (colorMap.get(e.player)==null){
            colorMap.put(e.player,getUniqueNewColor(10,.6f));
        }
    }

    /**
     * Removes the color from the field 
     * @see heurgame.event.PlayerListener#playerLeft(heurgame.event.PlayerEvent)
     */
    public void playerLeft(PlayerEvent e) {
        if (colorMap.get(e)!=null){
            colorMap.remove(e.player);
        }
    }

    /** 
     * Ignored
     * @see heurgame.event.PlayerListener#playerMoved(heurgame.event.PlayerEvent)
     */
    public void playerMoved(PlayerEvent e) { }

    /**
     * Ignored 
     * @see heurgame.event.PlayerListener#playerStatusChanged(heurgame.event.PlayerEvent)
     */
    public void playerStatusChanged(PlayerEvent e) { }
    
    /**
     * Outputs a random color
     * 
     * @param darkLimit the dimmest value acceptable
     * @return a new random color no darker than the grey of the limit.
     */
    private Color getRandomColor(float darkLimit){
        float range = .98f - darkLimit;
        float randR = (float)Math.random()*range + darkLimit;
        float randG = (float)Math.random()*range + darkLimit;
        float randB = (float)Math.random()*range + darkLimit;
        return new Color(randR,randG,randB,1.0f);
    }
    /**
     * Produces a relatively unique Color that should be dissimilar
     * from the current color list
     * 
     * @param trials		Number of random colors to test
     * @param darkLimit		Dimmest possible value for the color.
     * @return			Relatively unique color.
     */
    private Color getUniqueNewColor(int trials, float darkLimit){
        Color bestColor = null;
        double bestScore = -1;
        Color[] testSet = new Color[trials];
        
        Object[] cvalues = colorMap.values().toArray();
        Color[] existing = new Color[cvalues.length];
        for(int i=0;i<existing.length;i++){
        	existing[i] = (Color)cvalues[i];
        }
        double[] resultSet = new double[trials];
        
        for(int i=0;i<testSet.length;i++){
            testSet[i] = getRandomColor(darkLimit);
            Color currentColor = testSet[i];
            for(int e=0;e<existing.length;e++){
                Color existingColor = existing[e];
                resultSet[i]*= (Math.abs(existingColor.getRed() - currentColor.getRed())
                			 +Math.abs(existingColor.getGreen() - currentColor.getGreen())
                			 +Math.abs(existingColor.getBlue() - currentColor.getBlue()))/existing.length;
            }
            if (bestScore < resultSet[i]){
                bestScore = resultSet[i]; 
                bestColor = currentColor;
            }
        }
        return bestColor;
    }
}
