/*
 * Created on Oct 10, 2004
 */
package heurgame;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;

import heurgame.logging.*;
/**
 * @author David Kaplin
 *
 *The role of the GameServer is to abstract away the plumbing that is necessary 
 *for a socket mediated mode of communication.
 *
 *The GameServer also acts as a Facade hiding the interface of Game 
 *and Referee from some parts of the user interface.
 *
 *Hopefully this class will remove the nead for more plumbing.
 */
public class GameServer implements Runnable {
    private ServerSocket gameServer;
    private int port;
    private Referee gameRef;
    private Game game;
    private volatile String serverAddress;
    private String hostname;
    private LogBox debugging;
    protected volatile boolean active;
    
    /**
     * Constructs a new GameServer
     * 
     * @param hostname	for the server to bind
     * @param port		also for binding
     * @param thisGame	allowing the proper construction of Game Specific HuClients and PlayerProxies
     * @param official	Referees are involved in all player related transactions
     */
    public GameServer(String hostname,int port,Game thisGame,Referee official){
        gameRef = official;
        game = thisGame;
        serverAddress = "Nowhere";
        active = false;
        this.hostname = hostname;
        this.port = port;
        debugging = new LogBox("Game Server");
    }
    /**
     * Actual guts of the Server code.  
     * This is meant to be stuffed inside of a thread!
     *  
     * @see java.lang.Runnable#run()
     */
    public void run(){
        try {
            gameServer = new ServerSocket();
            gameServer.bind(new InetSocketAddress(hostname,port));
            
            this.serverAddress = gameServer.getInetAddress().getHostName() +":"+ port;
            debugging.addEntry("Server Bound","Server @ "+serverAddress);
        } catch (IOException e) {
            debugging.addUrgentEntry("ServerSocket Setup",e.getLocalizedMessage(),1);
            //e.printStackTrace();
            return;
        }
        active = true;
        while(active == true){
            try {
                Socket s = gameServer.accept();
                PlayerProxy p = game.buildPlayerProxy(s);
                gameRef.addPlayer(p);
                debugging.addEntry(p.getToken().getName()+" connected",s.getInetAddress().toString());
                Thread.sleep(100);
            } catch (InterruptedException ie){
                debugging.addUrgentEntry("Server Interrupted",ie.getLocalizedMessage(),1);
                //ie.printStackTrace();
            } catch (IOException e) {
                debugging.addUrgentEntry("Server Accept",e.getLocalizedMessage(),1);
            }
        }
    }
    /**
     * Called to reorder the current connected players
     * 
     * @param newOrder New Established Order
     */
    public void reorderPlayers(PlayerToken[] newOrder){
        game.reorderPlayers(newOrder);
    }
    /**
     * Called to disconnect all connected players from the game.
     */
    public void reset(){
        gameRef.rejectAllPlayers("Game Reset");
    }
    /**
     * Delegates the creation of a HuClient, a human client to the server.
     * @param name of the client
     */
    public void makeHuClient(String name){
        game.buildHumanClient(name);
    }
    
    public int getMinimumPlayers(){
        return game.getMinimumPlayers();
    }
    
    /**
     * @return String representing the ip address of the server and its port.
     */
    public String getServerAddress(){
        return serverAddress;
    }
    /**
     * @return Name of the game.
     */
    public String getGameName(){
        return game.getName();
    }
    public boolean isActive(){
        return active;
    }
    /**
     * Causes the Server spinloop to gracefully halt.
     */
    public void stopServer(){
        debugging.addEntry("Server Shutdown","Shutdown explicitly called.");
        active = false;
    }
}
