package nanomunchers.ui;

import heurgame.ui.DisplayUtilities;
import heurgame.ui.builder.AWTInterfaceBuilder;
import heurgame.ui.builder.ComponentOrientedBuilder;
import heurgame.ui.builder.DefaultInterfaceDirector;
import heurgame.ui.builder.MutableTextWrapper;
import heurgame.ui.builder.ZeroLayoutBuilder;
import heurgame.ui.form.BoundedIntegerValidator;
import heurgame.ui.form.ValidatingInputField;

import java.awt.Choice;
import java.awt.Component;
import java.awt.Container;
import java.awt.FileDialog;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.TextField;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import nanomunchers.graph.Graph;
import nanomunchers.graph.GraphFactory;
/*
 * Created on Oct 29, 2004
 */

/**
 * A rather small panel that allows all sorts of graphs to be read.
 * 
 * @author David Kaplin
 */
public class GraphGeneration extends Container {
    public static final int GRAPH_COMPLETE = 0;
    public static final int GRAPH_RANDOM = 1;
    public static final int GRAPH_FILE = 2;
    public static final int GRAPH_URL = 3;
    private final static String completeString = "Complete Grid"; 
    private final static String  partialString = "Partial Grid";
    private final static String  fileString = "File";
    private final static String  webString = "Web URL";
    private int graphOption = 0;
    private Choice graphTypes = new Choice();
    private TextField widthText = new TextField(3);
    private TextField heightText = new TextField(3);
    private TextField randomText = new TextField(3);;
    private TextField urlText = new TextField(40);;
    private int browseHandle;
    private int proxyHandle;
    private int proxyWindowHandle;
    private int moreInfoHandle;
    private Component widthLabel;
    private Component heightLabel;
    private Component percentageLabel;
    private Component fileLabel;
    private Component status;
    private Component extraStatus;
    private Component moreInformation;
    private Runnable onFinish;
    private Graph product;
    private boolean readyToGenerate = false;
    
    private String useFileOrURL = new String();
    private int useWidth = 10;
    private int useHeight = 10;
    private int useRandom = 50;
    
    private ComponentOrientedBuilder aBuilder;
    private FileDialog browseWindow;

    /**
     * @param cBuilder context to make sure the widgets agree
     */
    public GraphGeneration(ComponentOrientedBuilder cBuilder) {
        super();
        aBuilder = cBuilder;
        widthLabel = aBuilder.getComponent(aBuilder.buildTextDisplay("Width"));
        heightLabel = aBuilder.getComponent(aBuilder.buildTextDisplay("Height"));
        percentageLabel = aBuilder.getComponent(aBuilder.buildTextDisplay("Average nodes remaining"));
        fileLabel = aBuilder.getComponent(aBuilder.buildTextDisplay("(no file selected)"));
        status = aBuilder.getComponent(aBuilder.buildTextDisplay(""));
        extraStatus = aBuilder.getComponent(aBuilder.buildMultilineTextDisplay(""));;
        final int moreInfoWindow = aBuilder.buildScreen("More Information");
        int closeAction = aBuilder.buildAction("Close",new Runnable(){
           public void run(){
               aBuilder.getScreen(moreInfoWindow).hide();
           }
        });
        moreInfoHandle =aBuilder.buildAction("More...",new Runnable(){
            public void run(){
                Frame f =  aBuilder.getScreen(moreInfoWindow);
                DisplayUtilities.packAndCenterWindow(f);
                f.show();
            }
         });
        
        ZeroLayoutBuilder zBuild = new ZeroLayoutBuilder(cBuilder);
        zBuild.startRegion(moreInfoWindow);
        zBuild.setMainRegion(aBuilder.buildWrappedComponent(extraStatus));
        zBuild.addToCommands(closeAction);
        zBuild.finishRegion(moreInfoWindow);
        
        moreInformation = aBuilder.getComponent(moreInfoHandle);
        browseHandle = aBuilder.buildAction("Browse",new Runnable(){
            public void run(){
        	    browseWindow = new FileDialog((Frame) DisplayUtilities.getParentWindow(GraphGeneration.this), "Browse", FileDialog.LOAD);
     	    browseWindow.show();
     	    if (browseWindow.getDirectory() !=null){
     	        String filename = browseWindow.getDirectory() + browseWindow.getFile();
     	        MutableTextWrapper fWrap = new MutableTextWrapper(fileLabel);
     	        fWrap.setText(filename);
     	    }
     	    DisplayUtilities.ensureWindowFits(GraphGeneration.this);
                
            }
         });

        proxyHandle = aBuilder.buildAction("Web Proxy...",new Runnable(){
            public void run(){
                Frame f =aBuilder.getScreen(proxyWindowHandle);
                DisplayUtilities.packAndCenterWindow(f);
                f.show();
            }
         });
        GridBagLayout gb = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();
        constraints.insets = new Insets(2,2,2,2);
        setLayout(gb);
        constraints.gridwidth = 1;
        constraints.weightx = 0;
        Component graphType = aBuilder.getComponent(aBuilder.buildTextDisplay("Graph Type"));
        gb.setConstraints(graphType,constraints);
        constraints.weightx = 1;
        add(graphType);
        
        graphTypes.add(completeString);
        graphTypes.add(partialString);
        boolean readFiles = true;
        boolean readURLs = true;
        boolean useProxy = true;
        SecurityManager s = System.getSecurityManager();
        if (s != null){
            try {
            s.checkRead(".");
            } catch (SecurityException e){
                readFiles = false;
            }
            try {
                s.checkPropertiesAccess();
                //If we can't read or write the system properties
                //no use in setting a proxy.
            } catch (SecurityException se){
                useProxy = false;
            }
            /*
             * Too restrictive
             * 
             * try {
            s.checkConnect("http://www.w3.org",80);
            } catch (SecurityException e){
                readURLs = false;
            }*/
        }
        if (readFiles){
            graphTypes.add(fileString);
        }
        if (readURLs){
            graphTypes.add(webString);
        }
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.weightx=2;
        gb.setConstraints(graphTypes,constraints);
        add(graphTypes);
        
        if (useProxy){
            proxyWindowHandle = buildProxyWindow();
            constraints.weightx=0;
            constraints.gridwidth = GridBagConstraints.RELATIVE;
            gb.setConstraints(cBuilder.getComponent(proxyHandle),constraints);
            add(cBuilder.getComponent(proxyHandle));
        }
        
        urlText.setText("http://");
        constraints.weightx=0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        gb.setConstraints(urlText,constraints);
        add(urlText);
        
        constraints.gridwidth = 1;
        gb.setConstraints(fileLabel,constraints);
        add(fileLabel);
        
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        Component browseButton = aBuilder.getComponent(browseHandle);
        gb.setConstraints(browseButton,constraints);
        add(browseButton);
        
        constraints.gridwidth = 1;
        gb.setConstraints(percentageLabel,constraints);
        add(percentageLabel);
        
        gb.setConstraints(randomText,constraints);
        add(randomText);
        
        constraints.gridwidth =1;
        gb.setConstraints(widthLabel,constraints);
        add(widthLabel);
        
        gb.setConstraints(widthText,constraints);
        add(widthText);
        
        gb.setConstraints(heightLabel,constraints);
        add(heightLabel);
        
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        gb.setConstraints(heightText,constraints);
        add(heightText);
        
        
        constraints.gridwidth = GridBagConstraints.RELATIVE;
        constraints.gridheight = GridBagConstraints.RELATIVE;
        constraints.fill = GridBagConstraints.BOTH;
        gb.setConstraints(status,constraints);
        add(status);

        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.gridwidth = GridBagConstraints.RELATIVE;
        constraints.fill = GridBagConstraints.NONE;
        gb.setConstraints(moreInformation,constraints);
        add(moreInformation);
        
        //stitching the Listeners together
        graphTypes.addItemListener(new GraphChoiceListener());
        
        
    }
    
    
    private int buildProxyWindow(){
        final int proxyWindow = aBuilder.buildScreen("Web Proxy Settings");
        final TextField hostname = new TextField();
        final ValidatingInputField port = new ValidatingInputField("8080",new BoundedIntegerValidator(0,32767));
        int hostnameRegion = aBuilder.buildRegion("hostname");
        int portRegion = aBuilder.buildRegion("port");
        int mainRegion = aBuilder.buildRegion("Main");
        ZeroLayoutBuilder zBuilder = new ZeroLayoutBuilder(aBuilder);
        int useSettings = aBuilder.buildAction("Set Proxy",new Runnable(){
            public void run(){
                if (port.performValidation()==false){
                    port.displayError();
                    return;
                }
                System.setProperty("setProxy","true");
                System.setProperty("proxyHost",hostname.getText());
                System.setProperty("proxyPort",port.getText());
                aBuilder.getScreen(proxyWindow).hide();   
            }
        });
        int removeSettings =aBuilder.buildAction("Discard Proxy",new Runnable(){
            public void run(){
                System.setProperty("setProxy","false");
                aBuilder.getScreen(proxyWindow).hide();
            }
        }); 
        int closeAction = aBuilder.buildAction("Close",new Runnable(){
            public void run(){
                aBuilder.getScreen(proxyWindow).hide();
            }
        });
        
        zBuilder.startRegion(hostnameRegion);
        zBuilder.setLeftRegion(aBuilder.buildTextDisplay("Proxy Host:"));
        zBuilder.setMainRegion(aBuilder.buildWrappedComponent(hostname));
        zBuilder.finishRegion(hostnameRegion);
        
        zBuilder.startRegion(portRegion);
        zBuilder.setLeftRegion(aBuilder.buildTextDisplay("Port Number:"));
        zBuilder.setMainRegion(aBuilder.buildWrappedComponent(port));
        zBuilder.finishRegion(portRegion);
        
        zBuilder.startRegion(mainRegion);
        zBuilder.setTopRegion(hostnameRegion);
        zBuilder.setBottomRegion(portRegion);
        zBuilder.finishRegion(mainRegion);
        
        zBuilder.startRegion(proxyWindow);
        zBuilder.setMainRegion(mainRegion);
        zBuilder.addToCommands(useSettings);
        zBuilder.addToCommands(removeSettings);
        zBuilder.addToCommands(closeAction);
        zBuilder.finishRegion(proxyWindow);
        
        return proxyWindow;
    }
    /**
     * @return true if a graph could be generated
     */
    public boolean generateGraph(){
        double percentage = -1;
        int width = -1;
        int height = -1;
        String errorMessage = "";
        MutableTextWrapper sWrap = new MutableTextWrapper(status);
        MutableTextWrapper fWrap = new MutableTextWrapper(fileLabel);
        sWrap.setText("Working");
        //Step 1 validate input
        try {
            switch (graphOption){
        		case GraphGeneration.GRAPH_RANDOM:
        	    	int percent = Integer.parseInt(randomText.getText());
        			percentage = (double)percent/(double)100;
        		case GraphGeneration.GRAPH_COMPLETE:
        		    width  = Integer.parseInt(widthText.getText());
        			height = Integer.parseInt(heightText.getText());
        			break;
        	}
        } catch (NumberFormatException nfe){
            errorMessage = "Invalid: ";
	        if (percentage <= 0 && graphOption==GraphGeneration.GRAPH_RANDOM){
	            errorMessage+="Average Nodes Specified";
	        }else  if (width <= 0){
	            errorMessage+="Width";
	        }else if (height <= 0){
	            errorMessage+="Height";
	        }
	        
	    }
        
        try {
            switch (graphOption){
    			case GraphGeneration.GRAPH_RANDOM:
    			    product = GraphFactory.buildGridGraph(width,height);
    				product = GraphFactory.buildRandomized(product,percentage);
    			    break;
    			case GraphGeneration.GRAPH_COMPLETE:
    			    product = GraphFactory.buildGridGraph(width,height);
    			    break;
    			case GraphGeneration.GRAPH_URL:
    			    product = GraphFactory.loadHttpURL(urlText.getText());
    			    break;
    			case GraphGeneration.GRAPH_FILE:
    			    product = GraphFactory.loadFile(fWrap.getText());
    			    break;
            }
        }catch (Exception ex){
            //NullPointers will be thrown anyway, lets get the interesting ones
            if (ex.getClass().getName().equals("java.lang.NullPointException")==false){
                errorMessage += ex.toString();
            }
        }
        if (product==null && errorMessage.length()==0){
            errorMessage = "Could not find graph.";
        }
        if (errorMessage.length()>0){
            String shortMessage = errorMessage;
            if (errorMessage.length()> 20){
                shortMessage=errorMessage.substring(0,20);
                
                shortMessage = shortMessage+"...";
            }
            MutableTextWrapper mWrap = new MutableTextWrapper(extraStatus);
            mWrap.setText(errorMessage);
            sWrap.setText(shortMessage);
            DisplayUtilities.ensureWindowFits(status);
            moreInformation.setEnabled(true);
            return false;
        }
        sWrap.setText("Generation Complete");
        return true;
    }
    /**
     * @param r to be called when the graph ready to be used
     */
    public void setFinishingAction(Runnable r){
        onFinish = r;
    }
    /**
     * Sets defaults for the generation panel
     * 
     * @param mode GRAPH_COMPLETE, GRAPH_FILE, GRAPH_URL, GRAPH_RANDOM
     * @param width
     * @param height
     * @param randomKept range 0 - 100
     * @param fileOrURL 
     */
    public void setup(int mode,int width,int height,int randomKept,String fileOrURL){
        useWidth =width;
        useHeight = height;
        useRandom = randomKept;
        useFileOrURL = fileOrURL;
        setMode(mode);
    }

    /**
     * Changes the layout of the generation panel
     * @param mode GRAPH_COMPLETE, GRAPH_FILE, GRAPH_URL, GRAPH_RANDOM
     */
    public void setMode(int mode){
        graphOption = mode;
        Component proxyButton = aBuilder.getComponent(proxyHandle);
        moreInformation.setEnabled(false);
        percentageLabel.setVisible(false);
	    randomText.setVisible(false);
	    Component browseButton = aBuilder.getComponent(browseHandle);
	    browseButton.setVisible(false);
	    proxyButton.setVisible(false);
	    fileLabel.setVisible(false);
	    urlText.setVisible(false);
	    widthText.setVisible(false);
    	heightText.setVisible(false);
    	widthLabel.setVisible(false);
    	heightLabel.setVisible(false);

	    switch (graphOption){
        	case GraphGeneration.GRAPH_RANDOM://NOTE THIS USES FALLTHROUGH
        	    randomText.setText(""+useRandom);
        	    percentageLabel.setVisible(true);
        	    randomText.setVisible(true);
        	case GraphGeneration.GRAPH_COMPLETE:
        	    widthText.setText(""+useWidth);
        	    heightText.setText(""+useHeight);
        	    widthText.setVisible(true);
        	    heightText.setVisible(true);
        	    widthLabel.setVisible(true);
        	    heightLabel.setVisible(true);
        	    break;
        	case GraphGeneration.GRAPH_FILE:
        	    if (useFileOrURL.startsWith("http://")==false){
        	        MutableTextWrapper fWrap = new MutableTextWrapper(fileLabel);
        	        fWrap.setText(useFileOrURL);
        	    }
        	    browseButton.setVisible(true);
        	    fileLabel.setVisible(true);
        	    break;
        	case GraphGeneration.GRAPH_URL:
        	    if (useFileOrURL.startsWith("http://")){
        	        urlText.setText(useFileOrURL);
        	    }
        	    proxyButton.setVisible(true);
    	    	urlText.setVisible(true);
        	    break;
        }
	    DisplayUtilities.ensureWindowFits(this);
    }
    /**
     * @return GRAPH_COMPLETE, GRAPH_FILE, GRAPH_URL, GRAPH_RANDOM
     */
    public int getProductType(){
        return graphOption;
    }
    /**
     * @return the assembled graph
     */
    public Graph getGraph(){
        return product;
    }
    /**
     * Debugging method
     * @param args
     */
    public static void main(String args[]){
        GraphGeneration gg = new GraphGeneration(new AWTInterfaceBuilder(new DefaultInterfaceDirector()));
        int frameHandle = gg.aBuilder.buildMainScreen("Graph Editor");
        ZeroLayoutBuilder zBuild = new ZeroLayoutBuilder(gg.aBuilder);
        zBuild.startRegion(frameHandle);
        zBuild.setMainRegion(gg.aBuilder.buildWrappedComponent(gg));
        zBuild.finishRegion(frameHandle);
        gg.setup(GraphGeneration.GRAPH_COMPLETE,20,20,85,"http://www.moose.com");
        gg.setSize(500,350);
        Frame f = gg.aBuilder.getScreen(frameHandle);
        DisplayUtilities.packAndCenterWindow(f);
        f.show();
    }
    protected class GraphChoiceListener implements ItemListener{

        /** 
         * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
         */
        public void itemStateChanged(ItemEvent e) {
            if (e.getStateChange() == ItemEvent.SELECTED){
                int whatType = graphTypes.getSelectedIndex();
                String matches = graphTypes.getSelectedItem();
                if (matches.equals(completeString)){
                    whatType = GRAPH_COMPLETE;
                }else if (matches.equals(partialString)){
                    whatType = GRAPH_RANDOM;
                }else if (matches.equals(fileString)){
                    whatType = GRAPH_FILE;
                }else if (matches.equals(webString)){
                    whatType= GRAPH_URL;
                }
                setMode(whatType);
            }
            
        }
    }
}

