package nanomunchers;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Frame;

import nanomunchers.graph.Graph;
import nanomunchers.ui.GraphEditor;
import nanomunchers.ui.GraphGeneration;
import nanomunchers.ui.NanoSimulation;
import nanomunchers.ui.SimulationStatus;

import heurgame.GameEntry;
import heurgame.GameServer;
import heurgame.event.GameBroadcaster;
import heurgame.event.PlayerBroadcaster;
import heurgame.event.TimeBroadcaster;
import heurgame.ui.EstimatedTime;
import heurgame.ui.GraphicalLogInterface;
import heurgame.ui.PlayerColors;
import heurgame.ui.ScoreBoard;
import heurgame.ui.builder.AWTInterfaceBuilder;
import heurgame.ui.builder.ComponentOrientedBuilder;
import heurgame.ui.builder.InterfaceBuilder;
import heurgame.ui.builder.InterfaceDirector;
import heurgame.ui.builder.MutableTextWrapper;
import heurgame.ui.builder.SwingInterfaceBuilder;
import heurgame.ui.builder.ZeroLayoutBuilder;
import heurgame.ui.form.BoundedIntegerValidator;
import heurgame.ui.form.ValidatingInputField;

/*
 * Created on Dec 3, 2004
 */

/**
 * Applet and Game Entry for Nanomunchers.
 * 
 * @author David Kaplin
 *  
 */
public class NanoEntry extends GameEntry implements InterfaceDirector {
    private static GameEntry me;

    private int port = 20000;

    private String hostname = "127.0.0.1";

    private long gameTime = 1000 * 60 * 2;//2 minutes

    private ComponentOrientedBuilder aBuilder;

    private InterfaceBuilder iBuilder;

    private boolean sandboxed = false;

    private int mainHandle;

    private int welcomeHandle;

    private GameServer gameServer;

    private NanoRef official;

    private ScoreBoard scoreBoard;

    private NanoGame game;

    private int botNumberHandle;

    private int timeHandle;

    private int graphGenerationHandle;

    private boolean inReplay;

    private int replaySlowAction;

    private int replayFastAction;

    /**
     * @see heurgame.GameEntry#startGameApplet()
     */
    public void startGameApplet() {
        sandboxed = true;
        String useSwing = this.getParameter("builder");
        if (useSwing == null) {
            aBuilder = new AWTInterfaceBuilder(this);
        } else if (useSwing.equalsIgnoreCase("swing")) {
            aBuilder = new SwingInterfaceBuilder(this);
        } else {
            aBuilder = new AWTInterfaceBuilder(this);
        }
        iBuilder = (InterfaceBuilder) aBuilder;
        removeAll();
        setLayout(new BorderLayout());
        final int buttonHandle = aBuilder.buildAction("Start Nanomunchers",
                new Runnable() {
                    public void run() {
                        setEnabled(false);
                        if (gameServer == null) {
                            //hostname = getDocumentBase().getHost();
                            iBuilder.restrictApplet(true);
                            initInterface();
                        } else {
                            Thread sThread = new Thread(gameServer);
                            sThread.start();
                            Frame wFrame = aBuilder.getScreen(welcomeHandle);
                            wFrame.show();
                        }
                    }
                }, false);
        add(aBuilder.getComponent(buttonHandle));
    }

    /**
     * @see heurgame.GameEntry#startGame(java.lang.String[])
     */
    public void startGame(String[] params) {
        sandboxed = false;
        aBuilder = new AWTInterfaceBuilder(this);
        initInterface();
    }

    /**
     * @see heurgame.ui.builder.InterfaceDirector#initInterface()
     */
    public void initInterface() {
        iBuilder = (InterfaceBuilder) aBuilder;
        iBuilder.restrictApplet(sandboxed);
        game = new NanoGame();
        TimeBroadcaster timeBroadcaster = new TimeBroadcaster();
        EstimatedTime estimatedTime = new EstimatedTime();
        timeBroadcaster.addTimeListener(estimatedTime);
        game.setTimeBroadcaster(timeBroadcaster);
        GameBroadcaster gameBroadcaster = new GameBroadcaster();
        game.setGameBroadcaster(gameBroadcaster);
        final PlayerBroadcaster playerBroadcaster = new PlayerBroadcaster();
        game.setPlayerBroadcaster(playerBroadcaster);
        gameBroadcaster.addGameListener(estimatedTime);
        official = new NanoRef();
        game.setupTime(gameTime, gameTime / 4);
        game.setReferee(official);
        gameServer = new GameServer(hostname, port, game, official);
        PlayerColors picasso = new PlayerColors();

        Thread sThread = new Thread(gameServer);
        sThread.start();
        mainHandle = iBuilder.buildMainScreen(game.getName(), this
                .loadJarImage("nanomunchers/img/nano-icon.jpg"));
        welcomeHandle = iBuilder.buildWelcomeScreen(playerBroadcaster);
        scoreBoard = new ScoreBoard(picasso, false);
        playerBroadcaster.addPlayerListener(scoreBoard);
        gameBroadcaster.addGameListener(scoreBoard);
        final NanoSimulation ns = new NanoSimulation(picasso, playerBroadcaster);

        int statusHandle = aBuilder.buildTextDisplay("");

        MutableTextWrapper statusDisplay = new MutableTextWrapper(aBuilder
                .getComponent(statusHandle));
        SimulationStatus simStatus = new SimulationStatus(statusDisplay);
        ns.setFrameListener(simStatus);
        game.setupUI(ns, aBuilder);

        playerBroadcaster.addPlayerListener(picasso);

        //Layout the window
        ZeroLayoutBuilder ezBuild = new ZeroLayoutBuilder(aBuilder);
        ezBuild.startRegion(mainHandle);
        ezBuild.setMainRegion(aBuilder.buildWrappedComponent(ns));
        ezBuild.setRightRegion(aBuilder.buildWrappedComponent(scoreBoard));
        int newGameAction = aBuilder.buildAction("New Game", new Runnable() {
            public void run() {
                onEnd();
                aBuilder.getScreen(mainHandle).hide();
                aBuilder.getScreen(welcomeHandle).show();
            }
        });
        inReplay = false;
        GraphicalLogInterface.instance().composeLogWindow(aBuilder);
        replaySlowAction = aBuilder.buildAction("Faster", new Runnable() {
            public void run() {
                boolean didwork = ns.increaseSpeed();
                if (didwork == false) {
                    aBuilder.getComponent(replaySlowAction).setEnabled(false);
                }
                aBuilder.getComponent(replayFastAction).setEnabled(true);
            }
        }, false);
        replayFastAction = aBuilder.buildAction("Slower", new Runnable() {
            public void run() {
                boolean didwork = ns.decreaseSpeed();
                if (didwork == false) {
                    aBuilder.getComponent(replayFastAction).setEnabled(false);
                }
                aBuilder.getComponent(replaySlowAction).setEnabled(true);
            }
        }, false);
        int replayAction = aBuilder.buildAction("Replay", new Runnable() {
            public void run() {
                if (inReplay)
                    return;
                Component fast = aBuilder.getComponent(replayFastAction);
                Component slow = aBuilder.getComponent(replaySlowAction);
                fast.setEnabled(true);
                slow.setEnabled(true);
                inReplay = true;
                ns.setSimulationSpeed(4.0);
                ns.play();
                fast.setEnabled(false);
                slow.setEnabled(false);

                inReplay = false;
            }
        }, false);
        int logAction = aBuilder.buildAction("Log", new Runnable() {
            public void run() {
                GraphicalLogInterface.getLogWindow().show();
            }
        });

        ezBuild.addToCommands(newGameAction);
        ezBuild.addToCommands(replaySlowAction);
        ezBuild.addToCommands(replayAction);
        ezBuild.addToCommands(replayFastAction);
        ezBuild.addToCommands(logAction);
        ezBuild.addToStatus(aBuilder.buildWrappedComponent(estimatedTime
                .getEstimatedTime()));
        ezBuild.addToStatus(statusHandle);
        ezBuild.finishRegion(mainHandle);

        aBuilder.getScreen(welcomeHandle).show();
        GraphGeneration graphGen = (GraphGeneration) aBuilder
                .getComponent(graphGenerationHandle);
        graphGen.setup(GraphGeneration.GRAPH_COMPLETE, 20, 10, 40,
                "(unspecified)");
    }

    /**
     * @see heurgame.ui.builder.InterfaceDirector#getServer()
     */
    public GameServer getServer() {
        return gameServer;
    }

    /**
     * @see heurgame.ui.builder.InterfaceDirector#getGameOptionsInterface()
     */
    public Container getGameOptionsInterface() {
        int optionsHandle = iBuilder.buildRegion("options");
        int botOptions = iBuilder.buildRegion("bot options");
        int timeOptions = iBuilder.buildRegion("time options");
        int bundledOptions = iBuilder.buildRegion("Bundled options");
        timeHandle = iBuilder.buildWrappedComponent(new ValidatingInputField(
                "120", new BoundedIntegerValidator(9, false)));

        GraphGeneration graphGen = new GraphGeneration(aBuilder);
        graphGenerationHandle = aBuilder.buildWrappedComponent(graphGen);
        int editGraph = aBuilder.buildAction("Graph Editor", new Runnable() {
            public void run() {
                GraphEditor editor = new GraphEditor(aBuilder
                        .getScreen(welcomeHandle), aBuilder);
                editor.show();
            }
        });
        botNumberHandle = iBuilder
                .buildWrappedComponent(new ValidatingInputField("2",
                        new BoundedIntegerValidator(1, false)));

        ZeroLayoutBuilder zBuilder = new ZeroLayoutBuilder(aBuilder);
        zBuilder.startRegion(botOptions);
        zBuilder.setLeftRegion(iBuilder
                .buildTextDisplay("Nanomunchers Per Player (1+)"));
        zBuilder.setMainRegion(botNumberHandle);
        zBuilder.finishRegion(botOptions);

        zBuilder.startRegion(timeOptions);
        zBuilder.setLeftRegion(iBuilder
                .buildTextDisplay("Time Per Player (10+ seconds)"));
        zBuilder.setMainRegion(timeHandle);
        zBuilder.finishRegion(timeOptions);

        zBuilder.startRegion(bundledOptions);
        zBuilder.setTopRegion(botOptions);
        zBuilder.setBottomRegion(timeOptions);
        zBuilder.finishRegion(bundledOptions);

        zBuilder.startRegion(optionsHandle);
        zBuilder.setTopRegion(bundledOptions);
        zBuilder.setMainRegion(graphGenerationHandle);
        SecurityManager s = System.getSecurityManager();
        boolean canSave = true;
        if (s != null) {
            try {
                s.checkWrite("./foo.graph");
            } catch (SecurityException e) {
                canSave = false;
            }
        }
        if (canSave) {
            zBuilder.setBottomRegion(editGraph);
        }
        zBuilder.finishRegion(optionsHandle);
        return (Container) aBuilder.getComponent(optionsHandle);
    }

    /**
     * @see heurgame.ui.builder.InterfaceDirector#onStart()
     */
    public void onStart() {
        // TODO Auto-generated method stub
        GraphicalLogInterface.instance().clearLog();
        //final NanoSimulation ns = new NanoSimulation(picasso);
        //game.setupUI(ns);
        int bots = 2;
        GraphGeneration graphGeneration = (GraphGeneration) aBuilder
                .getComponent(graphGenerationHandle);
        if (graphGeneration.generateGraph() == false) {
            return;
        }
        ValidatingInputField b = (ValidatingInputField) aBuilder
                .getComponent(botNumberHandle);
        if (b.performValidation() == false) {
            b.displayError();
            return;
        }
        ValidatingInputField t = (ValidatingInputField) aBuilder
                .getComponent(timeHandle);
        if (t.performValidation() == false) {
            t.displayError();
            return;
        }
        Graph graph = graphGeneration.getGraph();
        bots = Integer.parseInt(b.getText());
        gameTime = 1000 * Integer.parseInt(t.getText());

        game.setupTime(gameTime, gameTime / 10);

        scoreBoard.clear();
        Component fast = aBuilder.getComponent(replayFastAction);
        Component slow = aBuilder.getComponent(replaySlowAction);
        fast.setEnabled(true);
        slow.setEnabled(true);
        Frame f = aBuilder.getScreen(mainHandle);
        int swidth = f.getToolkit().getScreenSize().width - 150;
        int sheight = f.getToolkit().getScreenSize().height - 150;
        int gwidth = graph.getWidth();
        int gheight = graph.getHeight();
        while (gwidth < swidth && gheight < sheight){
            gwidth+=graph.getWidth();
            gheight+=graph.getHeight();
        }
        gwidth-=graph.getWidth();
        gheight-=graph.getHeight();
    
        aBuilder.getScreen(welcomeHandle).hide();
        f.setSize(gwidth , gheight );
        f.show();
        game.newGame(bots,
                        graph,
                        graphGeneration.getProductType() == GraphGeneration.GRAPH_COMPLETE);
        
        //f.pack();
        game.start();
    }

    /**
     * @see heurgame.ui.builder.InterfaceDirector#onEnd()
     */
    public void onEnd() {
        official.rejectAllPlayers("Game Over");
        scoreBoard.clear();
        setEnabled(true);
    }

    /**
     * @see heurgame.ui.builder.InterfaceDirector#onExit(int)
     */
    public void onExit(int code) {
        if (code != 0) {

        }
        if (sandboxed == false) {
            System.out.println("Exit");
            System.exit(code);
        }
    }

    public static GameEntry instance() {
        if (me == null) {
            me = new NanoEntry();
        }
        return me;
    }

    public static void main(String[] args) {
        instance().startGame(args);
    }
}