package heurgame.event.turn;

import java.util.ListIterator;
import java.util.Vector;

import heurgame.PlayerToken;
import heurgame.event.TurnEvent;
import heurgame.logging.LogBox;

/*
 * Created on Oct 12, 2004
 */

/**
 * @author David Kaplin
 * 
 * This particular iterator is token-based.  In a given round the 
 * iterator offers turns in correspondance with each token.  For any
 * reason if the order changes in the middle of a round, the new order
 * will not take effect until the next round.  When a person is disqualified
 * it would be good to notify this iterator of the new order so they 
 * do not get a turn in the next round.
 */
public class RoundRobinIterator extends AbstractTokenBasedIterator {
    /**
     * Handle for easy debugging access <code>rrLog</code>
     */
    protected LogBox rrLog;

    /**
     * The source of the order is <code>playerTokens</code>
     */
    protected Vector playerTokens;

    /**
     * Round will start with 1 <code>currentRound</code>
     */
    protected volatile int currentRound;

    /**
     * Only positive numbers for <code>totalRounds</code>
     */
    protected volatile int totalRounds;

    /**
     * A ListIterator is doing the real dirty work <code>playerHelper</code>
     */
    protected ListIterator playerHelper;

    /**
     * handle to the current player <code>currentToken</code>
     */
    protected volatile PlayerToken currentToken;

    /**
     * Buffer to store incomming tokens <code>latestTokens</code>
     * This buffer will be used at the end of the current round.
     */
    protected Vector latestTokens;


    public RoundRobinIterator() {
        playerTokens = new Vector();
        latestTokens = new Vector();
        rrLog = new LogBox("Round Robin Turn Iterator",new heurgame.ui.GraphicalLogInterface());
    }

    /**
     * Sets or Resets the state of the Iterator
     * 
     * @param totalRounds	Positive integer 
     * @param pTokens		A Vector of Player Tokens in the desired order
     */
    public void setup(int totalRounds, Vector pTokens) {
        currentRound = 1;
        this.totalRounds = totalRounds;
        playerTokens = pTokens;
        playerHelper = playerTokens.listIterator();
        currentToken = (PlayerToken) playerHelper.next();
    }

    
    /** 
     * Shifts state to the next turn, and broadcasts the turnChanged Event
     * 
     * @see heurgame.event.turn.AbstractTurnIterator#next()
     */
    public void next() {
        logState("StartofTurn");
        TurnEvent e = new TurnEvent();
        e.totalRounds = totalRounds;
        e.roundNumber = currentRound;
        e.player = currentToken;
        notifyTurnListeners(e);
        //Increment round?
        if (playerHelper.hasNext() == false) {
            if (latestTokens.size() > 0) {
                playerTokens.clear();
                playerTokens.addAll(latestTokens);
                latestTokens.clear();
            }
            playerHelper = playerTokens.listIterator();
            currentRound++;
        }
        currentToken = (PlayerToken) playerHelper.next();
    }

    /**
     * @return True while the current round is less than or equal to the total.
     * If the game has 10 rounds the number of the last round will be 10 and the 
     * iterator will be exhausted before the start of 11.
     *  
     * @see heurgame.event.turn.AbstractTurnIterator#hasMore()
     */
    public boolean hasMore() {
        return (currentRound <= totalRounds);
    }

    /**
     * When there is some change in the order of players or the actual 
     * number of the players themselves it would be good for the 
     * iterator to compensate.
     * 
     * @param newTokens The new order of the tokens for the next round
     */
    public void updateActiveTokens(Vector newTokens) {
        latestTokens = newTokens;
    }

    /**
     * Helper method to simplify adding log entries
     * 
     * @param shortMessage Describes the turn
     */
    protected void logState(String shortMessage) {
        StringBuffer description = new StringBuffer();
        description.append("Total: ");
        description.append(Integer.toString(totalRounds));
        description.append(", Current: ");
        description.append(Integer.toString(currentRound));
        if (currentToken != null) {
            description.append(", Player: ");
            description.append(currentToken.getName());
        }
        rrLog.addEntry(shortMessage,description.toString());
    }

    /** 
     * @see heurgame.event.turn.TokenBased#getCurrentToken()
     */
    public PlayerToken getCurrentToken() {
        return currentToken;
    }
}