#include <stdio.h>
#include <assert.h>

void getFileInfo (char *filename, int *numContigs, int *seqLen, int *numHits){
  FILE *file;
  int dummy, i;
  
  if (!(file = fopen (filename, "r"))){
    fprintf (stderr, "fact: Error opening file: %s\n");
    exit (1);
  }
  
  fscanf (file, "numContigs = %d\n", numContigs);
  fscanf (file, "seqLen = %d\n", seqLen);
  
  *numHits = 0;
  while (!feof (file)){
    if (fscanf (file, "(%d %d)", &dummy, &dummy) == 2){
      for (i = 0; i < *numContigs; i++){
	fscanf (file, "%&d", &dummy);
      }
      while (fgetc (file) != '\n');
      (*numHits)++;
    }
  }

  fclose (file);
}

void getScores (char *filename, int numContigs, int seqLen, int numHits, int ***score, int ***ranges){
  FILE *file;
  int i, j;

  *score = (int **) malloc (sizeof (int *) * numHits); 
  assert (*score);
  *ranges = (int **) malloc (sizeof (int *) * numHits);
  assert (*ranges);
  for (i = 0; i < numHits; i++){
    (*score)[i] = (int *) malloc (sizeof (int) * numContigs);
    assert ((*score)[i]);
    (*ranges)[i] = (int *) malloc (sizeof (int) * 2);
    assert ((*ranges)[i]);
  }

  if (!(file = fopen (filename, "r"))){
    fprintf (stderr, "contigorder: Error opening file: %s\n");
    exit (1);
  }
  
  fscanf (file, "numContigs = %*d\n");
  fscanf (file, "seqLen = %*d\n");
  
  i = 0;
  while (!feof (file) && i < numHits){    
    if (fscanf (file, "(%d %d)", &((*ranges)[i][0]), &((*ranges)[i][1])) == 2){
      for (j = 0; j < numContigs; j++){
	fscanf (file, "%d", &((*score)[i][j]));
      }
      while (fgetc (file) != '\n');
      i++;
    }
  }

  fclose (file);
}

void floodfill (int *labels, int *first, int *last, int numContigs, int here, int groupNum){
  int i;

  labels[here] = groupNum;
  for (i = 0; i < numContigs; i++){
    if (i != here && labels[i] == -1){
      if (!(first[here] > last[i] || last[here] < first[i])){
	floodfill (labels, first, last, numContigs, i, groupNum);
      }
    }
  }
}

int *getLabels (int **score, int numContigs, int numHits){
  int *labels, *first, *last, i, j;
  
  labels = (int *) malloc (sizeof (int) * numContigs); assert (labels);
  first = (int *) malloc (sizeof (int) * numContigs); assert (first);
  last = (int *) malloc (sizeof (int) * numContigs); assert (last);

  for (j = 0; j < numContigs; j++){
    first[j] = -1;
    for (i = 0; i < numHits; i++){
      if (score[i][j] > 0){
	if (first[j] == -1) first[j] = i;
	last[j] = i;
      }
    }
  }

  j = 0;
  for (i = 0; i < numContigs; i++) labels[i] = -1;
  for (i = 0; i < numContigs; i++){
    if (labels[i] == -1){
      floodfill (labels, first, last, numContigs, i, j++);
    }
  }
  
  free (first);
  free (last);
  return labels;
}

int makeRanges (int **score, int numHits, int *cols, int numCols, int **first, int **last){
  int i, j, k, found, numRanges = 1;
  
  for (i = 0; i < numHits; i++){
    for (j = 0; j <= i; j++){
      for (k = found = 0; !found && k < numCols; k++){
	found = (score[i][cols[k]] > 0) && (score[j][cols[k]] > 0);
      }
      if (found) numRanges++;
    }
  }

  *first = (int *) malloc (sizeof (int) * numRanges); assert (*first);
  *last = (int *) malloc (sizeof (int) * numRanges); assert (*last);
  
  (*first)[0] = -1; // initial range
  (*last)[0] = -1; // initial range
  numRanges = 1;

  for (i = 0; i < numHits; i++){
    for (j = 0; j <= i; j++){
      for (k = found = 0; !found && k < numCols; k++){
	found = (score[i][cols[k]] > 0) && (score[j][cols[k]] > 0);
      }
      if (found){
	(*first)[numRanges] = j;
	(*last)[numRanges] = i;
	numRanges++;
      }
    }
  }

  return numRanges;
}

int **calcRangeScores (int **score, int *cols, int numCols, int *first, int *last, int numRanges){
  int i, j, k, **scoreOf;
  
  scoreOf = (int **) malloc (sizeof (int *) * numCols); assert (scoreOf);
  for (i = 0; i < numCols; i++){
    scoreOf[i] = (int *) malloc (sizeof (int) * numRanges); assert (scoreOf[i]);
    for (j = 0; j < numRanges; j++){
      scoreOf[i][j] = 0;
      
      if (j > 0){
	for (k = first[j]; k <= last[j]; k++){
	  scoreOf[i][j] += score[k][cols[i]];
	}
      }
    }
  }

  
  return scoreOf;
}

inline int max (int a, int b){
  if (a > b) return a;
  return b;
}

int getScore (int **score, int *cols, int numCols, int *order, int **mat, int numHits){
  int i, j;

  // compute first row
  mat[0][0] = score[0][order[0]];
  for (i = 1; i < numHits; i++){
    mat[0][i] = mat[0][i-1] + score[i][order[0]];
  }

  // compute rest
  for (i = 1; i < numCols; i++){
    mat[i][0] = max (score[0][order[i]], mat[0][i-1]);
    for (j = 1; j < numHits; j++){
      mat[i][j] = max (mat[i][j-1] + score[j][order[i]],
//		       mat[i-1][j-1] + score[j][cols[i]],
		       mat[i-1][j]);
    }
  }
  
  return mat[numCols - 1][numHits - 1];
}
	      
void search (int **score, int *cols, int numCols, int *order, int *used,
	     int *best, int *bestScore, int pos, int **mat, int numHits, int *first, int *last){
  int i, j, good;

  if (pos == 2){
    for (i = 0; i < 2; i++){
      fprintf (stderr, " %d", order[i]);
    }
    if (*bestScore > 0){
      fprintf (stderr, ", bestscore = %d, bestorder =", *bestScore);
      for (i = 0; i < numCols; i++)
	fprintf (stderr, " %d", best[i]);
      fprintf (stderr, "\n");
    }
    else
      fprintf (stderr, "\n");      
  }

  if (pos < numCols){
    for (i = 0; i < numCols; i++) if (!used[i]){
      for (j = 0, good = 1; j < numCols; j++)
	good = (i == j || used[j] || last[j] >= first[i]);
      if (good){
	used[i] = 1;
	order[pos] = cols[i];
	search (score, cols, numCols, order, used, best, bestScore, pos + 1, mat, numHits, first, last);
	used[i] = 0;
      }
    }
  }
  else {
    i = getScore (score, cols, numCols, order, mat, numHits);
    if (i > *bestScore){
      *bestScore = i;
      memcpy (best, order, sizeof (int) * numCols);
    }
  }
}

void solveOrder (int **score, int numContigs, int numHits, int *cols, int numCols, int **ranges){
  int *first, *last, *order, *used, **mat, *best, bestScore = 0, i, j;

  first = (int *) malloc (sizeof (int) * numCols); assert (first);
  last = (int *) malloc (sizeof (int) * numCols); assert (last);
  order = (int *) malloc (sizeof (int) * numCols); assert (order);
  used = (int *) malloc (sizeof (int) * numCols); assert (used);
  best = (int *) malloc (sizeof (int) * numCols); assert (best);
  mat = (int **) malloc (sizeof (int *) * numCols); assert (mat);
  for (i = 0; i < numCols; i++){
    mat[i] = (int *) malloc (sizeof (int) * numHits); assert (mat[i]);

    first[i] = last[i] = -1;
    for (j = 0; j < numHits; j++){
      if (score[j][i] > 0){
	if (first[i] == -1) first[i] = j;
	last[i] = j;
      }
    }
  }

  search (score, cols, numCols, order, used, best, &bestScore, 0, mat, numHits, first, last);

  fprintf (stderr, "Best score found = %d\n", bestScore);
  fprintf (stderr, "Order:");
  for (i = 0; i < numCols; i++){
    fprintf (stderr, " %d\n", best[i]);
  }
  fprintf (stderr, "\n");

  for (i = 0; i < numCols; i++) free (mat[i]);
  free (first);
  free (last);
  free (mat);
  free (order);
  free (used);
  free (best);
}

void findOrder (int numContigs, int seqLen, int numHits, int **score, int **ranges){
  
  int *labels, group, pos, i;
  int *columns;
  labels = getLabels (score, numContigs, numHits);
  columns = (int *) malloc (sizeof (int) * numContigs);
  assert (columns);

  group = pos = 0;
  while (1){
    for (i = 0; i < numContigs; i++){
      if (labels[i] == group)
	columns[pos++] = i;
    }
    if (pos == 0) break;

    solveOrder (score, numContigs, numHits, columns, pos, ranges);
    pos = 0;
    group++;
  }

  free (labels);
  free (columns);
}

int main (int argc, char **argv){
  int numContigs, seqLen, numHits, i;
  int **score, **ranges;

  if (argc != 2){
    fprintf (stderr, "Usage:\ncontigorder rangefile\n");
    exit (1);
  }
  
  getFileInfo (argv[1], &numContigs, &seqLen, &numHits);
  
  fprintf (stderr, "numContigs = %d, seqLen = %d, numHits = %d\n", numContigs, seqLen, numHits);
  
  getScores (argv[1], numContigs, seqLen, numHits, &score, &ranges);
  findOrder (numContigs, seqLen, numHits, score, ranges);
  
  for (i = 0; i < numHits; i++){
    free (score[i]);
    free (ranges[i]);
  }
  free (score);
  free (ranges);  
  
  return 0;
}
 
