import java.applet.Applet;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.Vector;
import java.text.*;

class Formats {
    static DecimalFormat scoreFormat = new DecimalFormat ("##.000");

    public static double getDouble0To1 (TextField text) throws Exception {
	Double x = Double.valueOf(text.getText());
	if (x.isNaN() || x.isInfinite() || x < 0. || x > 1.) {
	    throw (new Exception("Input must be finite between 0 and 1 inclusive"));
	} else { return x; }
    }

}

class PlanetWeightUpdater implements TextListener {
    TextField planet1Weight;
    public Label planet2Weight;

    PlanetWeightUpdater(TextField p1) {
	planet1Weight = p1;
	planet2Weight = new Label("");
    }

    public void textValueChanged (TextEvent te) {
	try {
	    double d = Formats.getDouble0To1(planet1Weight);
	    planet2Weight.setText(Formats.scoreFormat.format(1. - d));
	} catch (Exception e) {
	    planet2Weight.setText("");
	}
    }
}

class PlanetInputs
{
    public Button button = new Button("Submit planet parameters");
    public Panel contents = new Panel ();

    int input_width = 8;

    public TextField planet1_x = new TextField(input_width);
    public TextField planet1_y = new TextField(input_width);
    public TextField planet1_weight = new TextField(input_width);
    public TextField planet2_x = new TextField(input_width);
    public TextField planet2_y = new TextField(input_width);

    /** values here are just for testing - so I don't have to type
     * them in every time */
    double x1 = 0.4, y1 = 0.1, w1 = 0.2, x2 = 0.8, y2 = 0.2, w2 = 0.8;

    public void init () {
	Panel planet_screen = contents;
	planet_screen.setLayout(new BoxLayout(planet_screen, BoxLayout.Y_AXIS));

	Panel planet_inputs = new Panel ();
	GridLayout planet_layout = new GridLayout(2, 3);
	planet_layout.setHgap(5);
	planet_layout.setVgap(5);
	planet_inputs.setLayout(planet_layout);
	planet_inputs.setPreferredSize(new Dimension(600,50));
	planet_inputs.setMaximumSize(new Dimension(600,50));

	planet_inputs.add(new Label("Planet 1 (x, y, weight):"));
	planet_inputs.add(planet1_x);
	planet_inputs.add(planet1_y);
	planet_inputs.add(planet1_weight);

	PlanetWeightUpdater planetWeightUpdater = new PlanetWeightUpdater(planet1_weight);
	planet1_weight.addTextListener(planetWeightUpdater);

	planet_inputs.add(new Label("Planet 2 (x, y, weight):"));
	planet_inputs.add(planet2_x);
	planet_inputs.add(planet2_y);
	planet_inputs.add(planetWeightUpdater.planet2Weight);
	
	planet_screen.add(planet_inputs);
	button.setMaximumSize(new Dimension(300,25));
	planet_screen.add(button);
	Label instructions = new Label("Coordinates are from (0,0) to (1,1).  Total mass of planets is 1.  The target will be at (0.75, 0.75).");
	instructions.setAlignment(Label.CENTER);
	planet_screen.add (instructions);
    }
    
    public void parse () throws Exception {
	x1 = Formats.getDouble0To1(planet1_x);
	y1 = Formats.getDouble0To1(planet1_y);
	w1 = Formats.getDouble0To1(planet1_weight);
	x2 = Formats.getDouble0To1(planet2_x);
	y2 = Formats.getDouble0To1(planet2_y);
	w2 = 1. - w1;
    }
}

class Vector2 {
    double x, y;

    public Vector2(double a, double b) { x = a; y = b; }

    public double magnitude () {
	return Math.sqrt (x * x + y * y);
    }

    public Vector2 scale(double c) {
	return new Vector2(x * c, y * c);
    }

    public Vector2 norm() {
	return scale(1 / magnitude());
    }

    public Vector2 sub(Vector2 v2) {
	return new Vector2(x - v2.x, y - v2.y);
    }

    public Vector2 add(Vector2 v2) {
	return new Vector2(x + v2.x, y + v2.y);
    }

    public Vector2 direction(Vector2 from, Vector2 to) {
	return to.sub(from).norm();
    }
}

class Projectile {
    Vector2 position;
    Vector2 velocity;

    Projectile(Vector2 p, Vector2 v) {
	position = p;
	velocity = v;
    }
}

class Planet {
    Vector2 position;
    double mass;
    
    Planet(Vector2 p, double m) {
	position = p;
	mass = m;
    }
}

class GameStateDisplay extends Canvas {
    /** units are: distance ^ 3 / (mass * time ^ 2) */
    double gravitational_constant = 1e-8;

    int planet_size = 6;

    Vector<Planet> planets;
    Projectile projectile;
    Vector2 target;

    /** in milliseconds */
    int num_timesteps = 0;
    int max_num_timestamps = 10000;

    /** in milliseconds - our physics evaluations are discreet at this timescale */
    double step_size = 0.005;

    /** will be set to true if the projectile ever leaves (0,0) to
     * (1,1), even if only at a time that is not displayed */
    public boolean projectileLeftScreen = false;

    /** The best score of all steps (including non-displayed ones) */
    public double bestScore;

    double score () {
	return target.sub(projectile.position).magnitude ();
    }

    void maybeUpdateScore () { 
	double score = score () ;
	if (score < bestScore) { bestScore = score; }
	if(projectile.position.x < 0.
	   || projectile.position.y < 0.
	   || projectile.position.x > 1.
	   || projectile.position.y > 1.)
	    { projectileLeftScreen = true; }
    }

    void drawPoint(Graphics g, Color c, Vector2 p) {
	g.setColor(c);
	int norm_x = (int)((getSize().width - planet_size) * p.x);
	int norm_y = (int)((getSize().height - planet_size) * (1 - p.y));
	g.fillRect(norm_x,
		   norm_y,
		   planet_size, planet_size);
    }

    void drawPlanet(Graphics g, Planet p) {
	drawPoint(g, Color.black, p.position);
    }

    Timer timer;
    
    public void paint (Graphics g) {
	super.paint (g);
	setBackground( Color.blue );
	for (int i = 0 ; i < planets.size (); ++i) {
	    drawPlanet(g, planets.get(i));
	}
	drawPoint(g, Color.red, projectile.position);
	drawPoint(g, Color.green, target);
    }

    public void stop() { timer.stop (); }

    /** advance our internal state by one step_size in time */
    void do_step() {
	Vector2 acceleration = new Vector2(0., 0.);
	for(int i = 0; i < planets.size(); ++i) {
	    Planet planet = planets.get(i); 
	    Vector2 r1 = planet.position.sub(projectile.position);
	    /** a = G * m_planet / r^2 */
	    /** acceleration is measured in distance units / (time units ^ 2) */
	    double r1_mag = r1.magnitude ();
	    double f1_mag = 
		gravitational_constant
		* planet.mass
		/ (r1_mag * r1_mag);
	    /** total change to velocity then is (a * step_size) */
	    acceleration = acceleration.add(r1.scale (f1_mag / r1_mag));
	}
	Vector2 new_velocity = projectile.velocity.add(acceleration.scale(step_size));
	/** compute our average velocity during this time step */
	Vector2 average_velocity =
	    projectile.velocity.add(new_velocity).scale(0.5);
	/** compute our new position using the average velocity */
	Vector2 new_position = 
	    projectile.position.add(average_velocity.scale(step_size));
	projectile = new Projectile(new_position, new_velocity);
	maybeUpdateScore ();
    }

    public void step(int ms) {
	int steps = (int) ((double)ms / step_size);
	for(int i = 0; i < steps; ++i) {
	    do_step ();
	}
	repaint ();
    }

    static int delay = 30;
    
    GameStateDisplay (PlanetInputs planet_inputs,
		      Projectile p, 
		      Vector2 t) {
	projectile = p;
	target = t;
	bestScore = score ();
	planets = new Vector<Planet>();
	planets.add(new Planet(new Vector2(planet_inputs.x1, 
					   planet_inputs.y1),
			       planet_inputs.w1));
	planets.add(new Planet(new Vector2(planet_inputs.x2, 
					   planet_inputs.y2),
			       planet_inputs.w2));
	setPreferredSize(new Dimension(400,400));
	setMinimumSize(new Dimension(400,400));
	setMaximumSize(new Dimension(400,400));
    }
}

class GameState {
    public Timer displayUpdate;
    public Timer endGame;
    public GameStateDisplay display;

    public void step(int delay) {
	display.step(delay);
    }

    public void stop () {
	displayUpdate.stop ();
	endGame.stop ();
    }
}

class SeekerInputs extends Panel {
    TextField angle = new TextField (8);
    Button fire;

    public SeekerInputs(PlanetInputs pi, Projectile p, Vector2 t, double bestScore, int shotsRemaining, String name) {
	setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
	add(new GameStateDisplay(pi, p, t));
	Panel superText = new Panel ();
	Panel text = new Panel ();
	text.setLayout(new BoxLayout(text, BoxLayout.PAGE_AXIS));
	Panel angleInput = new Panel ();
	angleInput.setLayout(new BoxLayout(angleInput, BoxLayout.LINE_AXIS));
	text.add (new Label (name + "'s turn"));
	if(!(new Double(bestScore).isNaN())) {
	    text.add (new Label("Best score: " 
			   + Formats.scoreFormat.format(bestScore)));
	}
	text.add (new Label ("Shots remaining: " + shotsRemaining));
	angleInput.add(new Label("Angle (in degrees - 0 is along the x axis, 90 is along the y axis)"));
	angleInput.add(angle);
	angleInput.add(Box.createHorizontalGlue());
	text.add(angleInput);
	
	String fireText;
	
	if (shotsRemaining == 4) { fireText = "Let's do this!"; }
	else if (shotsRemaining == 3) { fireText = "Let's do this again!"; }
	else if (shotsRemaining == 2) { fireText = "Alright, now aim at the target this time."; }
	else if (shotsRemaining == 1) { fireText = "Last one, don't mess it up!"; }
	else { fireText = "Fire at will."; }

	fire = new Button (fireText);
	text.add (fire);
	superText.add(text);
	add(superText);
	validate ();
    }
}

public class GravityGame extends Applet implements MouseListener, ActionListener
{
    String text = "Hello Worldb";

    int state = 0;
    /** 0 = entering input about planets
        1 = player 2 taking shots
        2 = player 1 taking shots
        3 = score display
     */

    PlanetInputs planet_inputs = new PlanetInputs ();
    GameState game;
    Label scoreDisplay = new Label("");

    SeekerInputs seekerInputs;

    /** when has encoding 'None' as some particular value ever bit us? */
    double seekerBestScore = Double.NaN;
    int seekerShotsRemaining = 5;
    boolean seekerIsShooting = true;

    double placerBestScore = Double.NaN;
    int placerShotsRemaining = 1;

    /** Maximum time to let a projectile move (in ms) */
    int gameLength = 10000;

    double initialVelocity = 0.0003;
    Vector2 defaultTarget = new Vector2(0.75, 0.75);
    Projectile defaultProjectile = new Projectile(new Vector2(0., 0.), new Vector2(0., 0.));

    void doScoreDisplay () {
	add(new Label ("Placer score: " + Formats.scoreFormat.format(placerBestScore)));
	add(new Label ("Seeker score: " + Formats.scoreFormat.format(seekerBestScore)));
	String winner;
	if (seekerBestScore < placerBestScore) { winner = "Seeker"; } else { winner = "Placer"; }
	add(new Label ("Winner: " + winner));
	validate ();
    }

    public void doPlacerSeekerInputs(double bestScore, int shotsRemaining, String name) {
	seekerInputs = new SeekerInputs(planet_inputs,
					defaultProjectile,
					defaultTarget,
					bestScore,
					shotsRemaining,
					name);
	state = 1;
	add(seekerInputs);
	seekerInputs.fire.addActionListener(this);
	validate ();

    }

    public void doPlacerInputs () {
	seekerIsShooting = false;
	doPlacerSeekerInputs(placerBestScore, placerShotsRemaining, "Placer");
    }

    public void doSeekerInputs () {
	doPlacerSeekerInputs(seekerBestScore, seekerShotsRemaining, "Seeker");
    }

    double endSeekerInputs () throws Exception {
	Double x = Double.valueOf(seekerInputs.angle.getText());
	if (x.isNaN() || x.isInfinite() || x < 0. || x > 90.) {
	    throw (new Exception("Input must be finite between 0 and 90 inclusive"));
	} else {
	    remove(seekerInputs);
	    validate();
	    return x;
	}
    }

    double minIgnoreNan(double x1, double x2) {
        if (new Double(x1).isNaN()) { return x2; }
        else if (new Double(x2).isNaN()) { return x1; }
        else { return Math.min(x1, x2); }
    }

    public void stopGame() {
	remove(game.display);
	remove(scoreDisplay);
	validate ();
	game.stop();

	if(seekerIsShooting) { seekerBestScore = minIgnoreNan(seekerBestScore, game.display.bestScore); }
	else { placerBestScore = minIgnoreNan(placerBestScore, game.display.bestScore); }
	
	game = null;

	if(seekerShotsRemaining > 0) { doSeekerInputs (); }
	else if (placerShotsRemaining > 0) { doPlacerInputs(); }
	else { doScoreDisplay (); }
    }

    public void startGame(int ms, Projectile p) {
	game = new GameState();
	game.display = 
	    new GameStateDisplay(planet_inputs, 
				 p,
				 new Vector2(0.75, 0.75));
	add(game.display);
	add(scoreDisplay);
	validate ();

	ActionListener f = new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    stopGame();
		}
	    };

	game.endGame = new Timer(ms, f);
	game.endGame.setRepeats(false);
	game.endGame.start ();

	ActionListener f2 = new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    game.step(GameStateDisplay.delay);
		    scoreDisplay.setText
			("Score: " 
			 + Formats.scoreFormat.format(game.display.bestScore));
		    scoreDisplay.repaint ();
		    validate ();
		    repaint ();

		    if (game.display.projectileLeftScreen) { stopGame(); }
		}
	    };
	game.displayUpdate = new Timer(GameStateDisplay.delay, f2);
	game.displayUpdate.start ();
    }

    public void init() {
	BoxLayout layout = new BoxLayout(this, BoxLayout.PAGE_AXIS);
	setLayout(layout);
	scoreDisplay.setAlignment(Label.CENTER);

	planet_inputs.init ();
	
	add(planet_inputs.contents);
	planet_inputs.button.addActionListener(this);
	validate ();
    }

    public void actionPerformed(ActionEvent ae) {
	if (state == 0) { 
	    try { 
		planet_inputs.parse ();
		state = 1;
		remove(planet_inputs.contents);
		doSeekerInputs();
	    } catch (Exception e) {}
	} else if (state == 1) {
	    try {
		double angle = endSeekerInputs ();
		angle = Math.toRadians(angle);
		Vector2 projectileVelocity =
		    (new Vector2(Math.cos(angle),
				 Math.sin(angle))).scale(initialVelocity);
		if(seekerIsShooting) { seekerShotsRemaining--; } else { placerShotsRemaining--; }
		startGame(gameLength, new Projectile(new Vector2(0.,0.), projectileVelocity));
	    } catch (Exception e) {}
	}
    }

    public void mouseReleased(MouseEvent e) { }
    public void mouseClicked(MouseEvent e) { }
    public void mousePressed(MouseEvent e) { }
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}

    public void paint ( Graphics gr )
    {
	setBackground( Color.gray );
    }
}
