/*
 grb_finder_swt.c : This is the core file of the new algorithm

 Some points to note:
 (1) Please read the algorithm from yunyue's paper first.
 (2) The main structure of the codes consists of three parts: initializaton, build the tree and detect the bursts
 (3) Initialization:
     Besides zeroing the arrays or variables, the most important operation is to store the time series of each bucket
     in the "databuffer". If the number of data is more than space available, the sky will be divided into half.
 (4) Build the tree: 
     void build_tree(int x, int y);
 (5) Detect the bursts:
     detect_burst_bin();
 (6) Here addressing the current data files, we assume the longest time span is 600s
 (7) For debug purpose, some codes and flags are not removed.

 The basic idea is as follows:
 (1) Compute the background over the whole time length for later use
 (2) Store the time series in "databuffer" one bucket by one bucket
 (3) Detect the burst one bucket by one bucket
 (4) For each bucket, do the following operations:
     a. Build the tree
     b. In the burst detection procesure, the coarse detection is done first 
     c. As the algorithm suggests. If this first step is passed, detailed search is taken.
 */

#include <math.h>
#include "Reconstruction.h"
#include "Offline.h"
#include "cfortran.h"
#include "hbook.h"
#include "grb.h"
#include "stdio.h"
#include <stdlib.h>         /* For _MAX_PATH definition */
#include <malloc.h>
#include <assert.h>
#include "time.h"

#define LOCAL_DEBUG 0
#define SEARCH_THRESH 2           
#define TIME_RESOLUTION 0.01      //According to the resolution of old codes, new codes need not differentiate 0.0012 and 0.0013.
#define DATA_SIZE 70000           //600s/0.01=60000 near 2^16=65536
#define DATA_DENSE_SIZE 4000      //Maximal event number for a single bucket
#define TEST_THRESHOLD NDUR       //Control the number of tested durtion
#define NRA_BINS_LOCAL  NRA_BINS  //Our test window size
#define NDEC_BINS_LOCAL NDEC_BINS //Our test window size
#define SAMPLE_STEP 3             //jump evey 3 bucket 

#define MIN_MEMORY 140000000      //The size of databuffer, which is limited by system. This number is determined after lots of tries
/*-------------------------*/
double pois( double, double);
double intpois( double, double);

//For debug use, meaningless to the final program.  
  int ntotal=0;			    
  int nlarger2=0;
  int nlarger1p4=0;
  int npasscoarse=0;
  int ndetail=0;     
  int ndlarger1p4=0;
  int nburst=0;
  int largest_10s=0;
  int smallest_10s=1000000;
//--------------------------

  int globalfile=0;
  GRB_CANDIDATE cand_dur[TEST_THRESHOLD];
  int size=DATA_SIZE;				// size of the data
  int swtree[6200];                 // Shift wavelet tree
  int swt_index[6200];              // Two remember the position of starting data of each tree level in databuffer. This is for easy search.
  int last_non_empty[23];           // Because the length of time series is not the pow of 2, so 0 padding is made. This array is to remember the position from which 0 padding begins. 
  int swt_ladder[23]={0,2048,3072,4096,4608,5120,5376,5632,5760,5888,5952,6016,6048,6080,6096,6112,6120,6128,6132,6136,6138,6140,6141}; //The sum of each tree level and its below levels.
  float BINSF_span;
  int xallsky_lookup[60000][TEST_THRESHOLD];//A lookup table to save time
  int threshold_buffer_larger[30][3751]; //
  int** overall_threshold;          //For duration>10. Each time the same background is used to compute the threshold, the value is taken down for later use. This is to avoid computing the same threshold twice.
  int** overall_threshold_pow2;     //For the threshold computing when time length is the pow of 2. Each time the same background is used to compute the threshold, the value is taken down for later use. This is to avoid computing the same threshold twice.
  int duration_int[TEST_THRESHOLD]; // (int)(duration[i]/TIME_RESOLUTION)

  double current_start_time;        //the start time of current bucket
  double current_end_time;          //the end time of current bucket
  
  int nlevel=0;	            // level index
  int TIME_LENGTH;          // time length of each file. e.g. 600s for the first seached file
  int time_index=0;			// index on the time axis from the beginning of this searched file
  int nCounts[NRA_BINS_LOCAL][NDEC_BINS_LOCAL]; //The total events in a bucket
  int bucket_index[NRA_BINS_LOCAL][NDEC_BINS_LOCAL];//For initializtion use
  int current_i;			// the iha of current time series
  int current_j;			// the idec of current time series
  int current_dur;			// the index of current duration.
  int nBuff;
  double* time_event_table;	// reorganize the evbuffer in terms of TIME_RESOLUTION
  int time_index_table[DATA_SIZE]; // For the background computation
  int sigmap_index[NRA_BINS_LOCAL][NDEC_BINS_LOCAL][2]; // Remember the start and end index for the time series of each bucket in databuffer
  int databuffer[MIN_MEMORY]; 
    /*Put all the data in databuffer. 
	For example
	The original sky is as follows:
	a b c
	d e f
	g h i
	Then in databuffer they are stored in the following way:
	a1 a2 a3..., b1 b2 b3...., c1 c2 c3...,.....i1 i2 i3....
	Here a1 a2 a3 means the data in bucket "a" listed according to the time order, in other word,
	a1 a2 a3... is a time series. Others are similar.
	We store the data densely to save the space. Otherwise much more memory is needed.
  */

  int sample_step=0;
  double start_time, compute_time, end_time;
  int duration_cutoff_index2;  //Useless 
  int duration_cutoff_index10; //The duration index from which duration will be larger than 10
  int data_base_threshold[DATA_SIZE]; //In the case of smaller than 10. Each time the same background is used to compute the threshold, the value is taken down for later use. This is to avoid computing the same threshold twice.
  
  void find_duration_cutoff();
  int sum_time(double t_start, double t_end, int start_index, int* last_start, int* last_end, int last_value);
  int get_overall_events(int t_start, int length);
  int find_back_base_least_coarse(int t_start_event, int length);
  int find_back_base_least_detail(int t_start_event);
  void compute_10s_base_threshold();
  void burst_judge_bin(double start, int length, int i_signal);
  int burst_judge_bin_coarse(int start, int length, int i_signal, int is_even);
  int sum(double start, double end, int* last_start, int* last_end, int last_value);
  int sum_bin(double start, double end, int* last_start);
  void detect_burst_bin();
  void build_tree(int x, int y);
  int burst_judge_bin_coarse_sub(int start_i, int start_j, int length, int i_signal);
  int booltest();
  void valueTest();
/*The input will be set as global*/
  double tstart,tend;
  int ipublish, nEval;
  char  str[300];
  GRB_CANDIDATE cand;
  MAP *maps;
  PROB *prob;
  unsigned int julDate;
  int runum; 
  int subrunum;
  int nTail;
  EVENT *evbuffer;
  double* duration;
// Main program. Assign the events and launch the burst detection
int grb_finder_swt(EVENT *evbuffer0, MAP *maps0,  PROB *prob0, double* duration0, int nTail0, 
				   int nTailBack, int nBuff0, unsigned int julDate0,
				   int runum0, int subrunum0) 
{
  
  printf("Come into swt\n");
  start_time=clock();
  maps=maps0;
  prob=prob0;
  nBuff=nBuff0; ////////////////////////Here we made changes
  julDate=julDate0;
  runum=runum0; 
  subrunum=subrunum0;
  nTail=nTail0;
  evbuffer=evbuffer0;
  duration=duration0;
  int iha,idec;
  int i=0,j=0, k=0;
  int nhabins;
  int dha,xha,ddec,xdec, iend;
  ntotal=0;      //Total search number  
  nlarger2=0;
  nlarger1p4=0;
  npasscoarse=0;
  ndetail=0;     //The detailed search number
  ndlarger1p4=0;
  nburst=0;
  //Initialization---------------------
  globalfile++;
  for(i=0;i<TEST_THRESHOLD;i++){
	  cand_dur[i].grbprob=1;
	  duration_int[i]=(int)(duration[i]/TIME_RESOLUTION);
  }
  BINSF_span=PROB_BINSF/prob->span;
  printf("nBuff=%d\n", nBuff);
  time_event_table=(double*)calloc(nBuff,sizeof(double) ); //time_event_table=new int[nBuff]; 

  printf("find the duration cutoff2\n");
  find_duration_cutoff();
  printf("duration_cutoff_index2=%d, 10=%d\n", duration_cutoff_index2, duration_cutoff_index10);

  overall_threshold=(int**)calloc(TEST_THRESHOLD-duration_cutoff_index10, sizeof(int*));
  overall_threshold_pow2=(int**)calloc(TEST_THRESHOLD-duration_cutoff_index10, sizeof(int*));
  for(i=0;i<TEST_THRESHOLD-duration_cutoff_index10;i++){
	overall_threshold[i]=(int*)calloc(60000, sizeof(int));
	overall_threshold_pow2[i]=(int*)calloc(60000, sizeof(int));
  }

  if( time_event_table == NULL ){
      printf( "Insufficient memory available\n" );
  }  

  int* count_ptr=&nCounts[0][0];
  for(i=0;i<NRA_BINS_LOCAL*NDEC_BINS_LOCAL;i++){
	*(count_ptr+i)=0;
  }

  int* th_buffer_ptr=&threshold_buffer_larger[0][0];
  for(i=0;i<10*3751;i++){
	 *(th_buffer_ptr+i)=-1;
  }
  


  tstart      = evbuffer[nTail].time-duration[0]; 
  if (tstart<evbuffer[nTailBack].time) tstart=evbuffer[nTailBack].time;
  iend        = nTail+nBuff-1;
  if (iend>EVBUFFSIZE) iend-=EVBUFFSIZE;
  tend        = evbuffer[iend].time;
  //printf("monitor point6\n");

  TIME_LENGTH=(int)((tend-tstart)/TIME_RESOLUTION);
  printf("TIME_LENGTH=%d\n",TIME_LENGTH);
 

  
  int* indexptr=&time_index_table[0];
  int* thresholdptr=&data_base_threshold[0];
  for(i=0;i<DATA_SIZE;i++){
	  *(indexptr+i)=0;
	  *(thresholdptr+i)=0;
  } 

  nEval       = 0;
  ipublish    = 1;   

  sprintf(str," %f %f ",tstart,tend);
  printf(" tstart=%f tend=%f\n",tstart,tend);
  grb_finder_logger(str);
 
  printf("estimate the number for each bucket\n");
   
  i=nTail;
  j=0;
  int total_bucket=0;
  //finish the variable initialization-------------------------------------

  //The following is to measure the total events to determine if the 
  //space available is sufficient. And at the same time the information 
  //for data allocation is collected.
  while (evbuffer[i].time<=tend&&evbuffer[i].time>=tstart) {
		  iha = evbuffer[i].iha;
		  idec= evbuffer[i].idec;
		  nhabins = prob->habins[idec];
		  time_index=(int)((evbuffer[i].time-tstart)/TIME_RESOLUTION);
		  time_event_table[j]=evbuffer[i].time;
		  time_index_table[time_index]++;
		  if (iha+nhabins>=prob->hamin && iha-nhabins<=prob->hamax &&
			idec+SEARCHBIN>=prob->decmin[iha] && idec-SEARCHBIN<=prob->decmax[iha]) {
			total_bucket+=(2*nhabins+1)*(2*SEARCHBIN+1);
			for (dha=-nhabins;dha<=nhabins;dha++) {
			  for (ddec=-SEARCHBIN;ddec<=SEARCHBIN;ddec++) {			 
				  nCounts[iha+dha][idec+ddec]++;
			  }
			}
		  }
		  i++;
		  j++;
		  if (i==EVBUFFSIZE) i=0;
  } 
  
  //compute the background;
  compute_10s_base_threshold(); 

  int ii,jj;
  //Here when globalfile=1, there are more data than acceptable for the memory, so the strategy is 
  //to divide the whole data into two part. If the system allow the codes to use a little more 
  //memory(about 800M) this globalfile stuff can be removed. 
  //If in the new data file, some searched file has as many data as three times of what
  //memory allows, the data should be devided into three parts. Similarly, if no file contain large
  //data set, such If/else stuff can be completely removed.
  if(globalfile>=2){	 
	  int totaldata=0;
	  int i_start=0;
	  for(i=0;i<NRA_BINS_LOCAL;i++){
			for(j=0;j<NDEC_BINS_LOCAL;j++){
				if(nCounts[i][j]>=5){ //Don't compute on those bucket of less than 5 events.
					sigmap_index[i][j][0]=i_start;
					sigmap_index[i][j][1]=i_start+nCounts[i][j];
					bucket_index[i][j]=i_start;
					i_start=sigmap_index[i][j][1];
					totaldata+=nCounts[i][j];		
				}
			}
	  }
	  if(totaldata>MIN_MEMORY){
			printf("insufficient space****************************\n");
			printf("totaldata=%d\n", totaldata);
	  }else{
 			printf("sufficient space\n");
			printf("totaldata=%d\n", totaldata);
	  }
	  //The following is to store the time series of each bucket in databuffer
	  printf("store the events in databuffer\n");
	  i=nTail;
	  j=0;
	  int local_x;
	  int local_y;
	  double dis_start=clock();
	  while (evbuffer[i].time<=tend&&evbuffer[i].time>=tstart) {
		  iha = evbuffer[i].iha;
		  idec= evbuffer[i].idec;
		  nhabins = prob->habins[idec];
		  if (iha+nhabins>=prob->hamin && iha-nhabins<=prob->hamax &&
			  idec+SEARCHBIN>=prob->decmin[iha] && idec-SEARCHBIN<=prob->decmax[iha]) {
			for (dha=-nhabins;dha<=nhabins;dha++) {
			  for (ddec=-SEARCHBIN;ddec<=SEARCHBIN;ddec++) {
				 if(iha+dha>=0&&iha+dha<NRA_BINS_LOCAL&&idec+ddec>=0&&idec+ddec<NDEC_BINS_LOCAL){
					 if(nCounts[iha+dha][idec+ddec]>=5){
						 local_x=iha+dha;
						 local_y=idec+ddec;
						 databuffer[bucket_index[local_x][local_y]]=i;
						 bucket_index[local_x][local_y]++;                                         						 
					 }
				 }
			  }
			}
		  }
		  i++;
		  j++;
		  if (i==EVBUFFSIZE) i=0;
	  }

	  double dis_end=clock();
	  printf("-----The distribute time=%f\n", (double)(dis_end-dis_start));
      //Burst detection----------------------------------
	  for(i=1;i<NRA_BINS_LOCAL;i+=SAMPLE_STEP){
		   if(i==(int)(i/10)*10) printf("i=%d--------------------------\n",i);
		   current_i=i;
		   for(j=1;j<NDEC_BINS_LOCAL;j+=SAMPLE_STEP){ 
			  current_j=j;
			  current_start_time=evbuffer[databuffer[sigmap_index[i][j][0]]].time;
			  current_end_time=evbuffer[databuffer[sigmap_index[i][j][1]-1]].time;
			  if(nCounts[i][j]>=5) //only search when beyond the filter threshold
			  {	
				 build_tree(i,j);
				 for(k=0;k<TEST_THRESHOLD;k++){ 
					current_dur=k;
					detect_burst_bin();					 				
				 } //end of the current_dur				   
			  }// 
		   }// end of j
	  } //end of i
	  //End of burst detection----------------------------------
	  //Report the burst
	  if(ipublish==0){
		  ipublish=2;
		  for(i=0;i<TEST_THRESHOLD;i++){
			  if(cand_dur[i].grbprob!=1){
				grb_finder_publish(&ipublish, &cand_dur[i], evbuffer, prob);
			  }
		  }
	  }  
  }else{ //globalfile=1
	  //Just as pointed above, memory limitation results in that data are divided into half.
	  int NRA_half_size=900;
	  int totaldata=0;
	  int i_start=0;
	  for(i=0;i<NRA_half_size;i++){
			for(j=0;j<NDEC_BINS_LOCAL;j++){
				if(nCounts[i][j]>=5){
					sigmap_index[i][j][0]=i_start;
					sigmap_index[i][j][1]=i_start+nCounts[i][j];
					bucket_index[i][j]=i_start;
					i_start=sigmap_index[i][j][1];
					totaldata+=nCounts[i][j];		
				}
			}
	  }
	  if(totaldata>MIN_MEMORY){
	     printf("insufficient space****************************\n");
		 printf("totaldata=%d\n", totaldata);
	  }else{
 	 	printf("sufficient space\n");
		printf("totaldata=%d\n", totaldata);
	  }
	  printf("distribute the events\n");
	  i=nTail;
	  j=0;
	  double dis_start=clock();
	  while (evbuffer[i].time<=tend&&evbuffer[i].time>=tstart) {
		  iha = evbuffer[i].iha;
		  idec= evbuffer[i].idec;
		  nhabins = prob->habins[idec];
		  if (iha+nhabins>=prob->hamin && iha-nhabins<=prob->hamax &&
			  idec+SEARCHBIN>=prob->decmin[iha] && idec-SEARCHBIN<=prob->decmax[iha]) {
			for (dha=-nhabins;dha<=nhabins;dha++) {
			  for (ddec=-SEARCHBIN;ddec<=SEARCHBIN;ddec++) {
				 if(iha+dha>=0&&iha+dha<NRA_half_size&&idec+ddec>=0&&idec+ddec<NDEC_BINS_LOCAL){
					if(nCounts[iha+dha][idec+ddec]>=5){
						 databuffer[bucket_index[iha+dha][idec+ddec]]=i;
						 bucket_index[iha+dha][idec+ddec]++;
                                         						 
					}
				 }
			  }
			}
		  }
		  i++;
		  j++;
		  if (i==EVBUFFSIZE) i=0;
	  }
	  double dis_end=clock();
	  printf("-----The distribute time=%f\n", (double)(dis_end-dis_start));
	  
	  //Now Compute 
      
	  for(i=1;i<NRA_half_size;i+=SAMPLE_STEP){
		   if(i==(int)(i/10)*10) printf("i=%d--------------------------\n",i);
		   current_i=i;
		   for(j=1;j<NDEC_BINS_LOCAL;j+=SAMPLE_STEP){ 
			  current_j=j;
			  current_start_time=evbuffer[databuffer[sigmap_index[i][j][0]]].time;
			  current_end_time=evbuffer[databuffer[sigmap_index[i][j][1]-1]].time;
			  if(nCounts[current_i][current_j]>=5) //only search when beyond the filter threshold
			  {	
				 build_tree(i,j);
				 for(k=0;k<TEST_THRESHOLD;k++){ 
					current_dur=k;
					detect_burst_bin();					 				
				 } //end of the current_dur				   
			  }// 
		   }// end of j
	  } //end of i
	  
	  if(ipublish==0){
		  ipublish=2;
		  for(i=0;i<TEST_THRESHOLD;i++){
			  if(cand_dur[i].grbprob!=1){
				grb_finder_publish(&ipublish, &cand_dur[i], evbuffer, prob);
			  }
		  }
	  } 

	  //second half--------------------------------------------------------------------------------
	  
	  totaldata=0;
	  i_start=0;
	  for(i=NRA_half_size;i<NRA_BINS_LOCAL;i++){
		 for(j=0;j<NDEC_BINS_LOCAL;j++){
			if(nCounts[i][j]>=5){
				sigmap_index[i][j][0]=i_start;
				sigmap_index[i][j][1]=i_start+nCounts[i][j];
				bucket_index[i][j]=i_start;
				i_start=sigmap_index[i][j][1];
				totaldata+=nCounts[i][j];		
			}
		 }
	  }
	  if(totaldata>MIN_MEMORY){
	     printf("insufficient space****************************\n");
		 printf("totaldata=%d\n", totaldata);
	  }else{
 	 	printf("sufficient space\n");
		printf("totaldata=%d\n", totaldata);
	  }
	  printf("distribute the events\n");
	  i=nTail;
	  j=0;
	  dis_start=clock();
	  while (evbuffer[i].time<=tend&&evbuffer[i].time>=tstart) {
		  iha = evbuffer[i].iha;
		  idec= evbuffer[i].idec;
		  nhabins = prob->habins[idec];
		  if (iha+nhabins>=prob->hamin && iha-nhabins<=prob->hamax &&
			  idec+SEARCHBIN>=prob->decmin[iha] && idec-SEARCHBIN<=prob->decmax[iha]) {
			for (dha=-nhabins;dha<=nhabins;dha++) {
			  for (ddec=-SEARCHBIN;ddec<=SEARCHBIN;ddec++) {
				 if(iha+dha>=NRA_half_size&&iha+dha<NRA_BINS_LOCAL&&idec+ddec>=0&&idec+ddec<NDEC_BINS_LOCAL){
					 if(nCounts[iha+dha][idec+ddec]>=5){
						 databuffer[bucket_index[iha+dha][idec+ddec]]=i;
						 bucket_index[iha+dha][idec+ddec]++;
                                         						 
					 }
				 }
			  }
			}
		  }
		  i++;
		  j++;
		  if (i==EVBUFFSIZE) i=0;
	  }
	  dis_end=clock();
	  printf("-----The distribute time=%f\n", (double)(dis_end-dis_start));
	  
	  //Now Compute 
  
	  for(i=1+NRA_half_size;i<NRA_BINS_LOCAL;i+=SAMPLE_STEP){
		   if(i==(int)(i/10)*10) printf("i=%d--------------------------\n",i);
		   current_i=i;
		   for(j=1;j<NDEC_BINS_LOCAL;j+=SAMPLE_STEP){ 
			  current_j=j;
			  current_start_time=evbuffer[databuffer[sigmap_index[current_i][current_j][0]]].time;
			  current_end_time=evbuffer[databuffer[sigmap_index[current_i][current_j][1]-1]].time;
			  if(nCounts[current_i][current_j]>=5) //only search when beyond the filter threshold
			  {	
				 build_tree(i,j);
				 for(k=0;k<TEST_THRESHOLD;k++){ 
					current_dur=k;
					detect_burst_bin();					 				
				 } //end of the current_dur				   
			  }// 
		   }// end of j
	  } //end of i
	 
	  if(ipublish==0){
		  ipublish=2;
		  for(i=0;i<TEST_THRESHOLD;i++){
			  if(cand_dur[i].grbprob!=1){
				grb_finder_publish(&ipublish, &cand_dur[i], evbuffer, prob);
			  }
		  }
	  }

  }

  end_time=clock();

  //--------------------------------------------
  printf("-----The total time costed=%f, the compute_time=%f\n", (double)(end_time-start_time),(double)(end_time-compute_time));
  printf("-----The total time costed=%f\n", (double)(end_time-start_time));
} 


/* 
 * build_tree() do the following things
 * (1) Pad the data to the nearest pow of 2
 * (2) Build the shift wavelet tree. To save the space, "swtree" is reused for each sky bucket
 */

void build_tree(int x, int y)
{
	int i,j,k,s;
	int *swt;
	int *swt_last;
	int *swt_i;
	int *swt_last_i;
	//Zero the swtree	
	int* swt_id=&swt_index[0];
	int* swt_data=&swtree[0];
	for(i=0;i<6200;i++){
		*(swt_id+i)=-1;
		*(swt_data+i)=0;
	}
	
	for(i=0;i<23;i++){
		last_non_empty[i]=0;
	}
	int time_length=(int)((current_end_time-current_start_time)*100);
	float size2log=log(time_length)/log(2);
	if(size2log==(int)size2log){
	   size=time_length;
	}else{
	   size=2<<(int)size2log;//(int)pow(2,(int)size2log+1);
	}
    int div2=size>>6;
	i=1;
	while(div2>0){
	   div2=div2>>1;
	   i++;
	}
	int nlayer=i;
	double last_end=sigmap_index[x][y][0];
	swt_index[0]=sigmap_index[x][y][0];
	k=0;
	int index=0;
	int level_index_even=0;
	int level_index_odd=0;
	for(i=sigmap_index[x][y][0];i<sigmap_index[x][y][1];i++){
		int temp=(int)((evbuffer[databuffer[i]].time-evbuffer[databuffer[sigmap_index[x][y][0]]].time)*100);
		index=temp>>5;
		swtree[swt_ladder[0]+index]++;	
		if(index>last_non_empty[0]){
			last_non_empty[0]=index;
		}
		if(swt_index[swt_ladder[0]+index]==-1){
			swt_index[swt_ladder[0]+index]=i;
		}
		for(j=1;j<nlayer;j++){
			level_index_even=index>>1;
			swtree[swt_ladder[2*j]+level_index_even]++;
			if(swt_index[swt_ladder[2*j]+level_index_even]==-1){
				swt_index[swt_ladder[2*j]+level_index_even]=i;
			}
			if(level_index_even>last_non_empty[2*j]){
			    last_non_empty[2*j]=level_index_even;
			}
			//odd level
			if(index>0)
			{
				level_index_odd=(index-1)>>1;
			}		
			swtree[swt_ladder[2*j-1]+level_index_odd]++;	
			if(swt_index[swt_ladder[2*j-1]+level_index_odd]==-1){
				swt_index[swt_ladder[2*j-1]+level_index_odd]=i;
			}
			if(level_index_odd>last_non_empty[2*j-1]){
			    last_non_empty[2*j-1]=level_index_odd;
			}
			index=index>>1;
		}
	}	
}


//*********************************************************************
//Judge if this is a burst
void burst_judge_bin(double start, int length, int i_signal){
  //return;
	if(i_signal<SEARCH_THRESH) {return;}
	//Compute the 10s back events
	float dur_local=duration[current_dur];
	int nsignal=0;
	int ii,jj;
	float xallsky;
	int nallsky_10s, ixallsky;
    int t_start=(int)((start-tstart)*100);
	//return;
	if(duration[current_dur]<10){
		nallsky_10s=find_back_base_least_detail(t_start);
		if(nallsky_10s==-1) return;
	}else{
		nallsky_10s=(int)(( (double)get_overall_events(t_start, length))*10./dur_local);
	}
	if(xallsky_lookup[nallsky_10s][current_dur]==0){
		xallsky = log((((float) nallsky_10s)*dur_local/10.))*BINSF_span;       //5s
		//return 0; //19.9s
		ixallsky = (int) (xallsky+.5);
		xallsky_lookup[nallsky_10s][current_dur]=ixallsky;
	}
    int i_back  = maps->backmap_index[current_i][current_j]+xallsky_lookup[nallsky_10s][current_dur]; 
	//---------------------------
    if (i_back>=0 && i_back<PROB_BINS && i_signal>=0 && i_signal<MAX_SIG) {
	   prob->count[current_dur][i_signal][i_back]++;
	} else { 
	   sprintf(str,"Overflow: %d %d %d %d %f %f %f %f\n",i_signal,i_back,nallsky_10s,
			   ixallsky,xallsky,dur_local,PROB_BINSF,prob->span);
			   grb_finder_logger(str);
	}
    /* search more finely for low prob */
    if (prob->value[i_signal][i_back]<1.e-4) 
    {
	  ndlarger1p4++;
	  for(ii=-1;ii<=1;ii++){
	     for(jj=-1;jj<=1;jj++){
			if(ii==0&&jj==0){
			  nsignal=i_signal;
			}else{
			  int last_start=sigmap_index[current_i+ii][current_j+jj][0];
			  if(start<evbuffer[databuffer[sigmap_index[current_i+ii][current_j+jj][1]-1]].time)
			  {
   			     if(start+length*0.01<evbuffer[databuffer[sigmap_index[current_i+ii][current_j+jj][1]-1]].time)
			     {
				     nsignal=sum_bin(start, start+length*0.01, &last_start);
			     }else{
				     nsignal=sum_bin(start, evbuffer[databuffer[sigmap_index[current_i+ii][current_j+jj][1]-1]].time, &last_start);
			     }
			  }else{
				nsignal=0;
			  }
 				  
			}
	        if (prob->value[nsignal][i_back]<prob->publish_threshold[current_dur]) {
				  cand.iha         = current_i;
				  cand.idec        = current_j;
				  cand.t0          = t_start;
				  cand.duration    = dur_local;
				  cand.runum       = runum;
				  cand.subrunum    = subrunum;
	 			  cand.signal      = nsignal;
				  cand.back        = exp(((float)i_back)*prob->span/PROB_BINSF+prob->minback);
				  cand.grbprob     = prob->value[nsignal][i_back];
				  cand.nallsky_10s = nallsky_10s;
				  cand.iback       = i_back;
				  cand.julDate     = julDate;
				  cand.nTail       = nTail;
				  cand.nBuff       = nBuff;
				  cand.idur        = current_dur;
				  ipublish=0;
				  //Take down the burst with the smallest probability
                  if(cand_dur[current_dur].grbprob>prob->value[nsignal][i_back]){
					  cand_dur[current_dur].iha         = current_i;
					  cand_dur[current_dur].idec        = current_j;
					  cand_dur[current_dur].t0          = t_start;
					  cand_dur[current_dur].duration    = dur_local;
					  cand_dur[current_dur].runum       = runum;
					  cand_dur[current_dur].subrunum    = subrunum;
	 				  cand_dur[current_dur].signal      = nsignal;
					  cand_dur[current_dur].back        = exp(((float)i_back)*prob->span/PROB_BINSF+prob->minback);
					  cand_dur[current_dur].grbprob     = prob->value[nsignal][i_back];
					  cand_dur[current_dur].nallsky_10s = nallsky_10s;
					  cand_dur[current_dur].iback       = i_back;
					  cand_dur[current_dur].julDate     = julDate;
					  cand_dur[current_dur].nTail       = nTail;
					  cand_dur[current_dur].nBuff       = nBuff;
					  cand_dur[current_dur].idur        = current_dur;
				  }				  
				  if (ii!=0 || jj!=0)  /* don't look twice at (0,0) */
				  {
					prob->count[current_dur][nsignal][i_back]++;
				  }
				  nburst++;
				  printf("start=%f, length=%d\n", start, length);
				  printf("bin i=%d, j=%d, ii=%d, jj=%d, dur=%d, at interval[%d,%d] a burst of %d, back=%d\n", current_i+ii, current_j+jj, ii, jj, current_dur, t_start, t_start+length, nsignal, nallsky_10s);
			   }
		 }
	  }
	}			

}


/* Burst detection for duration>0.2 */
/*Take cell (i,j) as a time series and run the burst detection algorithm as follows*/
void detect_burst_bin(){

	int dur_int=duration_int[current_dur];
	double dur_local=duration[current_dur];
	int i,j;
    double k;
	double jump_step=dur_local/10;
	int pow2_int=(int)(log(dur_int)/log(2));  
	//we will interested in level (pow2+1)*2 and (pow2+1)*2-1
	int level_even=(pow2_int+2-5)*2;
	int level_odd=level_even-1;
	int* swt_even;
	int* swt_odd;
	int sum_duration;
    int last_start, last_end, last_value=0;
	swt_even=&swtree[swt_ladder[level_even]+0];
	swt_odd=&swtree[swt_ladder[level_odd]+0];
	//int pow2=pow(2,pow2_int+2);
    int pow2=2<<(pow2_int+1);
	int pow2_sub=pow2/2;
	double step_large=pow2*0.01;      //step = 2^(i+1)
	double step_small=pow2_sub*0.01;
    int* swt_even_sub=&swtree[swt_ladder[level_even-2]+0];
	double stop_time;
	//Search the even level first
	for(i=0;i<=last_non_empty[level_even];i++)
    {
		//2^(i+1) filter
		ntotal++;		
		if(*(swt_even+i)>2)
		{
			nlarger2++;
			if(burst_judge_bin_coarse(i, pow2, *(swt_even+i), 1))
			{
			  for(j=0;j<2;j++)
			  { //2^i filter
	  		    if(burst_judge_bin_coarse_sub(i, j, pow2_sub, *(swt_even_sub+2*i+j)))
                {   
					k=current_start_time+i*step_large+j*step_small;
					last_start=swt_index[swt_ladder[level_even]+i];
					last_end=last_start;
					last_value=0;
					stop_time=current_start_time+i*step_large+(j+1)*step_small-dur_local;
					ndetail++;
					//detailed search
					while(k<=stop_time){
					
				  		ndetail++;
						if(k>=current_end_time)
						{
							break;
						}					 
                        sum_duration=sum(k,k+dur_local,&last_start, &last_end,last_value);
						burst_judge_bin(k, dur_int,sum_duration);
						last_value=sum_duration;
						k+=jump_step;					  
					}
			    }else{
			        if(j==0){	                        
                        k=current_start_time+i*step_large+(j+1)*step_small-dur_local;
						last_start=swt_index[swt_ladder[level_even]+i];
						last_end=last_start;
						last_value=0;
						stop_time=current_start_time+i*step_large+(j+1)*step_small;
						ndetail++;
						while(k<=stop_time){
						//detailed search
							ndetail++;
							if(k>=current_end_time)
							{
								break;
							}						
							sum_duration=sum(k,k+dur_local,&last_start, &last_end,last_value);
							burst_judge_bin(k, dur_int, sum_duration);
							last_value=sum_duration;
							k+=jump_step;
						
						}
					}
				}
			  }
			}
		}
	}
	//Search the odd_level then
	for(i=0;i<=last_non_empty[level_odd];i++){
		//2^(i+1) filter	
		ntotal++;	
		if(*(swt_odd+i)>2){
			nlarger2++;
			if(burst_judge_bin_coarse(i, pow2, *(swt_odd+i), 0)){
				//detailed search
				k=current_start_time+(i+1)*step_large-dur_local;
				last_value=0;
				last_start=swt_index[swt_ladder[level_odd]+i];
				last_end=last_start;
				stop_time=current_start_time+(i+1)*step_large;
				ndetail++;
				while(k<=stop_time){
				  	ndetail++;
					if(k>=current_end_time)
					{
						break;
					}				  
					sum_duration=sum(k,k+dur_local,&last_start,&last_end, last_value);		
					burst_judge_bin(k, dur_int,sum_duration);
                    last_value=sum_duration;		
				    k+=jump_step;				  
				}
			}
		}
	}

}
//Judge if this is a burst in the 2^i search
int burst_judge_bin_coarse_sub(int start_i, int start_j, int length, int i_signal){
  //return 0;
	if(i_signal<SEARCH_THRESH) {return 0;}
	int iha=current_i;
	int icount=current_j;
	//Compute the 10s back events
	int nsignal=0;
    float xallsky;
	float dur_local=duration[current_dur];
	int nallsky_10s, ixallsky;
	float start_time;
    start_time=current_start_time+(start_i*length*2+start_j*length)*0.01;
	int t_start=(int)((start_time-tstart)/TIME_RESOLUTION);
	  //return 0;
	if(duration[current_dur]<10){
		//if(booltest()) {printf("come into judge\n");}
		nallsky_10s=find_back_base_least_coarse(t_start, length);
		//if(booltest()) {printf("nallsky_10s=%d\n", nallsky_10s);}
		if(nallsky_10s==-1) return 0;
	}else{
		nallsky_10s=(int)(((double)get_overall_events(t_start, length))*10./dur_local);
	}
	  //return 0;
	if(xallsky_lookup[nallsky_10s][current_dur]==0){
		xallsky = log((((float) nallsky_10s)*dur_local/10.))*BINSF_span;       //5s
		//return 0; //19.9s
		ixallsky = (int) (xallsky+.5);
		xallsky_lookup[nallsky_10s][current_dur]=ixallsky;
	}
	
	//return 0;  //21.1s
    int i_back  = maps->backmap_index[current_i][current_j]+xallsky_lookup[nallsky_10s][current_dur]; 
	//---------------------------
    /* search more finely for low prob */
    return (prob->value[i_signal][i_back]<prob->publish_threshold[current_dur]);
	       
}
//Judge if this is a burst in the 2^(i+i) search
int burst_judge_bin_coarse(int start, int length, int i_signal, int is_even){
  //return 0;  //9s
	if(i_signal<SEARCH_THRESH) {return 0;}
	int iha=current_i;
	int icount=current_j;
	//Compute the 10s back events
	int nsignal=0;
    float xallsky;
	float dur_local=duration[current_dur];
	int nallsky_10s, ixallsky;
	float start_time;
	float f_length=length*0.01;
	start_time=current_start_time+start*f_length;
	if(!is_even){
       start_time+=f_length/2;
	}
	int t_start=(int)((start_time-tstart)/TIME_RESOLUTION);
	//return 0   // 11s

	if(duration[current_dur]<10){
		//if(booltest()) {printf("come into judge\n");}
		nallsky_10s=find_back_base_least_coarse(t_start, length); //3s
		//if(booltest()) {printf("nallsky_10s=%d\n", nallsky_10s);}
		if(nallsky_10s==-1) return 0;
	}else{
		nallsky_10s=(int)(((double)get_overall_events(t_start, length))*10./dur_local);
	}
	//return 0  //15s
	
	if(xallsky_lookup[nallsky_10s][current_dur]==0){
		xallsky = log((((float) nallsky_10s)*dur_local/10.))*BINSF_span;       //5s
		//return 0; //19.9s
		ixallsky = (int) (xallsky+.5);
		xallsky_lookup[nallsky_10s][current_dur]=ixallsky;
	}
	
	//return 0;  //21.1s
    int i_back  = maps->backmap_index[current_i][current_j]+xallsky_lookup[nallsky_10s][current_dur];  
	//return 0;  //21.
	//---------------------------
    /* search more finely for low prob */
    int result=(prob->value[i_signal][i_back]<prob->publish_threshold[current_dur]);
    return result;
}

/********Utilities*********/
//Sum the subsequence;
int sum(double start, double end, int* last_start, int* last_end, int last_value){
	if(end>current_end_time){
		end=current_end_time;
	}
	int summation=last_value;
	int i=*last_start;
	int *data_ptr=&databuffer[0];
    while(evbuffer[*(data_ptr+i)].time<start){
       summation--;
       i++;
    }
    *last_start=i;
    i=*last_end;        
    while(evbuffer[*(data_ptr+i)].time<end){
       summation++;
       i++;
    }
    *last_end=i;
    return summation;
}

int sum_bin(double start, double end, int* last_start){
	int summation=0;
	int i=*last_start;
	int *data_ptr=&databuffer[0];
	while(evbuffer[*(data_ptr+i)].time<start){
       i++;
    }
    while(evbuffer[*(data_ptr+i)].time<end){
       summation++;
       i++;
    }
	*last_start=i;
    return summation;
}

//Find the index that duration[index]>=10
void find_duration_cutoff(){	
	int i=0;
	while(duration[i]<0.2){
		i++;
	}
	duration_cutoff_index2=i;
	while(duration[i]<10){
		i++;
	}
	duration_cutoff_index10=i;
}

//In the case of coarse search, that is 2^i and 2^(i+1)
//Find the back noise in a duration 
//Find the central point and get the value within a interval of (-5,5)
int find_back_base_least_coarse(int t_start_event, int length){
	if(t_start_event>=TIME_LENGTH) {return -1;}
	int dur_local=duration_int[current_dur];
	int i,j,k;
	int nallsky_10s=0;
	int smallest=1000000;
	int large_jump=dur_local>>2;
	if(threshold_buffer_larger[current_dur][t_start_event]>0){
		smallest=threshold_buffer_larger[current_dur][t_start_event];
	}else{
		for(k=t_start_event+dur_local>>1;k<t_start_event+length-dur_local>>1;k+=large_jump){
			if(k>=0&&k<TIME_LENGTH){
				if(smallest>data_base_threshold[k]){
					smallest=data_base_threshold[k];
				}
			}else{
				if(k<0){
					if(smallest>data_base_threshold[0]){
						smallest=data_base_threshold[0];
					}
				}else{
					if(smallest>data_base_threshold[TIME_LENGTH-1]){
						smallest=data_base_threshold[TIME_LENGTH-1];
					}				
					break;
				}			
			}			
		}
		threshold_buffer_larger[current_dur][t_start_event]=smallest;
	}	
    nallsky_10s=smallest;
	return nallsky_10s;

}
//In the case of detailed search
//Find the back noise in a duration 
//Find the central point and get the value within a interval of (-5,5)
int find_back_base_least_detail(int t_start_event){
	if(t_start_event>=TIME_LENGTH) {return -1;}
	int dur_local=duration_int[current_dur];
	int nallsky_10s=0;
	int n_threshold=0;
  	int k=t_start_event+dur_local>>1;
	if(k>=0&&k<TIME_LENGTH){
		nallsky_10s=data_base_threshold[k];
		return nallsky_10s;
	}
	if(k<0){
		nallsky_10s=data_base_threshold[0];
		return nallsky_10s;
	}
	if(k>=TIME_LENGTH){	
		nallsky_10s=data_base_threshold[TIME_LENGTH-1];
		return nallsky_10s;
	}	
}
/*********The following is to compute the background threshold *************/

//Sum the event in time interval;
int sum_time(double t_start, double t_end, int start_index, int* last_start, int* last_end, int last_value){
	int summation=0;
	int i;
	double* event_ptr;
	event_ptr=&time_event_table[0];
	if(t_start==tstart){
		i=1;
		while(*(event_ptr+i)<t_end){
			summation++;
			i++;
		}
		*last_start=0;
		*last_end=i;
	}else if(t_end==tend){
		i=nBuff-2;
		while(*(event_ptr+i)>t_start){
			summation++;
			i--;
		}
	}else{
		//printf("last_start=%d,last_end=%d\n", last_start,last_end);
		i=*last_start;
		while(*(event_ptr+i)<t_start){
			last_value--;
			i++;
		}
		*last_start=i;
		i=*last_end;
		while(*(event_ptr+i)<t_end){
			last_value++;			
			i++;
		}
		*last_end=i;
		summation=last_value;
	}
	/*
	if(start_index==(start_index/100000)*100000){
		printf("j=%d, time_event_table[j]=%f, t_start=%f, t_end=%f, summation=%d\n", start_index, time_event_table[start_index], t_start, t_end, summation);
	}
	*/
	return summation;
}

//Compute the background thresholdbase when duration<10s 
void compute_10s_base_threshold(){
	int j, step;
	double t0, t0_10s, t1_10s;
	int last_value=0;
	int last_start=0;
	int last_end=0;
	//if(booltest()) {printf("tstart=%f\n", tstart);}
	for(j=0;j<TIME_LENGTH;j+=1){
		t0=j*TIME_RESOLUTION+tstart;
		t0_10s = t0-5.;
		t1_10s = t0_10s+10.;
		if (t0_10s<tstart) {t0_10s=tstart;   t1_10s=tstart+10.;}
		if (t1_10s>tend)   {t0_10s=tend-10.; t1_10s=tend; }				
		last_value=sum_time(t0_10s, t1_10s, j, &last_start, &last_end, last_value);
		data_base_threshold[j]=last_value;
	}/*
    for(j=-5;j<=5;j++){
		printf("data_base_threshold[%d]=%d\n", 368+j, data_base_threshold[368+j]);
	}*/
}

//compute the background for durations >=10s
int get_overall_events(int t_start, int length){
	int i;
	int nevents=0;
	if(length==duration_int[current_dur]){ //detailed search
		if(overall_threshold[current_dur-duration_cutoff_index10][t_start]>0){
			nevents=overall_threshold[current_dur-duration_cutoff_index10][t_start];
			//printf("t_start=%d, length=%d, nevents=%d\n", t_start, length, nevents);
		}else{
			if(t_start+length>=TIME_LENGTH){
				for(i=t_start;i<TIME_LENGTH;i++){
					nevents+=time_index_table[i];
				}
			}else{
				for(i=0;i<length;i++){
					nevents+=time_index_table[t_start+i];
				}
			}
			overall_threshold[current_dur-duration_cutoff_index10][t_start]=nevents;
			//printf("t_start=%d, length=%d, nevents=%d\n", t_start, length, nevents);
		}
	}else{      //pow2 search
		if(overall_threshold_pow2[current_dur-duration_cutoff_index10][t_start]>0){
			nevents=overall_threshold_pow2[current_dur-duration_cutoff_index10][t_start];
			//printf("t_start=%d, length=%d, nevents=%d\n", t_start, length, nevents);
		}else{
			if(t_start+length>=TIME_LENGTH){
				for(i=t_start;i<TIME_LENGTH;i++){
					nevents+=time_index_table[i];
				}
			}else{
				for(i=0;i<length;i++){
					nevents+=time_index_table[t_start+i];
				}
			}
			overall_threshold_pow2[current_dur-duration_cutoff_index10][t_start]=nevents;
			//printf("t_start=%d, length=%d, nevents=%d\n", t_start, length, nevents);
		}
	}
	return nevents;
}

// For debug purpose--------------

int booltest(){

	if(current_i==115&&current_j==556&&current_dur==9){
		return 1;
	}
	return 0;
}
void valueTest(){
	int i,j;
	double sig, back;
	printf("back---------------------");
	//for (i=0;i<MAX_SIG;i++) {
           for (j=0;j<PROB_BINS;j++) {
	     //sig = (double) i;
	     back = exp(((double)j)*prob->span/PROB_BINSF+prob->minback);
	     //printf("  nsignal=%f, iback=%f, value=%f  ", sig, back, prob->value[i][j]);
	     //printf("%e\n",prob->value[i][j]);
	     //    printf("%e\n", back);
    }
	printf("back---------------------");	
}
//-----------------------
/**********/


