#ifndef SATNODE_H
#define SATNODE_H

#include <vector>
#include <string>

using namespace std;

// a state node to represent a SAT

class SATNode 
{
public:
	SATNode();
	SATNode(const SATNode& node);
	virtual ~SATNode();

	SATNode& operator=(const SATNode& node);

	void getLayer(int& h, int& s, int l=-1) const;
	void getLayer(int& h, int& s, vector<int>& hstruct, int l=-1) const;
	void addLayer(int h, int s, vector<int>& hstruct);
	void removeTopLayer();
	int  numLayer() const;
	double insCost();			// use this cost to insertToFrontier
	double cost() const { return _cost; }
	void setCost(double cost) { _cost = cost; }

	double density() const;

	void generateNextNodes(vector<SATNode*>& candNodes) const;

	const string& idstring();
	void output(ostream& ofs);
	string tostring();
	void clear();

	vector<double>	_alarmProb;

protected:
	vector<pair<int, int> >	 _hs;		// a vector of window sizes and shifts
	vector<vector<int> > _hstruct;		// a vector of the children's window sizes
	double		_cost;					// cost for this SAT state
	string		_idstring;				// string representation of this SAT
};


inline
void SATNode::getLayer(int& h, int& s, int l) const
{
	if (-1 == l)
		l = _hs.size()-1;
	h = _hs[l].first;
	s = _hs[l].second;
}

inline
void SATNode::getLayer(int& h, int& s, vector<int>& hstruct, int l) const
{
	if (-1 == l)
		l = _hs.size()-1;
	h = _hs[l].first;
	s = _hs[l].second;
	hstruct = _hstruct[l];
}

inline
void SATNode::addLayer(int h, int s, vector<int>& hstruct)
{
	_hs.push_back( pair<int,int>(h, s) );
	_hstruct.push_back( hstruct );
	_alarmProb.push_back( 0.0 );
	_idstring = "";
}

inline
void SATNode::removeTopLayer()
{
	_hs.erase( _hs.end()-1 );
	_hstruct.erase( _hstruct.end()-1 );
	_alarmProb.erase( _alarmProb.end()-1 );
	_idstring = "";
}

inline
int SATNode::numLayer() const
{
	return _hs.size();
}

inline
double SATNode::insCost()
{
	return _cost;
}


inline
void SATNode::clear()
{
	_hs.clear();
	_hstruct.clear();
	_alarmProb.clear();
	_cost = -1.0;
	_idstring = "";
}

inline
const string& SATNode::idstring()
{
	if (_idstring.size() > 0)
		return _idstring;

	char buf[128];
	for (int i = _hs.size()-1; i >= 0; -- i)
	{
		sprintf(buf, "%d/%d,", _hs[i].first, _hs[i].second);
		_idstring += buf;
	}
	//sprintf(buf, "%d,", _hs.size());
	//_idstring += buf;
	return _idstring;
}

#endif
