#include "algebra.h"

///////////////////////////////////////////////
//
//	Global variables
//
///////////////////////////////////////////////
bool QUERY_WITH_DEGREE=false;		//flag to control if querying with degree information
bool OUTPUT_DATA_COUNT=false;		//flag to control if showing the current matched record with each operation
bool QUIET_MODE=true;				//flag to control if showing the debug information
bool DEBUG_OPFILE=false;				//flag to control if showing the debug information of OPFILE

int DEFAULT_LP=4;						//declaration of the Default LP

int NUM_INTS=256;						//The Number of Intergers of a FingerPrint. Default is 256. Used in the Frowns Filtering process.
int INT_SIZE=sizeof(int)*8;				//The size of an int (in bits). Default is 32. Used in Frowns Filtering process.
int NUM_FP=400;						//The group size of fingerprints. Default is 4000. Used in Frowns Filtering process.

const int NodeTypeBitLen=sizeof(NodeType)*8; //The size of an NodeType (in bits). The information is needed when storing data into BekerlyDB

#include "graphbuild.h"

////////////////////////////////////////////
//
// Class: GraphBuild
// Usage: Responsible for founding all the patterns and fingerprint of a graph and insert into DB
//
/////////////////////////////////////////////
class GraphBuild{
private:
	Db* _db;				//Pointer to the data DB
	Db* _db_fp;				//Pointer to the FP DB
	ifstream* fin;			//Input file stream pointer
	int graphCount;			//Counter that records the total graph processed
	unsigned int *tmp_fp;	//Pointer temp u_int array which stored the set of FPs
	GraphADT* readNextGraph();	//function reads graph from input dbfile and returns a graph pointer of the current graph

public:
	~GraphBuild(){			//Destructor
		delete _db;
		if(fin)
			delete fin;				
	}
	GraphBuild(char* dbfile);		//Constructor
	void run();						//Starting GraphBuild
	void start_DB_Action(Map_PathTableWithDegree&);	//Starting storing data into BerkeleyDB(with degree information)
	void start_DB_Action(Map_PathTable&);				//Starting storing data into BerkeleyDBwithout degree information

	void add2FP_DB(unsigned int* fp,int graphIndex);	       //Inserting FP group data into db
	void add_FP(unsigned int*,int );						//Adding a FP to the fp group 

};

////////////////////////////////////////////
//
//	Implementation of GraphBuild Class
//
/////////////////////////////////////////////
GraphBuild::GraphBuild(char* dbfile):graphCount(0){	//Construtor 
	fin=NULL;
	if(dbfile!=""){
		ifstream in(dbfile);
		if(!in){
			cout<<dbfile <<" doest not exist\n";
			exit(0);
		}
		
		fin=new ifstream(dbfile);	
	}
}

void GraphBuild::run(){
	//open FP database
	char FP_DBFile[]="fingerprint.db";
	(void)remove(FP_DBFile);
	_db_fp=new Db(0,0);
	
	_db_fp->set_errpfx("GraphBuild: ");
	_db_fp->set_pagesize(1024);		// Page size: 1K. 

	_db_fp->set_cachesize(0, 32 * NUM_INTS*NUM_FP, 0);
	_db_fp->open(NULL, FP_DBFile, NULL, DB_BTREE, DB_CREATE, 0664);


	//open Main DataBase
	char DBFile[]="graphdata.db";
	(void)remove(DBFile);
	_db=new Db(0,0);
	
	_db->set_error_stream(&cerr);
	_db->set_errpfx("GraphBuild: ");
	_db->set_pagesize(1024);		// Page size: 1K. 

	_db->set_cachesize(0, 32 * 1024, 0);
	_db->open(NULL, DBFile, NULL, DB_BTREE, DB_CREATE, 0664);

	int count=0;

	char f_index[]="index.dat";
	FILE *fp_index=fopen(f_index,"w");
	GraphADT* g;
	while((g=readNextGraph())!=NULL){	//Read each graph from the input file
		//Process the given graph
		vDFS_GraphBuild<GraphADT> myDFS(*g,DEFAULT_LP);
		fprintf(fp_index,"gn%d: %s\n",count,g->getGraphName().c_str());

		setclock("#DFS time",1);

		//Get the data map of the graph
		if(QUERY_WITH_DEGREE){
			Map_PathTableWithDegree *myPathMap=myDFS.getMapDegreePathTable();
			setclock("Inerting time into DB",0);	
			start_DB_Action(*myPathMap);		
			setclock("Inerting time into DB",1);	

		}else{
			Map_PathTable *myPathMap=myDFS.getMapPathTable();
			setclock("Inerting time into DB",0);	
			start_DB_Action(*myPathMap);		
			setclock("Inerting time into DB",1);	
		}

		//Get the fingerprint of the graph
		unsigned int* fp=myDFS.getFingerPrint();
		
		int itmp=0;
		for(int i=0;i<NUM_INTS;i++){
			if(fp[i]!=0){
				itmp++;
			}
		}

		if(!QUIET_MODE)
			printf("\nNumber of total pathes %d",itmp);

		//Add the fingerprint into current FP group		
		add_FP(fp,count);
		
		delete g;
		count++;
	}

	fclose(fp_index);
	cout<<"Number of graphs in DB: "<<count<<endl;   //output the total graphs processed
	add2FP_DB(tmp_fp,count);	//insert the current FP group into DB

	fin->close();
	
	_db->close(0);
	_db_fp->close(0);
}

void GraphBuild::add_FP(unsigned int* fp,int graphIndex)
{
	//counting the current position in the group of fp
	int base=graphIndex/NUM_FP;
	int remainder=graphIndex % NUM_FP;

	if( remainder != 0){	//insert FP in the right slot of the group
		unsigned int* pi=(tmp_fp+remainder*NUM_INTS);
		memcpy(pi,fp, NUM_INTS*sizeof(unsigned int));
	}else if(graphIndex==0){	//start a new group, setting the memory and copy the first FP
		tmp_fp=new unsigned int[NUM_FP*NUM_INTS];
		memset(tmp_fp,0,sizeof(unsigned int)*NUM_FP*NUM_INTS);
		memcpy(tmp_fp,fp, NUM_INTS*sizeof(unsigned int));
	}else{	//Already exceed the group size, insert into BerkeleyDB and start a new group
		add2FP_DB(tmp_fp,graphIndex);
		tmp_fp=new unsigned int[NUM_FP*NUM_INTS];

		//The new begin 
		memcpy(tmp_fp,fp, NUM_INTS*sizeof(unsigned int));
	}
}

//Insert 1 FP group into BerkeleyDB
void GraphBuild::add2FP_DB(unsigned int* fp,int graphIndex)
{
	int base= graphIndex/NUM_FP;
	int remainder= graphIndex % NUM_FP;

	char mykey[8];
	//itoa(graphIndex,mykey,10);
	sprintf(mykey,"%d",graphIndex);
	
	Dbt key(mykey,strlen(mykey)+1);
	
	int graphs=(remainder==0 && base!=0)?NUM_FP:(remainder+1);
	Dbt data(fp,sizeof(unsigned int)*NUM_INTS*graphs );
	
	int ret = _db_fp->put(0, &key, &data, DB_NOOVERWRITE);
	
	if (ret == DB_KEYEXIST) {
		cout << "Key " << mykey << " already exists.\n";
	}	

	delete tmp_fp;
	
	//count++;
	//cout<<"total table count"<<count<<endl;
}


//Insert the patterns of a graph into BerkeleyDB(with Degree information)
void GraphBuild::start_DB_Action(Map_PathTableWithDegree& myMapPath)
{
	Map_PathTableWithDegree::iterator it=myMapPath.begin(),
					  it_end=myMapPath.end();

	int count=0;
	while(it!=it_end){
		char *mykey=new char[(*it).first.size()+1];
		strcpy(mykey,(*it).first.c_str());

		Dbt key(mykey, strlen(mykey)+1);

		NodeType* int_arr=(*it).second->outputData();
		int size=(*it).second->getOutputdatasize();
		
		Dbt data(int_arr,size*sizeof(NodeType));

		int ret = _db->put(0, &key, &data, DB_NOOVERWRITE);

		if (ret == DB_KEYEXIST) {
			cout << "Key " << mykey << " already exists.\n";
		}	

		delete mykey;
		delete int_arr;

		++it;
		count++;
	}

	//cout<<"total table count"<<count<<endl;
}

//Insert the patterns of a graph into BerkeleyDB(without Degree information)
void GraphBuild::start_DB_Action(Map_PathTable& myMapPath)
{
	Map_PathTable::iterator it=myMapPath.begin(),
					  it_end=myMapPath.end();

	int count=0;
	while(it!=it_end){
		char *mykey=new char[(*it).first.size()+1];
		strcpy(mykey,(*it).first.c_str());

		Dbt key(mykey, strlen(mykey)+1);

		int rows=(*it).second->get_rows();
		int cols=(*it).second->get_cols();

		// The data in the array are:
		//----------------------
		//starting 4 positions    -- Global Header
		// pos[0]: 		Section count. Each section is a different degree.
		// pos[1]:    	Length-Path of this table
		// pos[2-3]: 	Total rows count in this table
		//----------------------
		//following each section  -- Section Header
		// pos[0-1]:		rows
		// pos[2-(2+LP)]:	degree array
		// pos[(3+LP)- ]:	data array

		int sec_header=2+cols;
		//start header size   = 1(section_number) + 1(cols)+ 2(rows)
		//section header size = sec_header* _section_count
		//total data size	  = _record_count*_LP

		int size=4+ + sec_header*1 + rows*cols;
		NodeType* int_arr=new NodeType[size];

		//global header
		int_arr[0]= 1;
		int_arr[1]= cols;
		int_arr[2]= rows>>NodeTypeBitLen;
		int_arr[3]= rows<<NodeTypeBitLen>>NodeTypeBitLen;

		//section header
		int_arr[4]= int_arr[2];
		int_arr[5]= int_arr[3];

		//copying degree array
		for(int p=0;p<cols;p++){
			int_arr[6+p]= 0;
		}

		for(int i=0;i<rows;i++)
			for(int j=0;j<cols;j++)
				int_arr[6+cols+i*cols+j]=(NodeType)(*it).second->get_item(i,j);
			
			
		Dbt data(int_arr,size*sizeof(NodeType));

		int ret = _db->put(0, &key, &data, DB_NOOVERWRITE);

		if (ret == DB_KEYEXIST) {
			cout << "Key " << mykey << " already exists.\n";
		}	

		delete mykey;
		delete int_arr;

		++it;
		count++;
	}

	//cout<<"total table count"<<count<<endl;
}

//function reads graph from input dbfile and returns a graph pointer of the current graph
GraphADT* GraphBuild::readNextGraph(){
	ifstream& f=*fin;
	GraphADT* retG=NULL;
	char line[51];
	string name;
	
	while(!f.eof()){
		f.getline(line,50);
		if(line[0]=='#'){
			try{
				char* p=line+1;
				name=string(p);	//set name of this graph 				

				f.getline(line,50);
				int numNode=atoi(line);
				retG=new GraphADT(numNode,false);
				retG->setGraphName(name);

				//setting table name
				char tabname[MAX_TABLENAME];
				sprintf(tabname,"gn%d",graphCount++);
				retG->setName(tabname);
				
				for(int i=0;i<numNode;i++){	//read node
					f.getline(line,50);
					(*retG).addNode(i,line);
				}
				f.getline(line,50);
				int numEdge=atoi(line);
				for(int j=0;j<numEdge;j++){
					f.getline(line,50);
					string tmp(line);
					vector<string> nodes=sep_words(tmp);
					if(nodes.size()<2)
						throw ReadGraphExcp(graphCount);

					(*retG).insert(Edge(nodes[0],nodes[1]));
					
				}

				return retG;
			}catch(ReadGraphExcp ex){
				graphCount--;
				delete retG;

				cout<< "error reading graph:" <<ex.getID()<<endl;
			}
		}
	}

	return retG;
}

////////////////////////////////////////////
//
//	Main Function
//
/////////////////////////////////////////////
int main(int argc, char** argv)
{
	try {
		setclock("Total running time",0);			

		bool HasDBFile=false;
		GraphBuild* app=NULL;
		
		//parse the input parameters
		for (int n = 1; n <argc; n++) {
			if (strcmp(argv[n],"-f")==0) {
				app=new GraphBuild(argv[++n]);
				HasDBFile=true;
			}else if(strcmp(argv[n],"-degree")==0) {
				QUERY_WITH_DEGREE=true;
			}else if(strcmp(argv[n],"-v")==0) {
				QUIET_MODE=false;
			}else if(strcmp(argv[n],"-lp")==0) {
				DEFAULT_LP=atoi(argv[++n]);
			}else if(strcmp(argv[n],"-fp")==0) {
				NUM_INTS=atoi(argv[++n]);
			}
			else {		
				cout<< "Usage: GraphBuild -f graphDBfile [-degree -lp lengthpath -fp length of FP]\n"<<
					"\t-degree            query with degree\n"<<
					"\t-lp lengthpath     input different LP, default is 4\n"<<
					"\t-fp length of FP   input different length of FP, default is 256"<<endl;

				exit(2);
			}
		}  // end for(....)

		if(app!=NULL){	//starting GraphBuild
			app->run();
			setclock("Total running time",1);
			printtime("Inerting time into DB");	
			printtime("Total running time");	
			delete app;

			//char ch;
			//cin>> ch;
		}else{
				cout<< "Usage: GraphBuild -f graphDBfile [-degree -lp lengthpath -fp length of FP]\n"<<
					"\t-degree            query with degree\n"<<
					"\t-lp lengthpath     input different LP, default is 4\n"<<
					"\t-fp length of FP   input different length of FP, default is 256"<<endl;
		}

		return (EXIT_SUCCESS);
	}
	catch (DbException &dbe) {
		cerr << "GraphBuild: " << dbe.what() << "\n";
		return (EXIT_FAILURE);
	}
}
