/*
  GraphBlast Algorithm
  opfile.cpp responsible for graph matching
*/

#include "common.h"
#include "algebra.h"

extern bool GRAPHGREP_VF;
extern float tot_s;

typedef deque<PathTable*> vPathTable;

////////////////////////////////////////
//
//	Class: OPFILE.  Use the Defination in algebra.h
//	Usage: Responsible for the table management and execution of operations. 
//		   It also maintains a table pool that contains the tables which was loaded from BerkeleyDB or 
//		   from the operation result.
//
///////////////////////////////////////

//Constructor
//Parameter 1: opfile name
//			2: query file name
//			3: A pointer to the database 
OPFile::OPFile(char* opfilefname,char* queryfname,Db* db){
  using std::ifstream;
  ifstream fIn(opfilefname);
  if(!fIn) throw GraphGrepExcp("No such file");
  
  //process each line of the opfile
  while(!fIn.eof()){
    char tmp[MAX_LINE];
    fIn.getline(tmp,MAX_LINE);
    //		cout << tmp;
    if(tmp[0]!='#' && strcmp(tmp,"")!=0) 
      opfile.push_back(string(tmp));
  }
  //setting data_member
  opcount=opfile.size();
  _db=db;
  fIn.close();
  
  char outname[100];
  if(GRAPHGREP_VF == false) {
    sprintf(outname,"%s.out",queryfname);
    
    fpout= fopen(outname,"a+");
  }
}

//Function that return table name without leading "^" char.
string OPFile::getTableName(const char* tabName){
  char tmp[MAX_TABLENAME],*pc,*degree,ret[MAX_TABLENAME];
  strcpy(tmp,tabName);
  
  if(tmp[0]=='^'){
    pc=tmp+1;
    degree=strchr(pc,'^');
    if(QUERY_WITH_DEGREE){	//if the name contains degree string
      sprintf(ret,"%s%s",strGraphID.c_str(),pc);
    }else{
      if(degree){	
	tmp[degree-pc+1]='\0';
	sprintf(ret,"%s%s",strGraphID.c_str(),pc);
      }else{
	sprintf(ret,"%s%s",strGraphID.c_str(),pc);
      }
    }
    return string(ret);
  }else{
    return string(tabName);
  }
  
  
}

//Add a table into the table pool
void OPFile::addTable(PathTable* tab){
	pool.push_front(tab);
};
 
//Release a table from table pool
void OPFile::releaseTable(const string& tablename){
  vPathTable::iterator pos;
  bool found=false;
  
  pos=pool.begin();
  while(pos!=pool.end()){
    if(strcmp( ((PathTable*)*pos)->get_Name(),tablename.c_str() )==0) {
      found=true;
      break;
    }
    pos++;
  }
  
  
  if(found){
    delete(*pos);
    pool.erase(pos,pos+1);
  }
  
}

//This funtion return a pointer to a table which either loaded from table pool or BerkeleyDB. 
PathTable* OPFile::loadTable(const string& tabname){
  vPathTable::iterator pos;
  bool found=false;
  clock_t beg,en;
  
  //looking for tables in the pool
  if(tabname[0]=='q'){
    pos=pool.begin();
    while(pos!=pool.end()){
      if(strcmp( ((PathTable*)*pos)->get_Name(),tabname.c_str() )==0) {
	if((*pos)->get_rows()==0)
	  throw GraphGrepExcp("empty table\n");
	return *pos;
      }
      pos++;
    }		
  }
  
  //not found in the pool, loading table from DB
  //	setclock("create from DB time",0);
  beg = clock();
  PathTable* tmp=new PathTable(tabname.c_str(),LoadTablefromDB(_db,tabname.c_str()));
  en = clock();
  tot_s += (float)(en-beg)/CLOCKS_PER_SEC;
  //setclock("create from DB time",1);
  
  pool.push_front(tmp);
  
  return tmp;		
}

//Parse the command string to int command
int OPFile::switchCmd2Code(const string& opstr){
  int ret=-1;
  char cmd[10];
  
  string::size_type pos1=0,prev_pos=0;
  pos1=opstr.find_first_of(' ',pos1);
  strcpy(cmd,(opstr.substr(prev_pos,pos1-prev_pos)).c_str());
  
  if(strcmp(cmd,"INTERSECT")==0)
    ret= INTERSECT;
  else if(strcmp(cmd,"TESTUNIQUE")==0)
    ret= TESTUNIQUE;
  else if(strcmp(cmd,"SPIT")==0)
    ret= SPIT;
  else if(strcmp(cmd,"CARTESIANPRODUCT")==0)
    ret= CARTESIANPRODUCT;
  else if(strcmp(cmd,"SELECT")==0)
    ret= SELECT;
  
  return ret;
}

//Reset the table pool. Erase all the tables
void OPFile::reset(){
  //using std::for_each;
  vPathTable::iterator it=pool.begin(),
    it_end=pool.end();
  
  //for_each(pool.begin(),pool.end(),del);
  while(it!=it_end){
    delete (*it);
    it++;
  }
  
  pool.erase(pool.begin(),pool.end());
}

//This function executes all the commands in this opfile.
void OPFile::run(){
  //for each operation
  for(int i=0;i<opcount;i++){
    //loading tables into pool
    //  setclock("load table times",0);	
    
    string strOP=opfile[i];
    Operation* op;
    switch(switchCmd2Code(strOP)){
    case INTERSECT:
      op=new OPInterSect(strOP,this);
      break;
    case TESTUNIQUE:
      op=new OPTestUnique(strOP,this);
      break;
    case SELECT:
      op=new OPSelect(strOP,this);
      break;
    case CARTESIANPRODUCT:
      op=new OPCartesianProduct(strOP,this);
      break;
    case SPIT:
      op=new OPSpit(strOP,this);
      break;
    default:
      //abort();
      break;
    }
    //setclock("load table times",0);	
    //loadTable(op->getNeedTableName());
    //setclock("load table times",1);	
    
    //execute operation
    if(!QUIET_MODE){
      cout<<i<<"\t";	
      cout <<strOP.c_str()<<"\n";
    }
    //}
    try{
      op->execute();
    }catch(GraphGrepExcp e){
      return;	
    }
    
    delete(op);
  }
}

//Utility function
NodeType* sep_DegreeString(const string& str,int lp){
  NodeType *tmp_words=new NodeType[lp];
  
  string::size_type begIdx,endIdx;
  
  begIdx= str.find_first_not_of(delims);
  int count=0;
  while(begIdx!=string::npos){
    endIdx=str.find_first_of(delims,begIdx);
    if(endIdx==string::npos)
      endIdx=str.length();
    
    //tmp_words.push_back( atoi( str.substr(begIdx,endIdx-begIdx).c_str()) );
    tmp_words[count]=atoi( str.substr(begIdx,endIdx-begIdx).c_str());
    begIdx=str.find_first_not_of(delims,endIdx);
    count++;
  }
  
  return tmp_words;
}

//Global Utility function
//Load a given table from BerkeleyDB into a NodeType array. 
//Param 1: database pointer
//      2: loading table name
NodeType* LoadTablefromDB(Db* db,const char* tablename){
  
  //Prepare for the database query
  Dbc *dbcp;	
  Dbt key1;
  Dbt data1;
  int ret;
  char c_tabname[MAX_TABLENAME];
  char c_tabdegree[MAX_TABLENAME];
  //strcpy(tmp,tablename);
  if(QUERY_WITH_DEGREE){
    int endp=strcspn(tablename,"^");
    strncpy(c_tabname,tablename,endp);
    strcpy(c_tabdegree,tablename+(endp+1));
    c_tabname[endp]='\0';
  }else{
    strcpy(c_tabname,tablename);
  }
  
  //cout<<c_tabname<<endl;
  //cout<<c_tabdegree<<endl;
  key1.set_data(c_tabname);
  key1.set_size(strlen(c_tabname)+1);
  
  //Query the database
  db->cursor(NULL, &dbcp, 0);
  if ((ret = dbcp->get(&key1, &data1, DB_SET)) == 0){		
    NodeType* data=(NodeType*)data1.get_data();
    
    //Loading global header	..see PathTableWithDegree::outputData() in algebra.cpp
    //!! If there's no degree information, the section count is always 1 
    //   and only 1 data section.
    int sec_count=data[0];
    int lp=data[1];
    int total_rows=(data[2]<<NodeTypeBitLen) + data[3];
    
    //Testing 
    //cout<< "sec_count:" <<sec_count<<" ";
    //cout<< "cols: "<< lp<<" ";
    //cout<< "total_rows: "<< total_rows<<endl;
    
    //allocate memory for return NodeArr
    NodeType *ret_arr=new NodeType[total_rows*lp+3];
    ret_arr[2]=lp;
    int ret_pos=3;
    
    //degree arr for this operation
    NodeType* degreeArr=sep_DegreeString(c_tabdegree,lp);
    
    int curpos=4;
    int ret_rows=0;
    //loading each section
    for(int i=0;i<sec_count;i++){			
      int sec_rows=(data[curpos]<<NodeTypeBitLen) + data[curpos+1];
      
      //testing 
      //cout<< "section: "<<i<<" ";
      //cout<< "section rows: " <<sec_rows<<" "<<endl;
      
      bool isLoadingSection=true;
      
      if(QUERY_WITH_DEGREE ){		//Compare only when the QUERY_WITH_DEGREE flag is set
	for(int m=0;m<lp;m++){
	  if( data[(curpos+2)+m] < degreeArr[m]){
	    isLoadingSection=false;
	    break;
	  }
	}
      }
      
      curpos += (2+lp);
      if(isLoadingSection){	//load the data
	for(int nrows=0;nrows<sec_rows;nrows++){
	  //cout<< nrows<<": ";
	  for(int ncols=0;ncols<lp;ncols++){
	    ret_arr[ret_pos++]=data[curpos+ ncols+ nrows*lp ];
	    
	    //testing
	    //cout << data[curpos+ ncols+ nrows*lp ]<<" ";
	  }
	  //cout<<endl;
	  ret_rows++;
	}
      }
      
      curpos += sec_rows*lp;
    }//end for
    
    delete degreeArr;
    
    ret_arr[0]= ret_rows>>NodeTypeBitLen;
    ret_arr[1]= ret_rows<<NodeTypeBitLen>>NodeTypeBitLen;
    
    return ret_arr;
  }else {
    throw GraphGrepExcp("table not found");
    //cout<<"table not found";
    
  }
  //	setclock("loading table time",1);
  //cout << "\nmin";
  
  return NULL;
  
}
