/*
  GraphBlast Algorithm
  graphblast.cpp responsible for graph matching
  2006(c) Modified by Diego Reforgiato Recupero
  2007(c) Modified by Dmitry Skripin
*/

#include "common.h"
#include "graphblast.h"
#include "algebra.h"

///////////////////////////////////////////////
//
//	Global variables
//
///////////////////////////////////////////////


/*

  FILTERING FROWNS + MATCHING VF [GF] : GFROWNS.
  FILTERING2 = false
  GRAPHGREP_VF = true
  ACC_VF = false

  FILTERING FROWNS + MATCHING VF [DG] : DAYGB.
  FILTERING2 = false
  GRAPHGREP_VF = true
  ACC_VF = true
  
  FILTERING GRAPHGREP + MATCHING VF [GB] : GRAPHBLAST.
  FILTERING2 = true
  GRAPHGREP_VF = true
  ACC_VF = true
  
  FILTERING GRAPHGREP + MATCHING GRAPHGREP [GG] : GRAPHGREP.
  FILTERING2 = true
  GRAPHGREP_VF = false
  ACC_VF = false

*/

Query *query;	

listwildconnection *wildhead=NULL;      //for inexact matching
bool FILTERING2 = false;                //FILTERING FROWNS 
bool GRAPHGREP_VF = true;               //MATCHING VF 
bool ACC_VF = false;                    //VF ACCELERATION
float tot_s = 0;
bool GLIDE = false;                     //flag for inexact matching
bool QUERY_WITH_DEGREE=false;		//flag to control if querying with degree information
bool OUTPUT_DATA_COUNT=false;		//flag to control if showing the current matched record with each operation
bool QUIET_MODE=true;			//flag to control if showing the debug information
bool DEBUG_OPFILE=false;		//flag to control if showing the debug information of OPFILE

int DEFAULT_LP=4;			//declaration of the Default LP

int NUM_INTS=256;			//The Number of Intergers of a FingerPrint. Default is 256. Used in the Frowns Filtering process.
int INT_SIZE=sizeof(int)*8;	        //The size of an int (in bits). Default is 32. Used in Frowns Filtering process.
int NUM_FP=400;				//The group size of fingerprints. Default is 4000. Used in Frowns Filtering process.

int METHOD=0;                           // 0 for gb; 1 for gg; 2 for gf; 3 for dg

char opfilename[100];			//op filename
char queryfilename[100];		//query filename
bool FILTERING=true;			//flag to control if using filtering mechinism
int nGraph=1;				//Number of graphs that you want to match in DB. This is only useful when FILTERING set false
char method[15]="GraphBlast";           //GraphBlast is default. GraphGrep and DayGB are the others.

const int NodeTypeBitLen=sizeof(NodeType)*8; //The size of an NodeType (in bits). The information is needed when storing data into BekerlyDB

                          //for graphgrep filtering
///////////////////////////////////////////////


////////////////////////////////////////////
//
// Class: Filtering
// Usage: Used for filtering graphs before matching. Frowns method
//
/////////////////////////////////////////////

class Filtering{
private:
  unsigned int* _fp;		//pointer to an FingerPrint
  bool checkFP(unsigned int* query_fp,unsigned int* graph_fp);	//function to check if 2 fps are matched
public:
  Filtering(unsigned int* fp):_fp(fp){};	//Constructor. Must pass a FP as parameter
  void run();				       	//Start filtering process
};

////////////////////////////////////////////
//
// Class: Filtering2
// Usage: Used for filtering graphs before matching. GraphGrep method
//
/////////////////////////////////////////////
class Filtering2{
  
public:
  void run();                                   //Start filtering process
  int Compute_Offset(int *dat,int i,int *cont); // It returns the starting offset where patterns are equal
};

// It returns the starting offset where patterns are equal
int Filtering2::Compute_Offset(int *dat,int i,int *cont){
  int offset = 0;
  int j;
  int glob;
  char d[DEFAULT_LP+2];
  do {
    // glob contains the number of collisions
    if(offset == 0)
      glob = dat[0];
    else
      glob--;
    for(j=0;j<dat[offset+1];j++) // stored pattern
      d[j]=dat[offset+2+j];
    d[j]='\0';
    if(strcmp(d,hash_query[i].c_str())!='\0') {
      if(glob==0) {
	return -1; // the query pattern is not present in any graphs of the database
      }
      offset = offset + 2 + dat[offset+1];
      offset = offset + dat[offset]*2;
    }
    else {
      cont[i] = dat[offset + 2 + dat[offset+1]]*2; // is the length of the array punt_h[i]
      return offset+3+dat[offset+1];
    }
  }
  while(glob);
}

// main method of Filtering2 class
void Filtering2::run() {
  int *punt_h[hash_query.size()];      // Pointers to the retrieved data
  int *punt;
  int offset[hash_query.size()];       // Offset to the data where patterns query = pattern h
  int graphs_start[hash_query.size()]; // Initial idx of the graph for each row
  time_t s1,s2,ret=0,ret1=0,k=0,f_count=0;
  float tot = 0;
  char d[10];
  int cont[hash_query.size()];
  FILE *trans=fopen(".//graphblastdata//translate.dat","w");
  
  Db db(0, 0);
  Dbc *dbcp;
  Dbt key;
  Dbt data;
  db.set_error_stream(&cerr);
  db.set_errpfx("HASH:");
  db.set_pagesize(1024);		/* Page size: 1K. */
  db.set_cachesize(0, 32 * 1024, 0);
  db.open(NULL, "dbHASH.db", NULL, DB_BTREE, DB_RDONLY, 0664);
  db.cursor(NULL,&dbcp,0);

  // computing hash table
  for(int i=0;i<hash_query.size();i++) {
    sprintf(d,"%d",vet_h[i]);
    key.set_data((char *)d);
    key.set_size(strlen(d)+1);
    s1 = clock();
    ret = dbcp->get(&key, &data, DB_SET);
    s2 = clock();
    tot += (float)((s2-s1)/CLOCKS_PER_SEC);
    if(ret==0) {
      punt = (int *)data.get_data();
      offset[i] = Compute_Offset(punt,i,cont);
      if(offset[i] == -1) { // All graphs filtered
	ret = 1;
	break;
      }
      punt_h[i] = (int *)malloc(sizeof(int)*cont[i]);
      for(k=offset[i];k<offset[i]+cont[i];k++) {
	punt_h[i][k-offset[i]] = punt[k];
      }
    }
    else {
      break;
    }
  }
  
  if(ret == 0) {
    for(k=0;k<hash_query.size();k++)
      offset[k]=0;
    for(int i=0;i<cont[0];i+=2) {
      if(ret==-2) {
	ret = 0;
	while(((punt_h[0][offset[0]] < ret1) && (offset[0] < cont[0])) || (punt_h[0][offset[0]+1]<cont_hash_query[0])) {
	  (offset[0])+=2;
	}
	if(offset[0]>=cont[0]) {
	  ret = -3;
	  break;
	}
      }
      if(ret==-3) {
	break;
      }
      if(punt_h[0][offset[0]+1] >= cont_hash_query[0]) {
	if(hash_query.size()>0)
	  for(k=1;k<hash_query.size();k++) {
	    
	    while(((punt_h[k][offset[k]] < punt_h[0][offset[0]]) && (offset[k]<cont[k]))||(punt_h[k][offset[k]+1]<cont_hash_query[k])) {
	      (offset[k])+=2;
	      if(offset[k]>=cont[k]) {
		ret = -3;
		break;
	      }
	    }
	    if((offset[k]>=cont[k]) || (ret == -3)) {
	      ret = -3;
	      break;
	    }
	    if((punt_h[k][offset[k]] > punt_h[0][offset[0]])) {
	      ret = -2;
	      ret1 = punt_h[k][offset[k]];
	      break;
	    }
	    if((punt_h[k][offset[k]] == punt_h[0][offset[0]]))
	      if(punt_h[k][offset[k]+1] < cont_hash_query[k]) {
		ret = -1;
		break;
	      }
	    offset[k]+=2;
	  }
	if(ret==-3) {
	  break;
	}
	if(ret!=-1 && ret!=-2) { // graph not pruned
	  f_count++;
	  fprintf(trans,"%d %d\n",f_count-1,punt_h[0][offset[0]]);
	  nGraphList.push_back(punt_h[0][offset[0]]); // list of graphs not filtered
	  offset[0]+=2;
	}
      }
      else {
	ret = -1;
	offset[0]+=2;
      }
    }
  }
  cout<<"Graphs after filtering:"<<f_count<<endl;
  cout << "Time load filtering:"<<(float)tot<<endl;
  db.close(0);
  fclose(trans);
  if(f_count==0) 
    exit(1);
}

////////////////////////////////////////////
//
// Class: GraphGrep
// Usage: Do subgraph matching
//
/////////////////////////////////////////////
class GraphGrep {
public:
  GraphGrep(){};				//Default Constructor
  void run();					//Start Matching
};

////////////////////////////////////////////
//
//	Implementation of Filtering Class
//
/////////////////////////////////////////////

//Checking 2 fp to see if the graph_fp contains the query_fp. 
bool Filtering::checkFP(unsigned int* graph_fp,unsigned int* query_fp){	
  unsigned int logicAnd=0;
  
  //check each int of the fp int array
  for(int i = 0; i < NUM_INTS; i++) { 
    if(query_fp[i]!=0){
      logicAnd = graph_fp[i] & query_fp[i];  	//boolean operation
      if(logicAnd != query_fp[i])
	return false;
    }
  }
  
  return true;
}

// main method of Filtering class
void Filtering::run() {
  time_t s1,s2;
  float tot=0;
  //Initial BerkeleyDB and setting its parameters
  Db db(0, 0);
  db.set_error_stream(&cerr);
  db.set_errpfx("GraphGrep:");
  db.set_pagesize(1024);		/* Page size: 1K. */
  
  db.set_cachesize(0, 32 * NUM_INTS*NUM_FP, 0);
  db.open(NULL, "fingerprint.db", NULL, DB_BTREE, DB_RDONLY, 0664);
  
  try {
    Dbt key;		//key
    Dbt data;		//value
    Dbc *dbcp;		//DB cursor
    db.cursor(NULL, &dbcp, 0);
    
    int count=0,f_count=0;
    
    db.cursor(NULL, &dbcp, 0);
    s1 = clock();
    while (dbcp->get(&key, &data, DB_NEXT) == 0) {		//fetch every data
      s2 = clock();
      tot += (float)((s2-s1)/CLOCKS_PER_SEC);
      char *key_string = (char *)key.get_data();
      //Here we divide the FP into groups, each group contains NUM_FP fingerprints.
      int igraph_end=atoi(key_string);
      int remainder=igraph_end % NUM_FP;
      int base=igraph_end / NUM_FP;
      
      int igraph_count=(remainder==0?NUM_FP: (igraph_end - (base)*NUM_FP));
      int igraph_start=igraph_end - igraph_count;
      
      //for debuging
      if(!QUIET_MODE){
	cout<< "igraph_start" <<igraph_start<<endl;
	cout<< " igraph_end" <<igraph_end <<endl;
	cout<< " igraph_count" <<igraph_count <<endl;
	cout<<"fp" <<*_fp<<endl;
      }
      
      unsigned int *data_string=(unsigned int*)data.get_data();
      
      for(int i=0;i<igraph_count;i++){
	unsigned int *p_fp=data_string+i*NUM_INTS;
	
	if(checkFP(p_fp,_fp)){  // graph i not pruned
	  f_count++;
	  nGraphList.push_back(igraph_start+i);
	}
      }
      
      count++;		
      s1 = clock();
    }
    cout<<"Graphs after filtering:"<<f_count<<endl;
    cout << "Time load filtering:"<<(float)tot<<endl;
    
  }
  catch (DbException &dbe) {
    cerr << "GraphGrep: " << dbe.what() << "\n";
  }
  //	db.close(0);
}


// internal method for finding key value from vector
int Find_node(int *h,int c,int m1) {
  int i;
  for(i=0;i<c;i++) {
    if(h[i]==m1) {
      return 1;
    }
  }
  return 0;
}

////////////////////////////////////////////
//
//	Implementation of GraphGrep Class
//
/////////////////////////////////////////////

// main method of GraphGrep class
void GraphGrep::run() {
  FILE *VF = NULL;                           // VF file
  FILE *VF_NEW = NULL;                       // VF with acceleration file
  setclock("DB start time",0);			
  Db db(0, 0);
  
  db.set_errpfx("GraphGrep:");
  db.set_pagesize(1024);		     /* Page size: 1K. */
  
  db.set_cachesize(0, 32 * 1024, 0);
  db.open(NULL, "graphdata.db", NULL, DB_BTREE, DB_RDONLY, 0664);
  try {
    if(GRAPHGREP_VF == true) {
      remove("Map"); // Map file not necessary here
    }
    setclock("DB start time",1);
    
    setclock("#Matching_time",0);
    
    OPFile opfile(opfilename,queryfilename,&db);
    vector<string> graphIDList;
    
    if(FILTERING) { // FROWN FILTERING
      graphIDList.reserve(nGraphList.size());  // nGraphList.size() = number of graphs after filtering
      if(GRAPHGREP_VF == true) { // VF GRAPH MATCHING
	if(nGraphList.size()>0) { 
	  for(int cic = 0;cic < lab_path.size();cic++) { // if one of pattern of the query has length 1
	    if(strlen(lab_path[cic].c_str())==1) {       // acceleration is not possible
	      ACC_VF=false;
	      break;
	    }
	  }
	  if(ACC_VF == true)
	    VF_NEW = fopen("Data_VF","w+");
	  else
	    VF = fopen("Map","w+");
	}
      }
      for(int x=0;x<nGraphList.size();x++){ // for every graph in the resulting filtering list
	char tabname[10];
	sprintf(tabname,"gn%d",nGraphList[x]);
	
	if(GRAPHGREP_VF == true) {
	  if(ACC_VF == false)
	    fprintf(VF,"%d\n",nGraphList[x]);
	  
	  /* */
	  // Acceleration is performed.
	  // Graphs are shrinked where possible according to the query patterns
	  if(ACC_VF == true) {
	    Dbc *dbcp;
	    Dbt key;
	    Dbt data;

	    int len = 0;
	    int *hash_h[5000];
	    int cont_hash[5000];
	    for(int op=0;op<5000;op++) {
	      cont_hash[op]=0;
	      hash_h[op]=NULL;
	    }
	    
	    int grows = 0;
	    int edg = 0;
	    char *edg_str=NULL;
	    char temp_edg[10];
	    
	    int m1,m2;

	    // lab_path contains the query labels paths
	    for(int cic = 0;cic < lab_path.size();cic++) {
	      int i,j=0;
	      char *tab=NULL;
	      tab = (char *)malloc(sizeof(char)*(strlen(tabname)+strlen(lab_path[cic].c_str())+1));
	      
	      strcpy(tab,tabname);
	      strcat(tab,lab_path[cic].c_str());
	      key.set_data(tab);
	      key.set_size(strlen(tab)+1);
	      db.cursor(NULL,&dbcp,0);
	      // retrieving graphs from database...
	      int ret1=dbcp->get(&key, &data, DB_SET);
	      if(ret1!=0)
	      	continue;

	      NodeType* dat=(NodeType*)data.get_data();
	      int sec_count=dat[0];
	      int lp=dat[1];
	      int total_rows=(dat[2]<<NodeTypeBitLen) + dat[3];
	      
	      int ret_pos=3;
	      int curpos=4;
	      int ret_rows=0;
	      for(int i=0;i<sec_count;i++){
		int sec_rows=(dat[curpos]<<NodeTypeBitLen) + dat[curpos+1];
		
		bool isLoadingSection=true;
		curpos += (2+lp);
		if(isLoadingSection){	//load the data
		  for(int nrows=0;nrows<sec_rows;nrows++){ // main cycle
		    for(int ncols=0;ncols<lp;ncols++){
		      m1 = m2;
		      m2 = dat[curpos+ncols+nrows*lp];
		      if(ncols>0) {
			if(m1<m2) {
			  if(Find_node(hash_h[m1],cont_hash[m1],m2)==0) {
			    hash_h[m1] = (int *)realloc(hash_h[m1],sizeof(int)*(cont_hash[m1]+1));
			    hash_h[m1][cont_hash[m1]] = m2;
			    cont_hash[m1]++;
			    if(m1 > grows)
			      grows = m1;
			    edg++;
			    sprintf(temp_edg,"%d %d\n",m1,m2);
			    int ll=0;
			    if(edg_str!=NULL)
			      ll = strlen(edg_str);
			    edg_str = (char *)realloc(edg_str,sizeof(char)*(strlen(temp_edg)+ll+1));
			    if(edg==1)
			      strcpy(edg_str,temp_edg);
			    else
			      strcat(edg_str,temp_edg);
			  }
			  if(m2>len)
			    len = m2;
			}
			else {
			  if(Find_node(hash_h[m2],cont_hash[m2],m1)==0) {
			    hash_h[m2] = (int *)realloc(hash_h[m2],sizeof(int)*(cont_hash[m2]+1));
			    hash_h[m2][cont_hash[m2]] = m1;
			    cont_hash[m2]++;
			    if(m2 > grows)
			      grows = m2;
			    edg++;
			    sprintf(temp_edg,"%d %d\n",m2,m1);
			    int ll=0;
			    if(edg_str!=NULL)
			      ll = strlen(edg_str);
			    edg_str = (char *)realloc(edg_str,sizeof(char)*(strlen(temp_edg)+ll+1));
			    if(edg==1)
			      strcpy(edg_str,temp_edg);
			    else
			      strcat(edg_str,temp_edg);
			  }
			  if(m1 > len)
			    len = m1;
			}
		      }
		    }
		    ret_rows++;
		  }
		}
		curpos += sec_rows*lp;
		j=0;
	      }
	      free(tab);
	    }
	    if(len+1<=1)
	      continue;
	    fprintf(VF_NEW,"#%s\n",tabname);
	    fprintf(VF_NEW,"%d\n",len+1);
	    char *lab=NULL;
	    if(ACC_VF == true && GRAPHGREP_VF == true) { // if VF and ACCELERATION
	      FILE *ff11;
	      char sf[40];
	      int ccoo;
	      sprintf(sf,".//graphblastdata//gn%d.node",nGraphList[x]);
	      ff11 = fopen(sf,"r");
	      fscanf(ff11,"%d",&ccoo);
	      char ty[50];
	      int r1=0;
	      // retrieving graph information from dn%.node
	      for(r1=0;r1<=len;r1++) {
		fscanf(ff11,"%s",ty);
		int l=0;
		if(lab!=NULL)
		  l = strlen(lab);
		lab = (char *)realloc(lab,sizeof(char)*(strlen(ty)+2+l));
		if(r1==0) {
		  strcpy(lab,ty);
		}
		else
		  strcat(lab,ty);
		if(r1!=len)
		  strcat(lab,"\n");
	      }
	      fclose(ff11);
	    }
	    // writing in VF_NEW the new shrinked graphs
	    fprintf(VF_NEW,"%s\n",lab);
	    fprintf(VF_NEW,"%d\n",edg);
	    fprintf(VF_NEW,"%s\n\n",edg_str);
	    free(lab);
	    free(edg_str);
	  }
	  
	}
	if(!QUIET_MODE)
	  cout<< "gn"<<nGraphList[x];
	graphIDList.push_back(string(tabname));
      }
      if(GRAPHGREP_VF == true)
	if(nGraphList.size()>0){
	  if(ACC_VF == false)
	    fclose(VF); 
	  else
	    fclose(VF_NEW);
	}
    }
    if(GRAPHGREP_VF == true) {
      if(nGraphList.size()>0 || nGraph > 0)  {
	char tmp[30];	
	char tmp1[30];
	strcpy(tmp,"./Vento "); // preparing the running string to call the vflib algorithm
	
	if(ACC_VF == true)
	  strcat(tmp,"Data_VF");
	else {
	  VF = fopen(".//graphblastdata//Data","r");
	  fscanf(VF,"%s",tmp1);
	  fclose(VF);
	  strcat(tmp,tmp1);
	}
	strcat(tmp," ");
	if(GLIDE == true) { // if inexact matching selected read the information from the query
	  char st[1000];
	  char st1[1000];
	  int cont_nodes = 0;
	  int cont_edges = 0;
	  strcpy(st,"");
	  strcpy(st1,"");
	  FILE *query_VF = fopen("query_glide","w+");
	  FILE *tmp_glide = fopen("type.dat","r");
	  fprintf(query_VF,"#Query_Glide\n");
	  while(!feof(tmp_glide)) {
	    char a[10];
	    fscanf(tmp_glide,"%s",a);
	    if(feof(tmp_glide))
	      break;
	    strcat(st,a);
	    strcat(st,"\n");
	    cont_nodes++;
	  }
	  fclose(tmp_glide);
	  tmp_glide = fopen("edge.dat","r");
	  while(!feof(tmp_glide)) {
	    char a[10];
	    fscanf(tmp_glide,"%s",a);
	    if(feof(tmp_glide))
	      break;
	    strcat(st1,a);
	    if(cont_edges%2==0)
	      strcat(st1," ");
	    cont_edges++;
	    if(cont_edges%2==0) {
	      strcat(st1,"\n");
	    }
	  }
	  fclose(tmp_glide);
	  fprintf(query_VF,"%d\n",cont_nodes);
	  fprintf(query_VF,"%s",st);
	  fprintf(query_VF,"%d\n",cont_edges/2);
	  fprintf(query_VF,"%s",st1);
	  fclose(query_VF);
	  strcat(tmp,"query_glide");
	}
	else
	  strcat(tmp,queryfilename);
	system(tmp);	    
	
	if(GLIDE == true) {
	  strcpy(tmp,queryfilename);
	  strcat(tmp,".out");
	  strcat(tmp1,"query_glide.out");
	  rename(tmp1,tmp);
	}
	return;
      }	
    }
    opfile.setGraphIDs(graphIDList);
    for(int i=0;i<opfile.getGraphIDs().size();i++){  // for every graph in the dataset after filtering
      opfile.setCurrentGraphID(opfile.getGraphIDs()[i]);
      opfile.run();
      opfile.reset();
    }
  }
  catch (DbException &dbe) {
    cerr << "GraphGrep: " << dbe.what() << "\n";
  }
  setclock("#Matching_time",1);
  setclock("closeDB time",0);	
  //db.close(0);
  setclock("closeDB time",1);	
}

////////////////////////////////////////////
//
//	Static functions
//
/////////////////////////////////////////////

//read graph from a query file
static GraphADT* readGraph(ifstream& f){	
  GraphADT* retG;
  char line[51];
  
  while(!f.eof()){
    f.getline(line,50);
    if(line[0]=='#'){
      f.getline(line,50);
      int numNode=atoi(line);
      retG=new GraphADT(numNode,false);
      for(int i=0;i<numNode;i++){	//read node
	f.getline(line,50);
	(*retG).addNode(i,line);
      }
      f.getline(line,50);
      int numEdge=atoi(line);
      for(int j=0;j<numEdge;j++){
	f.getline(line,50);
	string tmp(line);
	vector<string> nodes=sep_words(tmp);
	(*retG).insert(Edge(nodes[0],nodes[1]));
      }
      
    }
  }
  
  return retG;
}

//generating opfile from a input query file. 
//for graphgrep use
//exact matching
static bool GenOpfile(char* queryfile){
  setclock("#Query_creating_time",0);
  GraphADT* g;
  
  if(queryfile!=""){
    ifstream fin(queryfile);
    if(! fin){
      cout<<queryfile <<" does not exist\n";
      exit(0);
    }
    g=readGraph(fin);
    fin.close();
  }else{
    return false;
  }
  
  vDFS_GraphGrep<GraphADT> myDFS(*g,DEFAULT_LP);
  if(FILTERING2 == true) {
    vet_h = (int *)malloc(sizeof(int)*hash_query.size());
    
    for(int i=0;i<hash_query.size();i++) {
      int h = 0;
      for(int j=0;j<hash_query[i].length();j++) {
	h = (64*h + (hash_query[i])[j])%HASHP;
      }
      vet_h[i] = h;
    }
  }
  
  char tmp[100];
  sprintf(tmp,"%s.out",queryfile);
  if(GRAPHGREP_VF == false)
    myDFS.writeOutputOrder(tmp);
  
  setclock("#Query_creating_time",1);
  if(FILTERING2 == false) {
    if(FILTERING){
      setclock("#Filtering_time",0);
      unsigned int* test_fp=myDFS.getFingerPrint();
      Filtering filter(test_fp);
      filter.run();
      setclock("#Filtering_time",1);
    }
  }
  else {
    setclock("#Filtering_time",0);
    Filtering2 filter2;
    filter2.run();
    setclock("#Filtering_time",1);
  }
  printtime("#Query_creating_time");
  printtime("#Filtering_time");
  return true;
}

//generating opfile from a input query file. 
//for graphgrep use
//inexact matching
static bool GenOpfileGlide(char* queryfile,Parse parse){ 
  setclock("#Query_creating_time",0); 
  GraphADT* g;
  char line[51];
  ifstream fin("type.dat");
  if(! fin){
    cout<<"types.dat does not exist\n";
    exit(0);
  }

  g = new GraphADT(parse.ParseReturnNumNode(),false);
  for(int i=0;i<parse.ParseReturnNumNode();i++){	//read node
    fin.getline(line,50);
    g->addNode(i,line);
  }
  fin.close();
  
  fin.open("edge.dat");
  if(!fin) {
    cout << "edge.dat does not exist\n";
    exit(0);
  }
  for(int j=0;j<parse.ParseReturnNumEdge();j++){
    fin.getline(line,50);
    string tmp(line);
    vector<string> nodes=sep_words(tmp);
    g->insert(Edge(nodes[0],nodes[1]));
  }
  fin.close();
  
  vDFS_GraphGrep<GraphADT> myDFS(*g,DEFAULT_LP);
  if(FILTERING2 == true) {
    vet_h = (int *)malloc(sizeof(int)*hash_query.size());
    
    for(int i=0;i<hash_query.size();i++) {
      int h = 0;
      for(int j=0;j<hash_query[i].length();j++) {
	h = (64*h + (hash_query[i])[j])%HASHP;
      }
      vet_h[i] = h;
    }
  }

  char tmp[100];

  sprintf(tmp,"%s.out",queryfile);
  if(GRAPHGREP_VF == false)
    myDFS.writeOutputOrder(tmp);
  setclock("#Query_creating_time",1);

  if(FILTERING2 == false) {
    if(FILTERING){
      setclock("#Filtering_time",0);
      unsigned int* test_fp=myDFS.getFingerPrint();
      Filtering filter(test_fp);
      filter.run();
      setclock("#Filtering_time",1);
    }
  }
  else {
    setclock("#Filtering_time",0);
    Filtering2 filter2;
    filter2.run();
    setclock("#Filtering_time",1);
  }

  printtime("#Query_creating_time");
  printtime("#Filtering_time");
  return true;
}             


////////////////////////////////////////////
//
//	Main Function
//
/////////////////////////////////////////////
static double app_start;
static double app_end;

// main function
int main(int argc, char** argv) {
  try {
    setclock("total running time",0);			
    app_start=clock();
    GraphGrep app;
    
    bool HasQueryFile=false;
    for (int n = 1; n <argc; n++) {
      if (strcmp(argv[n],"-v")==0) {
	QUIET_MODE=false;
      }
      else if (strcmp(argv[n],"-o")==0) {
	OUTPUT_DATA_COUNT=true;
      }
      else if (strcmp(argv[n],"-noFL")==0) {
	FILTERING=false;
      }
      else if (strcmp(argv[n],"-lp")==0) {
	DEFAULT_LP=atoi(argv[++n]);
      }
      else if (strcmp(argv[n],"-fp")==0) {
	NUM_INTS=atoi(argv[++n]);
      }
      else if (strcmp(argv[n],"-n")==0) {
	nGraph=atoi(argv[++n]);
	nGraphList.reserve(nGraph);
      }
      else if (strcmp(argv[n],"-x")==0) {
	strcpy(queryfilename,argv[++n]);
	char fileout[100];
	sprintf(fileout,"%s.out",queryfilename);
	
	strcpy(opfilename,"opfile");
	
	if(GRAPHGREP_VF == false)
	  (void)remove(fileout);
	HasQueryFile=true;
	GLIDE = true;
      } 
      else if (strcmp(argv[n],"-m")==0) {				
	strcpy(queryfilename,argv[++n]);
	
	char fileout[100];
	sprintf(fileout,"%s.out",queryfilename);
	
	strcpy(opfilename,"opfile");
	
	if(GRAPHGREP_VF == false)
	  (void)remove(fileout);
	HasQueryFile=true;
	
      }else if(strcmp(argv[n],"-degree")==0) {
	QUERY_WITH_DEGREE=true;
      }
      else {
	cout<< "Usage: GraphBlast type_of_match queryfile [-o -v -noFL -degree -n numGraphs -lp lengthpath -fp length_of_FP -select method]\n"<<
	  "\ttype_of_match    -m for exact matching, -x for inexact matching\n"<<
	  "\t-m               query in LNE for format exact matching\n"<<
	  "\t-x               query in GLIDE format for inexact matching\n"<<
	  "\t-o               output count\n"<<
	  "\t-v               verbose mode\n"<<
	  "\t-degree          query with degree\n"<<
	  "\t-noFL            disable filtering\n"<<
	  "\t-n  numGraphs    input number of graphs in DB(only with option -noFL)\n"<<
	  "\t-lp lengthpath   input different LP, default is 4\n"<<
	  "\t-fp length of FP input different length of FP, default is 256\n"<<
	  "\t-select method   gb for graphblast, default\n"<<
	  "\t                 gg for graphgrep\n"<<
	  "\t                 gf for growns\n"<<
	  "\t                 dg for daygb\n"<<
	  "\t                 Active only if graphbuild mode is manual"<<endl;
	exit(2);
      }
    }  // end for(....)

    ifstream autom;
    autom.open(".//method.dat"); // looking for the automatic method to run
    char str_auto[3];
    if(autom.is_open()) {
      autom.getline(str_auto,3);
      if(strcmp(str_auto,"gb")==0)
	METHOD = 0;
      if(strcmp(str_auto,"gg")==0)
	METHOD = 1;
      if(strcmp(str_auto,"gf")==0)
	METHOD = 2;
      if(strcmp(str_auto,"dg")==0)
	METHOD = 3;
    }
    autom.close();

    char *qrem=NULL;
    qrem = (char *)malloc(sizeof(char)*(strlen(queryfilename)+4));
    strcpy(qrem,"./");
    strcat(qrem,queryfilename);
    strcat(qrem,"\0");
    char *qrem1=NULL;
    qrem1 = (char *)malloc(sizeof(char)*(strlen(qrem)+9));
    strcpy(qrem1,qrem);
    strcat(qrem1,".out");
    strcat(qrem1,"\0");
    char *qrem2=NULL;
    qrem2 = (char *)malloc(sizeof(char)*(strlen(qrem)+12));
    strcpy(qrem2,qrem);
    strcat(qrem2,".output");
    strcat(qrem2,"\0");
    remove(qrem1);
    remove(qrem2);
    free(qrem);
    free(qrem1);
    free(qrem2);
    
    switch(METHOD) { // which method it has to run
    case 0:
      FILTERING2 = true;
      GRAPHGREP_VF = true;
      ACC_VF = true;
      break;
    case 1:
      FILTERING2 = true;
      GRAPHGREP_VF = false;
      ACC_VF = false;
      break;
    case 2:
      FILTERING2 = false;
      GRAPHGREP_VF = true;
      ACC_VF = false;
      break;
    case 3:
      FILTERING2 = false;
      GRAPHGREP_VF = true;
      ACC_VF = true;
      break;
    }
    if(HasQueryFile){
      if(GLIDE == false) { // exact matching
	if(GenOpfile(queryfilename)) {
	  app.run();
	  fprintf(stdout,"Load Table time :%f\n",tot_s);
	}
	else{
	  cout<<"error happened in generating opfile\n";
	  exit(0);
	}
	
	setclock("total running time",1);
	app_end=clock();
	
	if(GRAPHGREP_VF == false) {
	  printtime("#Matching_time");
	  //printtime("create from DB time");
	  
	  //  cout<<"#Total_running_time :"<<(app_end-app_start)/(CLOCKS_PER_SEC)<<"\n";
	}
	fprintf(stdout,"\n");
      }
      else { // inexact matching
	if(METHOD==0 || METHOD==1 || METHOD==3) {
	  cout << "\nInexact Matching available only with GFrowns." << endl << endl;
	  return -1;
	}
	FILE *fp;
	fp = fopen(queryfilename,"r+");
	Parse parse;
	parse.ParseCreate();
	parse.ParseParsePath(fp);
	fclose(fp);
	query = new Query(parse.ParseReturnNumNode());
	if(GenOpfileGlide(queryfilename,parse)) { // opfile creation for inexact matching
	  app.run();
	  char str1[30];
	  strcpy(str1,queryfilename);
	  strcat(str1,".out");
	  if(GRAPHGREP_VF == false) {
	    printtime("#Matching_time");
	    //printtime("create from DB time");
	    
	    //cout<<"#Total_running_time :"<<(app_end-app_start)/(CLOCKS_PER_SEC)<<"\n";
	  }
	  if(wildhead!=NULL) { // inexact matching structure for glide queries
	    if(GRAPHGREP_VF == true) {
	      query->CheckInOutTableWildConnection_VF(str1);
	    }
	    else
	      query->CheckInOutTableWildConnection(str1);
	  }
	}
	else {
	  cout<<"error happened in generating opfile\n";
	  exit(0);
	}
	setclock("total running time",1);
	app_end=clock();
      }
    }else{
	cout<< "Usage: GraphBlast type_of_match queryfile [-o -v -noFL -degree -n numGraphs -lp lengthpath -fp length_of_FP -select method]\n"<<
	  "\ttype_of_match    -m for exact matching, -x for inexact matching\n"<<
	  "\t-m               query in LNE for format exact matching\n"<<
	  "\t-x               query in GLIDE format for inexact matching\n"<<
	  "\t-o               output count\n"<<
	  "\t-v               verbose mode\n"<<
	  "\t-degree          query with degree\n"<<
	  "\t-noFL            disable filtering\n"<<
	  "\t-n  numGraphs    input number of graphs in DB(only with option -noFL)\n"<<
	  "\t-lp lengthpath   input different LP, default is 4\n"<<
	  "\t-fp length of FP input different length of FP, default is 256\n"<<
	  "\t-select method   gb for graphblast, default\n"<<
	  "\t                 gg for graphgrep\n"<<
	  "\t                 gf for growns\n"<<
	  "\t                 dg for daygb\n"<<
	  "\t                 Active only if graphbuild mode is manual"<<endl;
    }
    delete[] cont_hash_query;
    delete[] vet_h;
    return (EXIT_SUCCESS);
    
  }
  catch (DbException &dbe) {
    cerr << "GraphGrep: " << dbe.what() << "\n";
    return (EXIT_FAILURE);
  }
}
