function [X,y,Z,iter,compval,feasval,objval,termflag] = ...
   flsdp(G,w,X,y,Z,opt)
% FLSDP  Semidefinite Programming Solver using the XZ method
%        for SDP problems arising in computing the Lovasz theta
%        function of a graph. Thus we have n = #vertices of the
%        graph, m = #edges + 1, b = [0 ... 0 1]^T, and
%        C(i,j) = - sqrt(w(i) w(j)), where w is a vector of weights;
%        there is one constraint matrix A_k for each edge (p,q) with
%           A_k = E_pq = e_p e_q^T + e_q e_p^T, 1 <= k < m,
%        and the last constraint matrix A_m is just the identity.
%
%
% [X,y,Z,iter,compval,feasval,objval,termflag] = ...
%   flsdp(G,w,X,y,Z,opt)
%
%   It follows that the Schur complement matrix M is a
%   symmetric matrix of the form: let Zinv = Z^{-1}, and
%   for 1 <= k <= l <= m-1, let (p,q) be the k-th edge and
%   (i,j) be the l-th edge of the graph; then we have
%
%      M(k,l) =  trace((E_pq X E_ij Zinv)
%             =  X(p,i)Zinv(j,q) + X(p,j)Zinv(i,q)
%               +X(q,i)Zinv(j,p) + X(q,j)Zinv(i,p),
%   and
%      M(k,m) = (X*Zinv)(p,q) + (X*Zinv)(q,p),
%      M(m,m) = trace(X*Zinv),
%
%   and M is symmetric.
%
%   The constraints  <A_k,X> = bk  reduce to
%       X(p,q) + X(q,p) = 2 X(p,q) = 0,, i.e. X(p,q) = 0
%   if (p,q) = k-th edge, 1 <= k <= m-1, while the constraint
%   <A_m,X> = bm reduces to trace(X) = 1.
%
%   Note: the matrix G holds the edges of the graph in the
%   following format: if the k-th ege is (p,q), then
%               G(k,1) = p, G(k,2) = q.
%
%  input variables
%     - G               list of edges in the graph
%     - w               positive vector of weights
%     - X.s             initial guess for primal variable
%     - y               initial guess for dual variable
%     - Z.s             initial guess for dual slack
%     - opt             structure of options
%     -    maxit        maximum iterations allowed
%     -    tau          fraction of step to boundary of psd cone
%     -    steptol      tolerance for short steps
%     -    abstol       absolute tolerance on gap
%     -    reltol       relative tolerance on gap
%     -    gapprogtol   tolerance on sufficient progress
%     -    feasprogtol  tolerance on sufficient progress
%     -    bndtol       upper bound on size of solution
%     -    prtlevel     verbosity level
%     -    validate     check for graph connectedness
%
%  output variables
%     - X.s          computed primal solution
%     - y            computed dual solution
%     - Z.s          computed dual slack solution
%     - iter         the number of iterations performed by the algorithm
%     - compval       vector of gap values of length iter+1
%     - feasval      matrix of primal and dual feasibility residuals, of size (iter+1)x2
%     - objval       matrix of primal and dual objective values, of size (iter+1)x2
%     - termflag     termination flag

% SDPPACK Version 0.9 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton, S. Schmieta
% Last modified: 6/8/97

% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> either X or Z is not strictly positive definite, i.e.
%                       either chol(X) or chol(Z) failed, or Z is singular
%                       (i.e. inversion of Z failed)
%    termflag =  2  ==> not used by this special routine
%    termflag =  3  ==> SchurComp numerically indefinite or singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X or Z is not strictly positive
%                                   definite
%    termflag =  2 and iter = 0 ==> not used by this special routine
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 termflag = 7;
 iter = 0;
%
% check number of input arguments
%
 if nargin ~= 6
    if prtlevel > 0
       fprintf('flsdp: Incorrect number of input arguments.  Aborting...\n');
    end
    return
 end
%
% for convenience
%
 maxit = opt.maxit;
 tau = opt.tau;
 steptol = opt.steptol;
 abstol = opt.abstol;
 reltol = opt.reltol;
 gapprogtol = opt.gapprogtol;
 feasprogtol = opt.feasprogtol;
 bndtol = opt.bndtol;
 prtlevel = opt.prtlevel;
 validate = opt.validate;
%
% some simple data validation
%
 m = length(y);
 if m ~= size(G,1) + 1
    if prtlevel > 0
       fprintf('flsdp: y is improperly dimensioned\n');
    end
    return
 end
 n = length(w);
 if (n ~= length(X) | size(X,1) ~= size(X,2))
    if prtlevel > 0
       fprintf('flsdp: X is improperly dimensioned\n');
    end
    return
 end
 if (n ~= length(Z) | size(Z,1) ~= size(Z,2))
    if prtlevel > 0
       fprintf('flsdp: Z is improperly dimensioned\n');
    end
    return
 end
%
% too many edges?
%
 if m > n*(n-1)/2
    if prtlevel > 0
       fprintf('flsdp: Too many edges\n');
       fprintf('       A graph with %d vertices can have at most %d edges\n',...
             n, n*(n-1)/2);
    end
    return
 end
%
% is the graph connected?
%
 if validate
    Adj = zeros(n);
    for i=1:length(G)
       Adj(G(i,1), G(i,2)) = 1;
    end
    Adj = Adj + Adj';   % computes symmetric adj. matrix
    Adj = Adj + eye(n);
    Anew = Adj*Adj;
    Anew = (Anew > 0);
    while sum(sum(abs(Anew-Adj))) ~= 0
       Adj = Anew;
       Anew = Anew*Anew;
       Anew = (Anew > 0);
    end
    Missing = (Anew == 0);
    [I J] = find(tril(Missing));
    if length(I) > 0
       fprintf('flsdp: Graph not connected!  Proceeding anyway.\n\n');
    end
 end
%
% Compute chol(X) and chol(Z) so they are available on entry to the loop
%
 [chol_X,indef] = chol(X);
 if indef > 0
    fprintf('flsdp: initial X is not positive definite.  Aborting...\n');
    termflag = 1;
    return
 end
%
 [chol_Z,indef] = chol(Z);
 if indef > 0
    fprintf('flsdp: initial Z is not positive definite.  Aborting...\n');
    termflag = 1;
    return
 end
%
% Now some initialization
%
 vX = svec(X,n);
 vZ = svec(Z,n);
 C = - sqrt(w*w');
 vC = svec(C,n);
 Id = speye(n);
 comp = vX'*vZ;
 compval(1) = comp;
 nedges = m - 1;
 Xstep = 0;
 Zstep = 0;
 Xpos = 1;
 Zpos = 1;
 termflag = 6;
%
% compute the primal and dual residual rp and Rd
%
 rp = zeros(m,1);
 Rd = C - Z;
 for k = 1:nedges
    p = G(k,1);
    q = G(k,2);
    rp(k) = -2*X(p,q);
    Rd(p,q) = Rd(p,q) - y(k);
    Rd(q,p) = Rd(p,q);
 end
 rp(m) = 1 - trace(X);
 Rd = Rd - y(m)*Id;
 pri_infeas = norm(rp);
 dual_infeas = norm(svec(Rd,n));
 objval(1,1) = vC'*vX;         % = <C,X>
 objval(1,2) = y(m);           % = b^T y
 feasval(1,1) = pri_infeas;
 feasval(1,2) = dual_infeas;
%
% print header and initial info
%
 if prtlevel > 0
    fprintf('iter   p_step      d_step     p_infeas    d_infeas     X . Z');
    fprintf('        pobj        dobj\n');
    fprintf('%3.0f %11.3e %11.3e', 0, 0.0, 0.0);
    fprintf(' %11.3e %11.3e %11.3e', pri_infeas, dual_infeas, comp);
    fprintf(' %11.3e %11.3e\n',objval(1,1),objval(1,2));
 end
%
% start main loop
%
 for iter = 1:maxit
%
% construct Schur complement
%
    SchurComp = zeros(m);
    Zinv = inv(Z);
    if max(max(abs(Zinv))) == Inf
       if prtlevel > 0
          fprintf('flsdp: stop since limiting accuracy reached.\n');
          fprintf('      (Z is numerically singular)\n');
       end
       termflag = 1;
       iter = iter - 1;
       return
    end
    XZinv = X*Zinv;
    for k = 1:nedges
       p = G(k,1);
       q = G(k,2);
       for l = k:nedges
          i = G(l,1);
          j = G(l,2);
          SchurComp(k,l) = X(p,i)*Zinv(j,q) + X(p,j)*Zinv(i,q) +...
                           X(q,i)*Zinv(j,p) + X(q,j)*Zinv(i,p);
       end
       SchurComp(k,m) = XZinv(p,q) + XZinv(q,p);
    end
    SchurComp(m,m) = trace(XZinv);
%
% Now symmetrize SchurComp
%
    dSC = diag(SchurComp);
    SchurComp = SchurComp - diag(dSC);
    SchurComp = SchurComp + SchurComp';
    SchurComp = SchurComp + diag(dSC);
%
% and compute its Choleski factorization
%
    [Cholfac,indef] = chol(SchurComp);
    if indef > 0
       if prtlevel > 0
          fprintf('flsdp: stop since limiting accuracy reached.\n');
          fprintf('      (Schur complement is numerically indefinite)\n');
       end
       termflag = 3;
       iter = iter - 1;
       return
    end
    minpiv = min(abs(diag(Cholfac)));
    if minpiv == 0
       if prtlevel > 0
          fprintf('flsdp: stop since limiting accuracy reached.\n');
          fprintf('      (Schur complement is numerically singular)\n');
       end
       termflag = 3;
       iter = iter-1;
       return
    end
%
% predictor step
%
    Rc = - X*Z;
%
    tmp1 = (X*Rd - Rc)*Zinv;
    yrhs = rp;
    for k = 1:nedges
       p = G(k,1);
       q = G(k,2);
       yrhs(k) = yrhs(k) + tmp1(p,q) + tmp1(q,p);
    end
    yrhs(m) = yrhs(m) + trace(tmp1);
    dy = Cholfac\(Cholfac'\yrhs);
    tmp2 = zeros(n);
    for k = 1:nedges   % compute sum dy(k) A_k
       p = G(k,1);
       q = G(k,2);
       tmp2(p,q) = dy(k);
       tmp2(q,p) = tmp2(p,q);
    end
    tmp2 = tmp2 + dy(m)*Id;
    dZ = Rd - tmp2;
    dX = -tmp1 + X*tmp2*Zinv;
%
% symmetrize dX : necessary
% note that there is no need to symmetrize dZ
%
    dX = 0.5*(dX + dX');
%
% compute new gap
%
    Xstep = min(1,tau*sdbound(chol_X,dX,n));  % tau times step
    Zstep = min(1,tau*sdbound(chol_Z,dZ,n));  % to boundary
    Xnew = X + Xstep*dX;
    Znew = Z + Zstep*dZ;
    gapnew = full(svec(Xnew,n)'*svec(Znew,n));
%
% corrector step
%
    mu = (gapnew/comp)^2 * gapnew/n;   % Mehrotra's formula
    Rc = Rc + mu*Id - dX*dZ;           % included mu term this time
%
% Compute the corrector step
%
    tmp1 = (X*Rd - Rc)*Zinv;
    yrhs = rp;
    for k = 1:nedges
       p = G(k,1);
       q = G(k,2);
       yrhs(k) = yrhs(k) + tmp1(p,q) + tmp1(q,p);
    end
    yrhs(m) = yrhs(m) + trace(tmp1);
    dy = Cholfac\(Cholfac'\yrhs);
    tmp2 = zeros(n);
    for k = 1:nedges
       p = G(k,1);
       q = G(k,2);
       tmp2(p,q) = dy(k);
       tmp2(q,p) = tmp2(p,q);
    end
    tmp2 = tmp2 + dy(m)*Id;
    dZ = Rd - tmp2;
    dX = -tmp1 + X*tmp2*Zinv;
%
% symmetrize dX
%
    dX = 0.5*(dX + dX');
%
% and compute the stepsizes
%
    Xstep = min(1,tau*sdbound(chol_X,dX,n));
    Zstep = min(1,tau*sdbound(chol_Z,dZ,n));
%
% failure if steps get too short
%
    if Xstep < steptol | Zstep < steptol
       if prtlevel > 0
          fprintf('flsdp: stop since steps are too short\n');
       end
       termflag = 5;
       iter = iter-1;
       break
    end
%
% otherwise decide whether new point provides "substantial" improvement
% over previous iterate
%
    Xnew = X + Xstep*dX;
    Znew = Z + Zstep*dZ;
    ynew = y + Zstep*dy;        % end of corrector step and iteration
%
% check for positive definiteness of Xnew
%
    [chol_X,indef] = chol(Xnew);   % if Xnew is indefinite, we quit
    if indef > 0
       termflag = 1;
       Xpos = 0;
    end
%
% check for positive definiteness of Znew
%
    [chol_Z, indef] = chol(Znew);  % if Znew is indefinite, we quit
    if indef > 0
       termflag = 1;
       Zpos = 0;
    end
%
% decide whether to accept new point or quit
%
    vXnew = svec(Xnew,n);
    vZnew = svec(Znew,n);
    gapnew = full(vXnew'*vZnew);
    rpnew = zeros(m,1);
    Rdnew = C-Znew;
    for k = 1:nedges
       p = G(k,1);
       q = G(k,2);
       rpnew(k) = -2*Xnew(p,q);
       Rdnew(p,q) = Rdnew(p,q) - ynew(k);
       Rdnew(q,p) = Rdnew(p,q);
    end
    rpnew(m) = 1 - trace(Xnew);
    Rdnew = Rdnew - ynew(m)*Id;
    pri_infeas_new = norm(rpnew);
    dual_infeas_new = norm(svec(Rdnew,n));
%
    if gapnew < 100*abstol & gapprogtol*gapnew > comp & ...
       (pri_infeas_new > feasprogtol*pri_infeas | ...
        dual_infeas_new > feasprogtol*dual_infeas)
%
% reject the new iterate
%
       termflag = 4;
       iter = iter-1;
       if prtlevel > 0
          fprintf('Stop since new point is substantially worse than current iterate\n');
          fprintf('      X . Z = %11.3e\n',gapnew);
          fprintf('      pri_infeas = %11.3e\n', pri_infeas_new);
          fprintf('      dual_infeas = %11.3e\n', dual_infeas_new);
          break
       end
    else
%
% accept new point
%
       X =Xnew;
       y = ynew;
       Z = Znew;
       vX = vXnew;
       vZ = vZnew;
       comp = gapnew;
       rp = rpnew;
       Rd = Rdnew;
       pri_infeas = pri_infeas_new;
       dual_infeas = dual_infeas_new;
%
% compute objective values so they can be plotted later
%
       compval(iter+1) = comp;
       objval(iter+1,1) = vC'*vX;
       objval(iter+1,2) = y(m);
       feasval(iter+1,1) = pri_infeas;
       feasval(iter+1,2) = dual_infeas;
%
% and display information
%
       if prtlevel > 0
          fprintf('%3.0f %11.3e %11.3e', iter, Xstep, Zstep);
          fprintf(' %11.3e %11.3e %11.3e', pri_infeas, dual_infeas, comp);
          fprintf(' %11.3e %11.3e\n',objval(iter+1,1),objval(iter+1,2));
       end
    end
%
% termination test
%
    if termflag == 1       % X or Z does not lie inside psd cone, so quit
       if prtlevel > 0
          fprintf('flsdp: stop since limiting accuracy reached\n');
          if ~Xpos
             fprintf(' (smallest eigenvalue of X = %11.3e)\n', min(eig(X,n)));
          end
          if ~Zpos
             fprintf(' (smallest eigenvalue of Z = %11.3e)\n', min(eig(Z,n)));
          end
       end
       return
    end
    total_err = comp + pri_infeas + dual_infeas;
    normX = norm(vX);
    normZ = norm(vZ);
    if total_err < min(abstol,reltol*(normX + normZ))
       if prtlevel > 0
          fprintf('flsdp: stop since error reduced to desired value\n');
       end
       termflag = 0;
       break
    elseif normX > bndtol
       if prtlevel > 0
          fprintf('flsdp: stop since X is blowing up\n');
       end
       termflag = -2;
       break
    elseif normZ > bndtol
       if prtlevel > 0
          fprintf('flsdp: stop since Z is blowing up\n');
       end
       termflag = -1;
       break
    end
 end
%
% END function
