function [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
   fdsdp(C,b,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
         gapprogtol,feasprogtol,bndtol,prtlevel)
% FDSDP  Semidefinite Programming Solver using the XZ method
%        for problems with diagonal constraints, namely m = n =
%        sum(blk) and the constraint matrices are of the form
%
%                 Ak = Ek = ek ek^T,   1 <= k <= n
%
% [X,y,Z,iter,gapval,feasval,objval,termflag] = ...
%    fdsdp(C,b,blk,X,y,Z,maxit,tau,steptol,abstol,reltol,...
%          gapprogtol,feasprogtol,bndtol,prtlevel)
%
%   It follows that the Schur complement matrix is just the
%   Hadamard product of X with Z^{-1}, that is the (k,l)
%   entry is the product of the (k,l) entry of X with the
%   (k,l) entry of Z^{-1}.
%   The constraints  <Ak,X> = bk  reduce to  X(k,k) = bk,
%   while  A'*y  is simply  diag(y);
%
%  input variables
%     - C            cost matrix
%     - b            rhs of primal constraints
%     - blk          block structure vector
%     - X            initial guess for primal variable
%     - y            initial guess for dual variable
%     - Z            initial guess for dual slack
%     - maxit        maximum iterations allowed
%     - tau          fraction of step to boundary of psd cone
%     - steptol      tolerance for short steps
%     - abstol       absolute tolerance on gap
%     - reltol       relative tolerance on gap
%     - gapprogtol   tolerance on sufficient progress
%     - feasprogtol  tolerance on sufficient progress
%     - bndtol       upper bound on size of solution
%     - prtlevel     verbosity level
%
%  output variables
%     - X            computed primal solution
%     - y            computed dual solution
%     - Z            computed dual slack solution
%     - iter         the number of iterations performed by the algorithm
%     - gapval       vector of gap values of length iter+1
%     - feasval      matrix of primal and dual feasibility residuals, of size (iter+1)x2
%     - objval       matrix of primal and dual objective values, of size (iter+1)x2
%     - termflag     termination flag

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97

% termination flag:
%    termflag = -2  ==> X is blowing up
%    termflag = -1  ==> Z is blowing up
%    termflag =  0  ==> success
%    termflag =  1  ==> new point rejected since X indefinite:
%                       chol(X) failed
%    termflag =  2  ==> new point rejected since Z indefinite:
%                       chol(Z) failed or Z is singular
%    termflag =  3  ==> SchurComp is numerically indefinite or singular
%    termflag =  4  ==> new point rejected: worse than current point
%    termflag =  5  ==> shortsteps
%    termflag =  6  ==> exceeded maximum number of iterations
%    termflag =  7  ==> data failed validation test
%
% Also:
%    termflag =  1 and iter = 0 ==> initial X is not positive definite
%    termflag =  2 and iter = 0 ==> initial Z is not positive definite
%    termflag =  4 and iter = 0 ==> Initial pt. may be too close to bdry
%    termflag =  5 and iter = 0 ==> Initial pt. may be too close to bdry
%
% If maxit = 0, we assume the user wants to do just data validation.
%    termflag =  6  ==> data passed validation test
%
 termflag = 7;
 iter = 0;
%
% check number of input arguments
%
 if nargin ~= 15,
    if prtlevel > 0,
       fprintf('fdsdp: Incorrect number of input arguments.  Aborting...\n');
    end;
    return;
 end;
%
% some data validation
%
 sumblk2 = sum(blk .* (blk+1))/2;
 m = length(b);
 n = sum(blk);
 if m > sumblk2,
    if prtlevel > 0,
       fprintf('\nfdsdp: too many constraints.  Try "help preproc"\n');
    end;
    return;
 end;
 if m ~= n,
    if prtlevel > 0,
       fprintf('\nfdsdp: this routine handles diagonal constraints problems only!\n');
    end;
    return;
 end;
 if m ~= length(y),
    if prtlevel > 0,
       fprintf('\nfdsdp: y is improperly dimensioned\n');
    end;
    return;
 end;
 if (n ~= length(C) | size(C,1) ~= size(C,2)),
    if prtlevel > 0,
       fprintf('\nfdsdp: C is improperly dimensioned\n');
    end;
    return;
 end;
 if (n ~= length(X) | size(X,1) ~= size(X,2)),
    if prtlevel > 0,
       fprintf('\nfdsdp: X is improperly dimensioned\n');
    end;
    return;
 end;
 if (n ~= length(Z) | size(Z,1) ~= size(Z,2)),
    if prtlevel > 0,
       fprintf('\nfdsdp: Z is improperly dimensioned\n');
    end;
    return;
 end;
%
% Compute chol(X) and chol(Z) so they are available on entry to the loop
%
 [chol_X,indef] = chol(X);
 if indef > 0,
    if prtlevel > 0,
       fprintf('\nfdsdp: initial X is not positive definite.  Aborting...\n');
    end;
    termflag = 1;
    return;
 end;
%
 [chol_Z,indef] = chol(Z);
 if indef > 0,
    if prtlevel > 0,
       fprintf('\nfdsdp: initial Z is not positive definite.  Aborting...\n');
    end;
    termflag = 2;
    return;
 end;
%
% Now some initialization
%
 vX = svec(X,blk);
 vZ = svec(Z,blk);
 vC = svec(C,blk);           % used many times!
 Id = speye(n);
 dgap = full(vX'*vZ);
 rp = b - diag(X);
 pri_infeas = norm(rp);
 Rd = C - Z - spdiags(y,0,speye(n));  % sparse matrix with y on the diagonal
 dual_infeas = norm(svec(Rd,blk));
 gapval(1) = dgap;
 objval(1,1) = vC'*vX;
 objval(1,2) = b'*y;
 feasval(1,1) = pri_infeas;
 feasval(1,2) = dual_infeas;
 Xstep = 0;
 Zstep = 0;
 termflag = 6;
%
% print header and initial info
%
 if prtlevel > 0,
    fprintf('iter   p_step      d_step     p_infeas    d_infeas      X.Z ');
    fprintf('        pobj        dobj\n');
    fprintf('%3.0f %11.3e %11.3e', 0, 0.0, 0.0);
    fprintf(' %11.3e %11.3e %11.3e', pri_infeas, dual_infeas, dgap);
    fprintf(' %11.3e %11.3e\n',objval(1,1),objval(1,2));
 end;
%
% start main loop
%
 for iter = 1:maxit;
%
% construct Schur complement: Hadamard product, X .* Z{-1}
%
    Zinv = inv(Z);
    if max(max(abs(Zinv))) == Inf,
       if prtlevel > 0,
          fprintf('fdsdp: stop since limiting accuracy reached.\n');
          fprintf('      (Z is numerically singular)\n');
       end;
       termflag = 2;
       return;
    end;
    SchurComp = X .* Zinv;
    SchurComp = 0.5*(SchurComp + SchurComp');
%
% and compute its Choleski factorization
%
    [Cholfac,indef] = chol(SchurComp);
    if indef > 0,
       if prtlevel > 0,
          fprintf('fdsdp: stop since limiting accuracy reached.\n');
          fprintf('      (Schur complement is numerically indefinite)\n');
       end;
       termflag = 3;
       iter = iter - 1;
       return;
    end;
    minpiv = min(abs(diag(Cholfac)));
    if minpiv == 0,
       if prtlevel > 0,
          fprintf('fdsdp: stop since limiting accuracy reached.\n');
          fprintf('      (Schur complement is numerically singular)\n');
       end;
       termflag = 3;
       iter = iter-1;
       return;
    end;
%
% predictor step
%
    Rc = - X*Z;
%
    [dX,dy,dZ] =...
       dsdpsol(X,Zinv,Rd,rp,Rc,Cholfac);
%
% symmetrize dX : necessary
% note that there is no need to symmetrize dZ since
%    dZ = Rd - diag(dy)
% and so is guaranteed to be symmetric
%
    dX = 0.5*(dX + dX');
%
% compute new gap
%
    Xstep = min(1,tau*sdbound(chol_X,dX,blk));  % tau times step
    Zstep = min(1,tau*sdbound(chol_Z,dZ,blk));  % to boundary
    Xnew = X + Xstep*dX;
    Znew = Z + Zstep*dZ;
    gapnew = full(svec(Xnew,blk)'*svec(Znew,blk));
%
% corrector step
%
    mu = (gapnew/dgap)^2 * gapnew/n;   % Mehrotra's formula
    Rc = Rc + mu*Id - dX*dZ;           % included mu term this time
%
% Compute the corrector step
%
    [dX,dy,dZ] =...
       dsdpsol(X,Zinv,Rd,rp,Rc,Cholfac);
%
% symmetrize dX
%
    dX = 0.5*(dX + dX');
%
% and compute the stepsizes
%
    Xstep = min(1,tau*sdbound(chol_X,dX,blk));
    Zstep = min(1,tau*sdbound(chol_Z,dZ,blk));
%
% failure if steps get too short
%
    if Xstep < steptol | Zstep < steptol,
       if prtlevel > 0,
          fprintf('fdsdp: stop since steps are too short\n');
       end;
       termflag = 5;
       iter = iter-1;
       break;
    end;
%
% otherwise decide whether new point provides "substantial" improvement
% over previous iterate
%
    Xnew = X + Xstep*dX;
    Znew = Z + Zstep*dZ;
    ynew = y + Zstep*dy;        % end of corrector step and iteration
%
% check for positive definiteness of Xnew
%
    [chol_X,indef] =  chol(Xnew);  % if Xnew is indefinite, we quit
    if indef > 0,                  % and return the previous iterate
       termflag = 1;
       if prtlevel > 0,
          fprintf('fdsdp: stop since limiting accuracy reached.\n');
          fprintf('      (new X is indefinite, hence rejected)\n');
       end;
       iter = iter - 1;
       return;
    end;
%
% check for positive definiteness of Znew
%
    [chol_Z, indef] = chol(Znew);  % if Znew is indefinite, we quit
    if indef > 0,                  % and return the previous iterate
       termflag = 2;
       if prtlevel > 0,
          fprintf('fdsdp: stop since limiting accuracy reached.\n');
          fprintf('      (new Z is indefinite, hence rejected)\n');
       end;
       iter = iter - 1;
       return;
    end;
%
% decide whether to accept new point or quit
%
    vXnew = svec(Xnew,blk);
    vZnew = svec(Znew,blk);
    gapnew = full(vXnew'*vZnew);
    rpnew = b - diag(Xnew);
    Rdnew = C - Znew - spdiags(ynew,0,speye(n));
    pri_infeas_new = norm(rpnew);
    dual_infeas_new = norm(svec(Rdnew,blk));
%
    if gapnew < abstol & gapprogtol*gapnew > dgap & ...
       (pri_infeas_new > feasprogtol*pri_infeas | ...
        dual_infeas_new > feasprogtol*dual_infeas),
%
% reject the new iterate
%
       termflag = 4;
       iter = iter-1;
       if prtlevel > 0,
          fprintf('Stop since new point is substantially worse than current iterate\n');
          fprintf('      X.Z = %11.3e\n',gapnew);
          fprintf('      pri_infeas = %11.3e\n', pri_infeas_new);
          fprintf('      dual_infeas = %11.3e\n', dual_infeas_new);
          break;
       end;
    else,
%
% accept new point
%
       X =Xnew;
       y = ynew;
       Z = Znew;
       vX = vXnew;
       vZ = vZnew;
       dgap = gapnew;
       rp = rpnew;
       Rd = Rdnew;
       pri_infeas = pri_infeas_new;
       dual_infeas = dual_infeas_new;
%
% compute objective values so they can be plotted later
%
       gapval(iter+1) = dgap;
       objval(iter+1,1) = vC'*vX;
       objval(iter+1,2) = b'*y;
       feasval(iter+1,1) = pri_infeas;
       feasval(iter+1,2) = dual_infeas;
%
% and display information
%
       if prtlevel > 0,
          fprintf('%3.0f %11.3e %11.3e', iter, Xstep, Zstep);
          fprintf(' %11.3e %11.3e %11.3e', pri_infeas, dual_infeas, dgap);
          fprintf(' %11.3e %11.3e\n',objval(iter+1,1),objval(iter+1,2));
       end;
    end;
%
% termination test
%
    total_err = dgap + pri_infeas + dual_infeas;
    normX = norm(vX);
    normZ = norm(vZ);
    if total_err < min(abstol,reltol*(normX + normZ)),
       if prtlevel > 0,
          fprintf('fdsdp: stop since error reduced to desired value\n');
       end;
       termflag = 0;
       break;
    elseif normX > bndtol,
       if prtlevel > 0,
          fprintf('fdsdp: stop since X is blowing up\n');
       end;
       termflag = -2;
       return;
    elseif normZ > bndtol,
       if prtlevel > 0,
          fprintf('fdsdp: stop since Z is blowing up\n');
       end;
       termflag = -1;
       return;
    end;
 end;
%
% END function
