function failed = export(fname,A,b,C,blk)
% EXPORT  exports SDP data from Matlab to an ASCII file using a
%         variant of Nemirovskii's compact format
%
% failed = export(fname,A,b,C,blk)
%
%   Note: write one value per line so file can be loaded using
%   Matlab's 'load' command.
%
%   Inputs:
%     fname:        the name of the output file
%     A:            the matrix of constraints, A1,...,Am
%     b:            the vector defining the dual objective
%     C:            the matrix defining the primal objective
%     blk:          the block structure
%
%   The storage scheme is a follows:
%   -  m = # of constraints
%   -  nblk = # of blocks
%   -  sparseblocks
%   -  blk
%   -  b
%   ================ the matrix C: loop over the blocks
%   -  nz = # of nonzeros in block
%   -  <row#> <col#> <value>  of entry if sparseblocks
%   -  <value> for all entries in upper triangle ordered
%              rowise, otherwise
%   ================ loop over k = 1:m
%   ================ the matrix Ak using same scheme as for C
%   -  # of nonzeros in block
%   -  <row#> <col#> <value>  of entry if sparseblocks
%   -  <value> for all entries in upper triangle ordered
%              rowise, otherwise
%   ================
%   - etc...
%
% input variables:
%     - fname     the name of the output file
%     - A         matrix of primal constraints
%     - b         rhs of primal constraints
%     - C         cost matrix
%     - blk       block structure vector
%
% output variables:
%     - failed    flag: 0 = success, 1 = failure

% SDPPACK Version 0.8 BETA
% Copyright (c) 1997 by
% F. Alizadeh, J.-P. Haeberly, M. Nayakkankuppam, M.L. Overton
% Last modified: 3/24/97
%
 if nargin ~= 5,
    fprintf('\nexport: not enough arguments! Aborting...\n');
    failed = 1;
    return;
 end;

 if issparse(A),
    sparseblocks = 1;
 else,
    sparseblocks = 0;
 end;

 fp = fopen(fname,'wt');
 if fp == -1,
    failed = 1;
    return;
 else,
    failed = 0;
 end;
 m = length(b);
 nblk = length(blk);
 fprintf(fp,'%d\n',m);            % m = # constraints
 fprintf(fp,'%d\n',nblk);         % nblk = # blocks
 fprintf(fp,'%d\n',sparseblocks); % are blocks sparse?
 fprintf(fp,'%d\n',blk);          % write block structure
 fprintf(fp,'%20.16e\n',b);       % write the vector b
%
% make lookup table of starting points of blocks
%
start(1) = 0;
 for k = 2:length(blk)
    start(k) = start(k-1) + blk(k-1);
 end;
%
% write the matrix C
%
 for blkidx = 1:nblk,
    bsize = blk(blkidx);
    ulc = start(blkidx);
    lrc = ulc + bsize;
    ulc = ulc+1;
    tmpmat = C(ulc:lrc,ulc:lrc);
    if sparseblocks == 1,
%
% find row/columns indices and values of nonzero entries of block
% note that the entries are ordered columnwise
%
       [I J V] = find(tmpmat);
       nz = 0;               % compute the # of nonzero entries
       for k = 1:length(I),  % in the upper triangle of the block
          if J(k) >= I(k),
             nz = nz+1;
          end;
       end;
       fprintf(fp,'%d\n',nz);
       for k = 1:length(I),
          i = I(k);
          j = J(k);
          if j <= i,   % save the lower triangle only
             fprintf(fp,'%d\n',i);    % row index
             fprintf(fp,'%d\n',j);    % column index
             fprintf(fp,'%20.16e\n',V(k));  % value
          end;
       end;   % k loop
    else,   % matrix is full
       for i = 1:bsize,
          for j = i:bsize
             fprintf(fp,'%20.16e\n',tmpmat(i,j));
          end;
       end;
    end;  % if sparseblocks
 end;   % blkidx loop
%
% write the matrices A1,...,Am
%
 for cstridx = 1:m,
    Ak = smat(A(cstridx,:),blk);
    for blkidx = 1:nblk,
       bsize = blk(blkidx);
       ulc = start(blkidx);
       lrc = ulc + bsize;
       ulc = ulc+1;
       tmpmat = Ak(ulc:lrc,ulc:lrc);
       if sparseblocks == 1,
          [I J V] = find(tmpmat);
          nz = 0;               % compute the # of nonzero entries
          for k = 1:length(I),  % in the lower triangle of the block
             if J(k) >= I(k),
                nz = nz+1;
             end;
          end;
          fprintf(fp,'%d\n',nz);
          for k = 1:length(I),
             i = I(k);
             j = J(k);
             if j <= i,   % save the lower triangle only
                fprintf(fp,'%d\n',i);    % row index
                fprintf(fp,'%d\n',j);    % column index
                fprintf(fp,'%20.16e\n',V(k));  % value
             end;
          end;   % k loop
       else,   % matrix is full
          for i = 1:bsize,
             for j = i:bsize,
                fprintf(fp,'%20.16e\n',tmpmat(i,j));
             end;
          end;
       end;  % if sparseblocks
    end;   % blkidx loop
 end;   %cstridx loop
%
 fp = fclose(fp);
 if fp == -1,
    failed = 1;
 end;
%
% END function
