function [lambda_discrep,proj_discrep,x_discrep,y_discrep] = eigValProjVecVaryDemo(A0,displaylevel)
%
% eigValProjVecVaryDemo illustrates the perturbation results for a simple
% eigenvalue, its eigenprojector, and its associated right and left 
% eigenvector, using a variety of normalizations, as discussed in the paper 
% by Greenbaum, Li and Overton (give arXiv reference here)
% Michael Overton, 2019, mo1@nyu.edu

% input: 
%   A0, any square complex matrix whose largest eigenvalues in modulus are
%     simple: including nondiagonalizable matrices such as given by
%     A0=triu(randn(10),1)+eye(10); A0(1:10,11)=randn(10,1); A0(11,11)=2;
%     ***OR***, if A0 is a scalar, a randomly generated complex matrix with 
%     this dimension will be generated 
%   displaylevel: 0 for no output, 1 for summaries, 
%                 2 for vector displays (default), 3 for matrix displays
% output: 
%  relative norm discrepancies between derivatives and difference quotients:
%   lambda_discrep: eigenvalue 
%   proj_discrep: eigenprojector
%   x_discrep: right eigenvectors 
%      cell array: components 1..5 correspond to normalizations 0..4
%   y_discrep: left eigenvectors
%      cell array: components 1..5 correspond to normalizations 0..4

if nargin < 2
    displaylevel = 2;
end
nrows = size(A0,1);
ncols = size(A0,2);
if ncols ~= nrows
    error('A0 must be square')
elseif ncols > 1
    n = ncols;
else
    % A0 is a scalar
    n = A0;
    if displaylevel > 0
        fprintf('generating A0 randomly with dimension %d\n', n);
    end
    A0 = randn(n)+1i*randn(n);
end
%
% compute lambda0, a largest eigenvalue in modulus, assumed to be simple,
% and corresponding right and left eigenvectors x0, y0 normalized so that 
% y0'*x0 = 1, along with the eigenprojector Pi0=x0*y0', and matrices 
% X1, Y1, B1, whose existence is guaranteed by Lemma1 and which provide
% the block diagonalization
%            
%              Y'*A0*X = diag(lambda0, B1)
%
% where X=[x X1], Y=[y Y1]. Note that X1, Y1 are not generally
% eigenvectors, and cannot be if A0 is not diagonalizable.
%
[lambda0,x0,y0,Pi0,X1,Y1] = applyLemma1(A0);
if displaylevel > 2
    fprintf('The computations in applyLemma1 resulted in this block diagonalization\n')
    [y0 Y1]'*A0*[x0 X1]
end
% 
% generate complex perturbation matrix randomly
%
dA = randn(n)+1i*randn(n); % not supposed to be small
%
% check the formulas for the derivative of the eigenvalue and the
% eigenprojector against the difference quotients using a fixed
% perturbation scaling h (which could be changed or varied). Note that h
% should not be too small, because the smaller it is, the more rounding 
% error will dominate the computation. The best choice depends on the
% conditioning of the problem. Here we choose h to be approximately the
% square root of the machine precision.  The second argument lambda0 is 
% provided so that lambdapert can be chosen as the eigenvalue of the 
% perturbed matrix that is closest to lambda0.
%
h = 1e-8;
[lambdapert,Pipert,xpert,ypert] = getEigValProjVec(A0+h*dA,lambda0);
%
% Difference quotient for eigenvalue
%
difQuoLambda = (lambdapert - lambda0)/h;
%
% Derivative of eigenvalue -- see Theorem 1 
%
derivLambda = y0'*dA*x0;
%
% Compare these, relatively
%
lambda_discrep = abs(derivLambda - difQuoLambda)/abs(derivLambda);
if displaylevel > 0
    fprintf('\nEIGENVALUE\n')
    fprintf('                    derivative                                     difference quotient\n')
    displayVecs(derivLambda,difQuoLambda); % these are scalars, that's OK
    fprintf('relative discrepancy:        %e\n',lambda_discrep);
end
%
% Next the eigenprojector: compute the relative discrepancy between
% the derivative and the difference quotient, normwise
%
difQuoProj = (Pipert - Pi0)/h;
%
% Derivative of eigenprojector -- see Theorem 3 
%
B1 = Y1'*A0*X1;
I = eye(n-1);
S = X1*((B1-lambda0*I)\(Y1')); % group inverse (reduced resolvent)
derivProj = -Pi0*dA*S - S*dA*Pi0;
proj_discrep = norm(derivProj - difQuoProj)/norm(derivProj);
if displaylevel > 0
    fprintf('\nEIGENPROJECTOR\n')
end
if displaylevel > 2
    derivProj
    difQuoProj
end
if displaylevel > 0
    fprintf('relative discrepancy norm:        %e\n',proj_discrep);
end
%
% Now the eigenvectors. Begin with the derivatives of the analytic
% x(t),y(t) given in Theorem 2 of the paper.
%
dx = X1*((lambda0*I - B1)\(Y1'*dA*x0));
dy = Y1*((lambda0*I - B1)'\(X1'*dA'*y0));
%
% run through the eigenvector normalizations
% normalization 0 refers to the analytic family in Section 3.3 (eq. (25)) of the paper
% normalizations 1 to 4 are defined in Section 3.4 of the paper.
% However, we don't show results for normalization 4 because it is not well
% defined and formulas for the derivatives in this case don't make sense.
%
for normalization = 0:3
    if displaylevel > 0
        fprintf('\nEIGENVECTOR NORMALIZATION %d\n  ', normalization)
    end
    [x_discrep{normalization+1},y_discrep{normalization+1}] = ...
        eigVecVary(x0,y0,xpert,ypert,h,dx,dy,normalization,displaylevel);
    if displaylevel > 0
        if normalization < 3
            fprintf('press any key to continue\n\n\n')
            pause
        end
        if normalization == 4 % but this is disabled: the loop only goes to 3
            fprintf('large eigenvector discrepancy is expected for normalization 4 as it is not unique\n')
        end
    end
end