/*****************************************************************************/
/*!
 * \file translator.cpp
 * \brief Description: Code for translation between languages
 * 
 * Author: Clark Barrett
 * 
 * Created: Sat Jun 25 18:06:52 2005
 *
 * <hr>
 * Copyright (C) 2003 by the Board of Trustees of Leland Stanford
 * Junior University and by New York University. 
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation for any purpose is hereby granted without
 * royalty, subject to the terms and conditions defined in the \ref
 * LICENSE file provided with this distribution.  In particular:
 *
 * - The above copyright notice and this permission notice must appear
 * in all copies of the software and related documentation.
 *
 * - THE SOFTWARE IS PROVIDED "AS-IS", WITHOUT ANY WARRANTIES,
 * EXPRESSED OR IMPLIED.  USE IT AT YOUR OWN RISK.
 * 
 * <hr>
 * 
 */
/*****************************************************************************/


#include "translator.h"
#include "expr.h"
#include "theory_core.h"
#include "theory_uf.h"
#include "theory_arith.h"
#include "theory_bitvector.h"
#include "theory_array.h"
#include "theory_quant.h"
#include "theory_records.h"
#include "theory_simulate.h"
#include "theory_datatype.h"
#include "theory_datatype_lazy.h"
#include "smtlib_exception.h"


using namespace std;
using namespace CVCL;


string Translator::fileToSMTLIBIdentifier(const string& filename)
{
  string tmpName;
  string::size_type pos = filename.rfind("/");
  if (pos == string::npos) {
    tmpName = filename;
  }
  else {
    tmpName = filename.substr(pos+1);
  }
  char c = tmpName[0];
  string res;
  if ((c < 'A' || c > 'Z') && (c < 'a' || c > 'z')) {
    res = "B_";
  }
  for (unsigned i = 0; i < tmpName.length(); ++i) {
    c = tmpName[i];
    if ((c < 'A' || c > 'Z') && (c < 'a' || c > 'z') &&
        (c < '0' || c > '9') && (c != '.' && c != '_')) {
      c = '_';
    }
    res += c;
  }
  return res;
}


Expr Translator::preprocessRec(const Expr& e, ExprMap<Expr>& cache, Type desiredType)
{
  ExprMap<Expr>::iterator i(cache.find(e));
  if(i != cache.end()) {
    if (!d_auflira || desiredType.isNull() || (*i).second.getType() == desiredType)
      return (*i).second;
  }

  if (e.isClosure()) {
    Expr newBody = preprocessRec(e.getBody(), cache, Type());
    Expr e2(e);
    if (newBody != e.getBody()) {
      e2 = d_em->newClosureExpr(e.getKind(), e.getVars(), newBody);
    }
    cache[e] = e2;
    return e2;
  }

  bool forceReal = false;
  if (d_auflira) {
    if (!d_real2int &&
        desiredType == d_theoryArith->intType() &&
        e.getType() != d_theoryArith->intType()) {
      
      throw SmtlibException("INT expected, but instead we got "+
                            e.getType().getExpr().toString()+
                            "\n\nwhile processing the term:\n"+
                            e.toString(PRESENTATION_LANG));
    }

    // Try to force type-compliance
    switch (e.getKind()) {
      case EQ:
      case LT:
      case GT:
      case LE:
      case GE:
        if (!d_real2int && e[0].getType() != e[1].getType()) {
          if (e[0].getType() != d_theoryArith->intType() &&
              e[1].getType() != d_theoryArith->intType()) {
            throw SmtlibException("Expected both sides to be REAL or INT, but we got lhs: "+
                                  e[0].getType().getExpr().toString()+" and rhs: "+
                                  e[1].getType().getExpr().toString()+
                                  "\n\nwhile processing the term:\n"+
                                  e.toString(PRESENTATION_LANG));
          }
          forceReal = true;
          break;
        }
      case UMINUS:
      case PLUS:
      case MINUS:
      case MULT:
        if (!d_real2int && desiredType == d_theoryArith->realType())
          forceReal = true;
        break;
      case DIVIDE:
        if (d_real2int) {
          throw SmtlibException("DIVIDE not allowed if real2int is enabled");
        }
        forceReal = true;
        break;
      default:
        break;
    }
  }

  Expr e2(e);
  vector<Expr> children;
  bool diff = false;

  Type funType;
  if (e.isApply()) {
    funType = e.getOpExpr().getType();
  }

  for(int k = 0; k < e.arity(); ++k) {
    Type t;
    if (forceReal) t = d_theoryArith->realType();
    else if (e.isApply()) t = funType[k];
    // Recursively preprocess the kids
    Expr child = preprocessRec(e[k], cache, t);
    if (child != e[k]) diff = true;
    children.push_back(child);
  }
  if (diff) {
    e2 = Expr(e.getOp(), children);
  }
  else if (!d_real2int && e2.getKind() == RATIONAL_EXPR) {
    if (e2.getType() == d_theoryArith->realType() ||
        (e2.getType() == d_theoryArith->intType() &&
         desiredType == d_theoryArith->realType()))
      e2 = Expr(REAL_CONST, e2);
  }
  if (!d_real2int && !desiredType.isNull() && e2.getType() != desiredType) {
    throw SmtlibException("Type error: expected "+
                          desiredType.getExpr().toString()+
                          " but instead got "+
                          e2.getType().getExpr().toString()+
                          "\n\nwhile processing term:\n"+
                          e.toString(PRESENTATION_LANG));
  }

  switch (e2.getKind()) {
    case EQ:
      if (d_theoryArith->getBaseType(e[0]) != d_theoryArith->realType())
        break;
    case LT:
    case GT:
    case LE:
    case GE:
    case UMINUS:
    case PLUS:
    case MINUS:
    case MULT:
    case DIVIDE:
    case POW:
    case INTDIV:
    case MOD:

      if (d_iteLiftArith) {
        diff = false;
        children.clear();
        vector<Expr> children2;
        Expr cond;
        for (int k = 0; k < e2.arity(); ++k) {
          if (e2[k].getKind() == ITE && !diff) {
            diff = true;
            cond = e2[k][0];
            children.push_back(e2[k][1]);
            children2.push_back(e2[k][2]);
          }
          else {
            children.push_back(e2[k]);
            children2.push_back(e2[k]);
          }
        }
        if (diff) {
          Expr thenpart = Expr(e2.getOp(), children);
          Expr elsepart = Expr(e2.getOp(), children2);
          e2 = cond.iteExpr(thenpart, elsepart);
          e2 = preprocessRec(e2, cache, desiredType);
          cache[e] = e2;
          return e2;
        }
      }

      if (d_convertToDiff != "" && d_theoryArith->isAtomicArithFormula(e2)) {
        e2 = d_theoryArith->rewriteToDiff(e2);
        cache[e] = e2;
        return e2;
      }

      break;
    default:
      break;
  }
  cache[e] = e2;
  return e2;
}


Expr Translator::preprocess(const Expr& e)
{
  ExprMap<Expr> cache;
  Expr result;
  try {
    result = preprocessRec(e, cache, Type());
  } catch (SmtlibException& ex) {
    cerr << "Error while processing the formula:\n"
         << e.toString(PRESENTATION_LANG) << endl;
    throw ex;
  }
  return result;
}


Translator::Translator(ExprManager* em,
                       const bool& translate,
                       const bool& real2int,
                       const string& convertToDiff,
                       bool iteLiftArith)
  : d_em(em), d_translate(translate),
    d_real2int(real2int),
    d_convertToDiff(convertToDiff),
    d_iteLiftArith(iteLiftArith),
    d_dump(false), d_dumpFileOpen(false), d_translateFileOpen(false),
    d_auflia(false), d_auflira(false), d_unknown(false)
{}


bool Translator::start(const string& dumpFile)
{
  if (d_translate && d_em->getOutputLang() == SMTLIB_LANG) {
    d_osdumpFile.open(".cvcl__smtlib_temporary_file");
    if (!d_osdumpFile)
      throw Exception("cannot open temp file .cvcl__smtlib_temporary_file");
    else {
      d_dump = true;
      d_dumpFileOpen = true;
      d_osdump = &d_osdumpFile;
    }

    if (dumpFile == "") {
      d_osdumpTranslate = &cout;
    }
    else {
      d_osdumpTranslateFile.open(dumpFile.c_str());
      if(!d_osdumpTranslateFile)
        throw Exception("cannot open a log file: "+dumpFile);
      else {
        d_translateFileOpen = true;
        d_osdumpTranslate = &d_osdumpTranslateFile;
      }
    }
    *d_osdumpTranslate <<
      "(benchmark " << fileToSMTLIBIdentifier(dumpFile) << endl <<
      "  :source {" << endl;
    string tmpName;
    string::size_type pos = dumpFile.rfind("/");
    if (pos == string::npos) {
      tmpName = "README";
    }
    else {
      tmpName = dumpFile.substr(0,pos+1) + "README";
    }
    d_tmpFile.open(tmpName.c_str());
    char c;
    if (d_tmpFile.is_open()) {
      d_tmpFile.get(c);
      while (!d_tmpFile.eof()) {
        if (c == '{' || c == '}') *d_osdumpTranslate << '\\';
        *d_osdumpTranslate << c;
        d_tmpFile.get(c);
      }
      d_tmpFile.close();
    }
    else {
      *d_osdumpTranslate << "Source unknown";
    }
    *d_osdumpTranslate << endl;// <<
    //        "This benchmark was automatically translated into SMT-LIB format from" << endl <<
    //        "CVC format using CVC Lite" << endl;
    *d_osdumpTranslate << "}" << endl;

    d_zeroVar = new Expr();
    if (d_convertToDiff == "int") {
      dump(Expr(CONST, Expr(ID, d_em->newStringExpr("cvclZero")),
           d_theoryArith->intType().getExpr()));
      *d_zeroVar = d_theoryCore->newVar("cvclZero", d_theoryArith->intType());
    }
    else if (d_convertToDiff == "real") {
      dump(Expr(CONST, Expr(ID, d_em->newStringExpr("cvclZero")),
           d_theoryArith->realType().getExpr()));
      *d_zeroVar = d_theoryCore->newVar("cvclZero", d_theoryArith->realType());
    }

  }
  else {
    if (dumpFile == "") {
      if (d_translate) {
        d_osdump = &cout;
        d_dump = true;
      }
    }
    else {
      d_osdumpFile.open(dumpFile.c_str());
      if(!d_osdumpFile)
        throw Exception("cannot open a log file: "+dumpFile);
      else {
        d_dump = true;
        d_dumpFileOpen = true;
        d_osdump = &d_osdumpFile;
      }
    }
  }
  return d_dump;
}


void Translator::dump(const Expr& e, bool dumpOnly)
{
  DebugAssert(d_dump, "dump called unexpectedly");
  if (!dumpOnly || !d_translate) {
    *d_osdump << e << endl;
  }
}


void Translator::dumpAssertion(const Expr& e)
{
  if (d_translate && d_em->getOutputLang() == SMTLIB_LANG) {
    *d_osdump << "  :assumption" << endl;
    Expr e2 = preprocess(e);
    e2.getType();
    *d_osdump << e2 << endl;
  }
  else {
    *d_osdump << Expr(ASSERT, e) << endl;
  }
}


bool Translator::dumpQuery(const Expr& e)
{
  if (d_translate && d_em->getOutputLang() == SMTLIB_LANG) {
    *d_osdump << "  :formula" << endl;
    Expr e2 = preprocess(e.negate());
    e2.getType();
    *d_osdump << e2 << endl;
    // For now, just assume status is unknown.
    *d_osdumpTranslate << "  :status unknown" << endl;
    return true;
  }
  else {
    *d_osdump << Expr(QUERY, e) << endl;
    if (d_translate) return true;
  }
  return false;
}


void Translator::dumpQueryResult(QueryResult qres)
{
  if(d_translate && d_em->getOutputLang() == SMTLIB_LANG) {
    *d_osdumpTranslate << "  :status ";
    switch (qres) {
      case UNSATISFIABLE:
        *d_osdumpTranslate << "unsat" << endl;
        break;
      case SATISFIABLE:
        *d_osdumpTranslate << "sat" << endl;
        break;
      default:
        *d_osdumpTranslate << "unknown" << endl;
        break;
    }
  }
}


void Translator::finish()
{
  if (d_translate) {
    if (d_em->getOutputLang() == SMTLIB_LANG) {
      delete d_zeroVar;
      *d_osdumpTranslate << "  :logic ";
      if (d_unknown ||
          d_theoryRecords->theoryUsed() ||
          d_theorySimulate->theoryUsed() ||
          d_theoryBitvector->theoryUsed() ||
          d_theoryDatatype->theoryUsed()) {
        *d_osdumpTranslate << "unknown";
      }
      else {
        if (d_theoryArith->theoryUsed() &&
            (d_theoryArith->getLangUsed() == NONLINEAR) ||
            (d_theoryArith->realUsed() &&
             d_theoryArith->intUsed() &&
             !d_auflira)) {
          *d_osdumpTranslate << "unknown";
        }
        else {
          if (!d_theoryQuant->theoryUsed()) {
            *d_osdumpTranslate << "QF_";
          }
          if (d_theoryArray->theoryUsed()) {
            if (d_theoryArith->theoryUsed()) {
              if (d_theoryArith->realUsed()) {
                if (d_auflira) {
                  *d_osdumpTranslate << "AUFLIRA";
                }
                else {
                  *d_osdumpTranslate << "AUFLRA";
                }
              }
              else {
                if (d_auflira) {
                  *d_osdumpTranslate << "AAUFLIA";
                }
                else {
                  *d_osdumpTranslate << "AUFLIA";
                }
              }
            }
            else {
              *d_osdumpTranslate << "AUFLIA";
            }
          }
          else if (d_theoryUF->theoryUsed()) {
            if (d_theoryArith->theoryUsed()) {
              if (d_theoryArith->realUsed()) {
                if (d_theoryArith->getLangUsed() <= DIFF_ONLY) {
                  *d_osdumpTranslate << "UFRDL";
                }
                else {
                  *d_osdumpTranslate << "UFLRA";
                }
              }                  
              else {
                if (d_theoryArith->getLangUsed() <= DIFF_ONLY) {
                  *d_osdumpTranslate << "UFIDL";
                }
                else {
                  *d_osdumpTranslate << "UFLIA";
                }
              }
            }
            else {
              *d_osdumpTranslate << "UF";
            }
          }
          else if (d_theoryArith->theoryUsed()) {
            if (d_theoryArith->realUsed()) {
              if (d_theoryArith->getLangUsed() == DIFF_ONLY) {
                *d_osdumpTranslate << "RDL";
              }
              else {
                *d_osdumpTranslate << "LRA";
              }
            }
            else {
              if (d_theoryArith->getLangUsed() == DIFF_ONLY) {
                *d_osdumpTranslate << "IDL";
              }
              else {
                *d_osdumpTranslate << "LIA";
              }
            }
          }
          else {
            *d_osdumpTranslate << "UF";
          }
        }
      }
      *d_osdumpTranslate << endl;
      d_tmpFile.clear();
      d_tmpFile.open(".cvcl__smtlib_temporary_file");
      if (d_tmpFile.is_open()) {
        char c;
        d_tmpFile.get(c);
        while (!d_tmpFile.eof()) {
          *d_osdumpTranslate << c;
          d_tmpFile.get(c);
        }
        d_tmpFile.close();
      }
      *d_osdumpTranslate << ")" << endl;
    }
  }

  if (d_dumpFileOpen) d_osdumpFile.close();
  if (d_translateFileOpen) d_osdumpTranslateFile.close();
}


void Translator::translateArray(ExprStream& os, const Expr& e)
{
  if (isInt(Type(e[0]))) {
    if (isInt(Type(e[1]))) {
      if (!d_auflira) {
        d_auflia = true;
        os << "Array";
        return;
      }
    }
    else if (isReal(Type(e[1]))) {
      if (!d_auflia) {
        d_auflira = true;
        os << "Array1";
        return;
      }
    }
    else if (isArray(Type(e[1])) &&
             isInt(Type(e[1][0])) &&
             isReal(Type(e[1][1]))) {
      if (!d_auflia) {
        d_auflira = true;
        os << "Array2";
        return;
      }
    }
  }
  os << "Array";
  d_unknown = true;
}


Expr Translator::zeroVar()
{
  return *d_zeroVar;
}
