%{
/*****************************************************************************/
/*!
 * \file smtlib.y
 * 
 * Author: Sergey Berezin, Clark Barrett
 * 
 * Created: Apr 30 2005
 *
 * <hr>
 * Copyright (C) 2004 by the Board of Trustees of Leland Stanford
 * Junior University and by New York University. 
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation for any purpose is hereby granted without
 * royalty, subject to the terms and conditions defined in the \ref
 * LICENSE file provided with this distribution.  In particular:
 *
 * - The above copyright notice and this permission notice must appear
 * in all copies of the software and related documentation.
 *
 * - THE SOFTWARE IS PROVIDED "AS-IS", WITHOUT ANY WARRANTIES,
 * EXPRESSED OR IMPLIED.  USE IT AT YOUR OWN RISK.
 * 
 * <hr>
 * 
 */
/*****************************************************************************/
/* 
   This file contains the bison code for the parser that reads in CVC
   commands in SMT-LIB language.
*/

#include "vc.h"
#include "parser_exception.h"
#include "parser_temp.h"

// Exported shared data
namespace CVCL {
  extern ParserTemp* parserTemp;
}
// Define shortcuts for various things
#define TMP CVCL::parserTemp
#define EXPR CVCL::parserTemp->expr
#define VC (CVCL::parserTemp->vc)
#define ARRAYSENABLED (CVCL::parserTemp->arrFlag)
#define RAT(args) CVCL::newRational args

// Suppress the bogus warning suppression in bison (it generates
// compile error)
#undef __GNUC_MINOR__

/* stuff that lives in smtlib.lex */
extern int smtliblex(void);

int smtliberror(char *s)
{
  std::ostringstream ss;
  ss << CVCL::parserTemp->fileName << ":" << CVCL::parserTemp->lineNum
     << ": " << s;
  return CVCL::parserTemp->error(ss.str());
}

#define YYLTYPE_IS_TRIVIAL 1
#define YYMAXDEPTH 10485760

%}

%union {
  std::string *str;
  std::vector<std::string> *strvec;
  CVCL::Expr *node;
  std::vector<CVCL::Expr> *vec;
};


%start cmd

/* strings are for better error messages.  
   "_TOK" is so macros don't conflict with kind names */

%type <vec> bench_attributes sort_symbs fun_symb_decls pred_symb_decls
%type <vec> an_formulas quant_vars an_terms
%type <node> benchmark bench_name bench_attribute
%type <node> status fun_symb_decl fun_sig pred_symb_decl pred_sig
%type <node> an_formula quant_var an_atom prop_atom
%type <node> an_term basic_term sort_symb pred_symb fun_symb
%type <node> var fvar
%type <str> logic_name quant_symb connective user_value attribute annotation
%type <strvec> annotations

%token <str> NUMERAL_TOK
%token <str> SYM_TOK
%token <str> STRING_TOK
%token <str> AR_SYMB
%token <str> USER_VAL_TOK
%token TRUE_TOK
%token FALSE_TOK
%token ITE_TOK
%token NOT_TOK
%token IMPLIES_TOK
%token IF_THEN_ELSE_TOK
%token AND_TOK
%token OR_TOK
%token XOR_TOK
%token IFF_TOK
%token EXISTS_TOK
%token FORALL_TOK
%token LET_TOK
%token FLET_TOK
%token NOTES_TOK
%token CVC_COMMAND_TOK
%token SORTS_TOK
%token FUNS_TOK
%token PREDS_TOK
%token EXTENSIONS_TOK
%token DEFINITION_TOK
%token AXIOMS_TOK
%token LOGIC_TOK
%token COLON_TOK
%token LPAREN_TOK
%token RPAREN_TOK
%token SAT_TOK
%token UNSAT_TOK
%token UNKNOWN_TOK
%token ASSUMPTION_TOK
%token FORMULA_TOK
%token STATUS_TOK
%token BENCHMARK_TOK
%token EXTRASORTS_TOK
%token EXTRAFUNS_TOK
%token EXTRAPREDS_TOK
%token LANGUAGE_TOK
%token DOLLAR_TOK
%token QUESTION_TOK
%token DISTINCT_TOK
%token SEMICOLON_TOK
%token EOF_TOK

%%

cmd:
    benchmark
    {
      EXPR = *$1;
      delete $1;
      YYACCEPT;
    }
;

benchmark:
    LPAREN_TOK BENCHMARK_TOK bench_name bench_attributes RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr("SEQ",*$4));
      delete $4;
    }
  | EOF_TOK
    { 
      TMP->done = true;
      $$ = new CVCL::Expr();
    }
;

bench_name:
    SYM_TOK
    {
      $$ = NULL;
      delete $1;
    }
;

bench_attributes:
    bench_attribute
    {
      $$ = new std::vector<CVCL::Expr>;
      if ($1) {
	$$->push_back(*$1);
	delete $1;
      }
    }
  | bench_attributes bench_attribute
    {
      $$ = $1;
      if ($2) {
	$$->push_back(*$2);
	delete $2;
      }
    }
;

bench_attribute:
    COLON_TOK ASSUMPTION_TOK an_formula
    {
      $$ = new CVCL::Expr(VC->listExpr("ASSERT", *$3));
      delete $3;
    }
  | COLON_TOK FORMULA_TOK an_formula 
    {
      $$ = new CVCL::Expr(VC->listExpr("CHECKSAT", *$3));
      delete $3;
    }
  | COLON_TOK STATUS_TOK status 
    {
      $$ = NULL;
    }
  | COLON_TOK LOGIC_TOK logic_name 
    {
      ARRAYSENABLED = false;
      if (*$3 == "QF_UF") {
        $$ = new CVCL::Expr(VC->listExpr("TYPE", VC->idExpr("U")));
      }
      else if (*$3 == "QF_A" ||
               *$3 == "QF_AX") {
        ARRAYSENABLED = true;
        std::vector<CVCL::Expr> tmp;
        tmp.push_back(VC->listExpr("TYPE", VC->idExpr("Index")));
        tmp.push_back(VC->listExpr("TYPE", VC->idExpr("Element")));
        tmp.push_back(VC->listExpr("TYPEDEF", VC->idExpr("Array"),
                                   VC->listExpr("ARRAY",
                                                VC->idExpr("Index"),
                                                VC->idExpr("Element"))));
        $$ = new CVCL::Expr(VC->listExpr("SEQ", tmp));
      }
      else if (*$3 == "QF_AUFLIA" ||
               *$3 == "AUFLIA") {
        ARRAYSENABLED = true;
        $$ = new CVCL::Expr(VC->listExpr("TYPEDEF", VC->idExpr("Array"),
                                         VC->listExpr("ARRAY",
                                                      VC->idExpr("INT"),
                                                      VC->idExpr("INT"))));
      }
      else {
        $$ = NULL;
      }
    }
  | COLON_TOK EXTRASORTS_TOK LPAREN_TOK sort_symbs RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr("TYPE", *$4));
      delete $4;
    }
  | COLON_TOK EXTRAFUNS_TOK LPAREN_TOK fun_symb_decls RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr("SEQ", *$4));
      delete $4;
    }
  | COLON_TOK EXTRAPREDS_TOK LPAREN_TOK pred_symb_decls RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr("SEQ", *$4));
      delete $4;
    }
  | COLON_TOK NOTES_TOK STRING_TOK
    {
      $$ = NULL;
      delete $3;
    }
  | COLON_TOK CVC_COMMAND_TOK STRING_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr(VC->idExpr(*$3)));
      delete $3;
    }
  | annotation
    {
      $$ = NULL;
      delete $1;
    }
;

logic_name:
    SYM_TOK
    {
      $$ = $1;
    }
;

status:
    SAT_TOK { $$ = NULL; }
  | UNSAT_TOK { $$ = NULL; }
  | UNKNOWN_TOK { $$ = NULL; }
;

sort_symbs:
    sort_symb 
    {
      $$ = new std::vector<CVCL::Expr>;
      $$->push_back(*$1);
      delete $1;
    }
  | sort_symbs sort_symb
    { 
      $1->push_back(*$2);
      $$ = $1;
      delete $2;
    }
;

fun_symb_decls:
    fun_symb_decl
    {
      $$ = new std::vector<CVCL::Expr>;
      $$->push_back(*$1);
      delete $1;
    }
  |
    fun_symb_decls fun_symb_decl
    {
      $1->push_back(*$2);
      $$ = $1;
      delete $2;
    }
;

fun_symb_decl:
    LPAREN_TOK fun_sig annotations RPAREN_TOK
    {
      $$ = $2;
      delete $3;
    }
  | LPAREN_TOK fun_sig RPAREN_TOK
    {
      $$ = $2;
    }
;

fun_sig:
    fun_symb sort_symbs
    {
      if ($2->size() == 1) {
        $$ = new CVCL::Expr(VC->listExpr("CONST", VC->listExpr(*$1), (*$2)[0]));
      }
      else {
        CVCL::Expr tmp(VC->listExpr("ARROW", *$2));
        $$ = new CVCL::Expr(VC->listExpr("CONST", VC->listExpr(*$1), tmp));
      }
      delete $1;
      delete $2;
    }
;

pred_symb_decls:
    pred_symb_decl
    {
      $$ = new std::vector<CVCL::Expr>;
      $$->push_back(*$1);
      delete $1;
    }
  |
    pred_symb_decls pred_symb_decl
    {
      $1->push_back(*$2);
      $$ = $1;
      delete $2;
    }
;

pred_symb_decl:
    LPAREN_TOK pred_sig annotations RPAREN_TOK
    {
      $$ = $2;
      delete $3;
    }
  | LPAREN_TOK pred_sig RPAREN_TOK
    {
      $$ = $2;
    }
;

pred_sig:
    pred_symb sort_symbs
    {
      std::vector<CVCL::Expr> tmp(*$2);
      tmp.push_back(VC->idExpr("BOOLEAN"));
      CVCL::Expr tmp2(VC->listExpr("ARROW", tmp));
      $$ = new CVCL::Expr(VC->listExpr("CONST", VC->listExpr(*$1), tmp2));
      delete $1;
      delete $2;
    }
  | pred_symb
    {
      $$ = new CVCL::Expr(VC->listExpr("CONST", VC->listExpr(*$1),
                                       VC->idExpr("BOOLEAN")));
      delete $1;
    }
;

an_formulas:
    an_formula
    {
      $$ = new std::vector<CVCL::Expr>;
      $$->push_back(*$1);
      delete $1;
    }
  |
    an_formulas an_formula
    {
      $1->push_back(*$2);
      $$ = $1;
      delete $2;
    }
;

an_formula:
    an_atom
    {
      $$ = $1;
    }
  | LPAREN_TOK connective an_formulas annotations RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr(*$2, *$3));
      delete $2;
      delete $3;
      delete $4;
    }
  | LPAREN_TOK connective an_formulas RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr(*$2, *$3));
      delete $2;
      delete $3;
    }
  | LPAREN_TOK quant_symb quant_vars an_formula annotations RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr(*$2, VC->listExpr(*$3), *$4));
      delete $2;
      delete $3;
      delete $4;
      delete $5;
    }
  | LPAREN_TOK quant_symb quant_vars an_formula RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr(*$2, VC->listExpr(*$3), *$4));
      delete $2;
      delete $3;
      delete $4;
    }
  | LPAREN_TOK LET_TOK LPAREN_TOK var an_term RPAREN_TOK an_formula annotations RPAREN_TOK
    {
      CVCL::Expr e(VC->listExpr(VC->listExpr(*$4, *$5)));
      $$ = new CVCL::Expr(VC->listExpr("LET", e, *$7));
      delete $4;
      delete $5;
      delete $7;
      delete $8;
    }
  | LPAREN_TOK LET_TOK LPAREN_TOK var an_term RPAREN_TOK an_formula RPAREN_TOK
    {
      CVCL::Expr e(VC->listExpr(VC->listExpr(*$4, *$5)));
      $$ = new CVCL::Expr(VC->listExpr("LET", e, *$7));
      delete $4;
      delete $5;
      delete $7;
    }
  | LPAREN_TOK FLET_TOK LPAREN_TOK fvar an_formula RPAREN_TOK an_formula annotations RPAREN_TOK
    {
      CVCL::Expr e(VC->listExpr(VC->listExpr(*$4, *$5)));
      $$ = new CVCL::Expr(VC->listExpr("LET", e, *$7));
      delete $4;
      delete $5;
      delete $7;
      delete $8;
    }
  | LPAREN_TOK FLET_TOK LPAREN_TOK fvar an_formula RPAREN_TOK an_formula RPAREN_TOK
    {
      CVCL::Expr e(VC->listExpr(VC->listExpr(*$4, *$5)));
      $$ = new CVCL::Expr(VC->listExpr("LET", e, *$7));
      delete $4;
      delete $5;
      delete $7;
    }
;

quant_vars:
    quant_var
    {
      $$ = new std::vector<CVCL::Expr>;
      $$->push_back(*$1);
      delete $1;
    }
  | quant_vars quant_var
    {
      $1->push_back(*$2);
      $$ = $1; 
      delete $2;
    }
;

quant_var: 
    LPAREN_TOK var sort_symb RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr(*$2, *$3));
      delete $2;
      delete $3;
    }
;

quant_symb:
    EXISTS_TOK
    {
      $$ = new std::string("EXISTS");
    }
  | FORALL_TOK
    {
      $$ = new std::string("FORALL");
    }
;

connective:
    NOT_TOK
    {
      $$ = new std::string("NOT");
    }
  | IMPLIES_TOK
    {
      $$ = new std::string("IMPLIES");
    }
  | IF_THEN_ELSE_TOK
    {
      $$ = new std::string("IF");
    }
  | AND_TOK
    {
      $$ = new std::string("AND");
    }
  | OR_TOK
    {
      $$ = new std::string("OR");
    }
  | XOR_TOK
    {
      $$ = new std::string("XOR");
    }
  | IFF_TOK
    {
      $$ = new std::string("IFF");
    }
;

an_atom:
    prop_atom 
    {
      $$ = $1;
    }
  | LPAREN_TOK prop_atom annotations RPAREN_TOK 
    {
      $$ = $2;
      delete $3;
    }
  | LPAREN_TOK pred_symb an_terms annotations RPAREN_TOK
    {
      if ($4->size() == 1 && (*$4)[0] == "transclose" &&
          $3->size() == 2) {
        $$ = new CVCL::Expr(VC->listExpr("TRANS_CLOSURE",
                                        *$2, (*$3)[0], (*$3)[1]));
      }
      else {
        std::vector<CVCL::Expr> tmp;
        tmp.push_back(*$2);
        tmp.insert(tmp.end(), $3->begin(), $3->end());
        $$ = new CVCL::Expr(VC->listExpr(tmp));
      }
      delete $2;
      delete $3;
      delete $4;
    }
  | LPAREN_TOK pred_symb an_terms RPAREN_TOK
    {
      std::vector<CVCL::Expr> tmp;
      tmp.push_back(*$2);
      tmp.insert(tmp.end(), $3->begin(), $3->end());
      $$ = new CVCL::Expr(VC->listExpr(tmp));
      delete $2;
      delete $3;
    }
  | LPAREN_TOK DISTINCT_TOK an_terms annotations RPAREN_TOK
    {
      std::vector<CVCL::Expr> tmp;
      for (unsigned i = 0; i < (*$3).size(); ++i) {
	for (unsigned j = i+1; j < (*$3).size(); ++j) {
	  tmp.push_back(VC->listExpr("NEQ", (*$3)[i], (*$3)[j]));
	}
      }
      $$ = new CVCL::Expr(VC->listExpr("AND", tmp));
      delete $3;
      delete $4;
    }
  | LPAREN_TOK DISTINCT_TOK an_terms RPAREN_TOK
    {
      std::vector<CVCL::Expr> tmp;
      for (unsigned i = 0; i < (*$3).size(); ++i) {
	for (unsigned j = i+1; j < (*$3).size(); ++j) {
	  tmp.push_back(VC->listExpr("NEQ", (*$3)[i], (*$3)[j]));
	}
      }
      $$ = new CVCL::Expr(VC->listExpr("AND", tmp));
      delete $3;
    }
;

prop_atom:
    TRUE_TOK
    {
      $$ = new CVCL::Expr(VC->idExpr("TRUE"));
    }
  | FALSE_TOK
    { 
      $$ = new CVCL::Expr(VC->idExpr("FALSE"));
    }
  | fvar
    {
      $$ = $1;
    }
  | pred_symb 
    {
      $$ = $1;
    }
;  

an_terms:
    an_term
    {
      $$ = new std::vector<CVCL::Expr>;
      $$->push_back(*$1);
      delete $1;
    }
  | an_terms an_term
    {
      $1->push_back(*$2);
      $$ = $1; 
      delete $2;
    }
;

an_term:
    basic_term 
    {
      $$ = $1;
    }
  | LPAREN_TOK basic_term annotations RPAREN_TOK 
    {
      $$ = $2;
      delete $3;
    }
  | LPAREN_TOK fun_symb an_terms annotations RPAREN_TOK
    {
      std::vector<CVCL::Expr> tmp;
      tmp.push_back(*$2);
      tmp.insert(tmp.end(), $3->begin(), $3->end());
      $$ = new CVCL::Expr(VC->listExpr(tmp));
      delete $2;
      delete $3;
      delete $4;
    }
  | LPAREN_TOK fun_symb an_terms RPAREN_TOK
    {
      std::vector<CVCL::Expr> tmp;
      tmp.push_back(*$2);
      tmp.insert(tmp.end(), $3->begin(), $3->end());
      $$ = new CVCL::Expr(VC->listExpr(tmp));
      delete $2;
      delete $3;
    }
  | LPAREN_TOK ITE_TOK an_formula an_term an_term annotations RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr("IF", *$3, *$4, *$5));
      delete $3;
      delete $4;
      delete $5;
      delete $6;
    }
  | LPAREN_TOK ITE_TOK an_formula an_term an_term RPAREN_TOK
    {
      $$ = new CVCL::Expr(VC->listExpr("IF", *$3, *$4, *$5));
      delete $3;
      delete $4;
      delete $5;
    }
;

basic_term:
    var
    {
      $$ = $1;
    }
  | fun_symb 
    {
      $$ = $1;
    }
;

annotations:
    annotation
    {
      $$ = new std::vector<std::string>;
      $$->push_back(*$1);
      delete $1;
    }
  | annotations annotation
    {
      $1->push_back(*$2);
      $$ = $1;
      delete $2;
    }
  ;
  
annotation:
    attribute 
    { $$ = $1; }
  | attribute user_value 
    { $$ = $1; }
;

user_value:
    USER_VAL_TOK
    {
      $$ = NULL;
      delete $1;
    }
;

sort_symb:
    SYM_TOK 
    { 
      if (*$1 == "Real") {
	$$ = new CVCL::Expr(VC->idExpr("REAL"));
      } else if (*$1 == "Int") {
	$$ = new CVCL::Expr(VC->idExpr("INT"));
      } else {
	$$ = new CVCL::Expr(VC->idExpr(*$1));
      }
      delete $1;
    }
;

pred_symb:
    SYM_TOK
    {
      $$ = new CVCL::Expr(VC->idExpr(*$1));
      delete $1;
    }
  | AR_SYMB 
    { 
      if ($1->length() == 1) {
	switch ((*$1)[0]) {
	  case '=': $$ = new CVCL::Expr(VC->idExpr("EQ")); break;
	  case '<': $$ = new CVCL::Expr(VC->idExpr("LT")); break;
	  case '>': $$ = new CVCL::Expr(VC->idExpr("GT")); break;
	  default: $$ = new CVCL::Expr(VC->idExpr(*$1)); break;
	}
      }
      else {
	if (*$1 == "<=") {
	  $$ = new CVCL::Expr(VC->idExpr("LE"));
	} else if (*$1 == ">=") {
	  $$ = new CVCL::Expr(VC->idExpr("GE"));
	}
	else $$ = new CVCL::Expr(VC->idExpr(*$1));
      }
      delete $1;
    }
;

fun_symb:
    SYM_TOK
    {
      if (ARRAYSENABLED && *$1 == "select") {
        $$ = new CVCL::Expr(VC->idExpr("READ"));
      }
      else if (ARRAYSENABLED && *$1 == "store") {
        $$ = new CVCL::Expr(VC->idExpr("WRITE"));
      }
      else {
        $$ = new CVCL::Expr(VC->idExpr(*$1));
      }
      delete $1;
    }
  | AR_SYMB 
    { 
      if ($1->length() == 1) {
	switch ((*$1)[0]) {
	case '+': $$ = new CVCL::Expr(VC->idExpr("PLUS")); break;
	case '-': $$ = new CVCL::Expr(VC->idExpr("MINUS")); break;
	case '*': $$ = new CVCL::Expr(VC->idExpr("MULT")); break;
	case '~': $$ = new CVCL::Expr(VC->idExpr("UMINUS")); break;
	case '/': $$ = new CVCL::Expr(VC->idExpr("DIVIDE")); break;
	default: $$ = new CVCL::Expr(VC->idExpr(*$1));
	}
      }
      else $$ = new CVCL::Expr(VC->idExpr(*$1));
      delete $1;
    }
  | NUMERAL_TOK
    {
      $$ = new CVCL::Expr(VC->ratExpr(*$1));
      delete $1;
    }
;

attribute:
    COLON_TOK SYM_TOK
    {
      $$ = $2;
    }
;

var:
    QUESTION_TOK SYM_TOK
    {
      $$ = new CVCL::Expr(VC->idExpr(*$2));
      delete $2;
    }
;

fvar:
    DOLLAR_TOK SYM_TOK
    {
      $$ = new CVCL::Expr(VC->idExpr(*$2));
      delete $2;
    }
;

%%
