/*****************************************************************************/
/*!
 * \file context.cpp
 * 
 * Author: Clark Barrett
 * 
 * Created: Fri Jan 17 14:30:37 2003
 *
 * <hr>
 * Copyright (C) 2003 by the Board of Trustees of Leland Stanford
 * Junior University and by New York University. 
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation for any purpose is hereby granted without
 * royalty, subject to the terms and conditions defined in the \ref
 * LICENSE file provided with this distribution.  In particular:
 *
 * - The above copyright notice and this permission notice must appear
 * in all copies of the software and related documentation.
 *
 * - THE SOFTWARE IS PROVIDED "AS-IS", WITHOUT ANY WARRANTIES,
 * EXPRESSED OR IMPLIED.  USE IT AT YOUR OWN RISK.
 * 
 * <hr>
 * 
 */
/*****************************************************************************/


#include "context.h"


using namespace CVCL;
using namespace std;


///////////////////////////////////////////////////////////////////////////////
// Scope methods                                                             //
///////////////////////////////////////////////////////////////////////////////


void Scope::addToChain(ContextObjChain* obj)
{
//   TRACE("context verbose", "Scope::addToChain(", obj->name(), ")");
  if(d_restoreChain != NULL)
    d_restoreChain->d_restoreChainPrev = &obj->d_restoreChainNext;
  obj->d_restoreChainNext = d_restoreChain;
  obj->d_restoreChainPrev = &d_restoreChain;
  d_restoreChain = obj;
}


Scope::Scope(Context* context, Scope* prevScope)
  : d_context(context), d_prevScope(prevScope), d_restoreChain(NULL)
{
  if (prevScope) d_level = prevScope->level() + 1;
  else d_level = 0;
  IF_DEBUG(d_delayDelete = false);
}


Scope::~Scope()
{

  TRACE_MSG("context verbose", "~Scope() {");
  ContextObjChain* obj = d_restoreChain;
  // Now, delete restore chains queued up for deletion.
  IF_DEBUG(d_delayDelete=true);

  IF_DEBUG(
    if(debugger.trace("memory") && obj != NULL) {
      ostream& os = debugger.getOS();
      int n(0);
      ContextObjChain* tmp = obj;
      while(tmp != NULL) {
	tmp = tmp->d_restoreChainNext;
	n++;
      }
      os << "*** Warning: ~Scope(): found "<< n << " leaked objects "
	 << "in scope " << d_level << ":" << endl;
      if(debugger.flag("memory leaks")) {
	tmp = obj;
	while(tmp != NULL) {
	  os << tmp->name() << "\n";
	  tmp = tmp->d_restoreChainNext;
	}
      }
      os << flush;
    }
  )
  // Now actually delete the chain
  TRACE_MSG("context verbose", "~Scope(): Deleting the chain");
  while (obj != NULL) {
    ContextObjChain* tmp = obj->d_restoreChainNext;
    // When called from ~ContextManager(), the master objects may
    // still point to this scope.  Disconnect them here.
    if(obj->d_master!=NULL && obj->d_master->d_scope == this)
      obj->d_master->d_scope = NULL;
    delete obj;
    obj = tmp;
  }
  TRACE_MSG("context verbose", "~Scope() => }");
}


void Scope::restore(void)
{
  TRACE_MSG("context verbose", "Scope::restore() {");
  d_delayDelete=true;
  while (d_restoreChain != NULL) d_restoreChain = d_restoreChain->restore();
  d_delayDelete=false;
  TRACE_MSG("context verbose", "Scope::restore() }");
}


///////////////////////////////////////////////////////////////////////////////
// ContextObjChain methods                                                   //
///////////////////////////////////////////////////////////////////////////////


ContextObjChain::ContextObjChain(ContextObj* data, ContextObj* master, 
				 ContextObjChain* restore)
  : d_restoreChainNext(NULL), d_restoreChainPrev(NULL),
    d_restore(restore), d_data(data), d_master(master) { }


ContextObjChain* ContextObjChain::restore(void)
{
//   TRACE("context verbose", "ContextObjChain::restore(", name(), ")");
  // Assign 'next' pointer only when the master object is restored,
  // since this may change our next pointer (master may have other
  // context objects removed).
  ContextObjChain* next;
  if(d_master==NULL) { // We are marked for deletion.  Do nothing.
    next = d_restoreChainNext;
  } else if(d_data == NULL) {
    d_master->setNull();
    Scope*& masterScope(d_master->d_scope);
    masterScope = d_master->d_scope->prevScope();
//     DebugAssert(masterScope == masterScope->topScope(),"");
    DebugAssert(d_restore==NULL,"Expected NULL");
    next = d_restoreChainNext;
    masterScope->addToChain(this);
  }
  else {
    d_master->restoreData(d_data); // Data will be deleted in destructor
    d_master->d_scope = d_data->d_scope;
    d_master->d_restore = d_restore;
    next = d_restoreChainNext;
    delete this;
  }
  return next;
}


ContextObjChain::~ContextObjChain()
{
//   TRACE("context verbose", "~ContextObjChain(d_data=", name(), ")");
  if(d_data != NULL) delete d_data;
  // Disconnect the master object if it points to us; this may happen
  // within ~ContextManager() call.
  if(d_master!=NULL && d_master->d_restore == this)
    d_master->d_restore = NULL;
}


IF_DEBUG(
std::string 
ContextObjChain::name() const {
  if(d_master == NULL) return "Unknown";
  return d_master->name();
}
)

///////////////////////////////////////////////////////////////////////////////
// ContextObj methods                                                        //
///////////////////////////////////////////////////////////////////////////////

ContextObj::ContextObj(const ContextObj& co)
  : d_scope(co.d_scope), d_restore(co.d_restore) {
  IF_DEBUG(d_name=co.d_name);
  DebugAssert(co.d_active, "ContextObj["+co.name()+"] copy constructor");
  IF_DEBUG(d_active = co.d_active);
  TRACE("context verbose", "ContextObj()[", this, "]: copy constructor");
}


ContextObj& ContextObj::operator=(const ContextObj& co) {
  DebugAssert(false, "ContextObj::operator=(): shouldn't be called");
  return *this;
}


void ContextObj::update(int scope)
{
//   TRACE("context verbose", "update(", name(), ")");
//   IF_DEBUG
//     (static DebugTimer accum(debugger.timer("ContextObj::update time"));
//      static DebugTimer tmpTimer(debugger.newTimer());
//      static DebugCounter
//        updateCalls(debugger.counter("ContextObject::update calls"));
//      updateCalls++; debugger.setCurrentTime(tmpTimer));
  ContextObj* data = makeCopy();
  data->d_scope=d_scope;
  // The destructor of the copy should not destroy our older copies
  data->d_restore=NULL;
  IF_DEBUG(data->setName(name()+" [copy]"));
  ContextObjChain* obj = new ContextObjChain(data, this, d_restore);
  d_restore = obj;
  DebugAssert(scope < 0 || d_scope->level() < scope,
	      "ContextObj::update(scope="+int2string(scope)
	      +"): scope < d_scope->level() = "
	      +int2string(d_scope->level()));
  d_scope = d_scope->topScope();
  if(scope >= 0) {
    // Fetch the specified scope
    for(int i=level(); i>scope; --i) {
      d_scope = d_scope->prevScope();
      DebugAssert(d_scope != NULL, "ContextObj::update["
		  +name()+"]: d_scope == NULL");
    }
  }
  d_scope->addToChain(obj);
//   IF_DEBUG(debugger.setElapsed(tmpTimer); accum += tmpTimer);
}

// Create a new ContextObj.  The initial scope is set to the bottom
// scope by default, to reduce the work of pop() (otherwise, if the
// object is defined only on a very high scope, its scope will be
// moved down with each pop).  If 'atBottomScope' == false, the
// scope is set to the current scope.
ContextObj::ContextObj(Context* context, bool atBottomScope)
  IF_DEBUG(: d_name("ContextObj"))
{
  IF_DEBUG(d_active=true);
  DebugAssert(context != NULL, "NULL context pointer");
  if(atBottomScope) d_scope = context->bottomScope();
  else d_scope = context->topScope();
  d_restore = new ContextObjChain(NULL, this, NULL);
  d_scope->addToChain(d_restore);
  TRACE("context verbose", "ContextObj()[", this, "]");
}


ContextObj::~ContextObj()
{
//   TRACE("context verbose", "~ContextObj(", name(), ")");
  // Delete our restore copies
  TRACE("context verbose",
	string("~ContextObj(")+(d_scope && d_scope->d_delayDelete? "delayed" : "")+")[",
	this, "] {");
  IF_DEBUG(FatalAssert(d_scope==NULL || !d_scope->d_delayDelete,
                       "~ContextObj["+name()+"]"));
  IF_DEBUG(FatalAssert(d_active, "~ContextObj["+name()+"]"));
  IF_DEBUG(d_active=false); 
//   if(d_scope->d_delayDelete) {
//     (d_scope->d_deleted).push_back(d_restore);
//     // Mark the chain as "deleted"
//     d_restore->d_master=NULL;
//   } else {
  for(ContextObjChain* obj = d_restore; obj != NULL; ) {
    ContextObjChain* tmp = obj->d_restore;
    // Remove the object from the restore chain
    if(obj->d_restoreChainNext != NULL)
      obj->d_restoreChainNext->d_restoreChainPrev = obj->d_restoreChainPrev;
    *(obj->d_restoreChainPrev) = obj->d_restoreChainNext;
    // Delete the object and move to the next
    delete obj;
    obj = tmp;
  }
//   }
  TRACE("context verbose", "~ContextObj()[", this, "] }");
}


///////////////////////////////////////////////////////////////////////////////
// Context methods                                                           //
///////////////////////////////////////////////////////////////////////////////


Context::Context(ContextManager* cm, const string& name, int id)
  : d_cm(cm), d_name(name), d_id(id)
{
  d_topScope = new Scope(this);
  d_bottomScope = d_topScope;
  TRACE("context", "*** [context] Creating new context: name = "
	+ name + "id = ", id, "");
}


// Don't pop, just delete everything.  At this point, most of the
// system is already destroyed, popping may be dangerous.
Context::~Context()
{
  // popto(0);
  Scope* top = d_topScope;
  while(top != NULL) {
    top = d_topScope->prevScope();
    delete d_topScope;
    d_topScope = top;
  }
  // Erase ourselves from the notify objects, so they don't call us
  for(vector<ContextNotifyObj*>::iterator i=d_notifyObjList.begin(),
	iend=d_notifyObjList.end(); i!=iend; ++i) {
    (*i)->d_context = NULL;
  }
  // Do not delete d_bottomScope, it's deleted with the last d_topScope.
}


void Context::pop()
{
//   IF_DEBUG
//     (static DebugTimer accum(debugger.timer("Context::pop time"));
//      static DebugTimer tmpTimer(debugger.newTimer());
//      static DebugCounter calls(debugger.counter("Context::pop calls"));
//      calls++; debugger.setCurrentTime(tmpTimer));
  Scope* top = d_topScope;
  TRACE("context", "*** [context] Popping scope from level ", level(), "...");
  DebugAssert(top->prevScope() != NULL,
	      "Illegal to pop last scope off of stack.");
  // Notify before popping the scope
  for(vector<ContextNotifyObj*>::iterator i=d_notifyObjList.begin(),
	iend=d_notifyObjList.end(); i != iend; ++i)
    (*i)->notifyPre();
  // Pop the scope
  d_topScope = top->prevScope();
  top->restore();
  delete top;
  // Notify after the pop is done
  for(vector<ContextNotifyObj*>::iterator i=d_notifyObjList.begin(),
	iend=d_notifyObjList.end(); i != iend; ++i)
    (*i)->notify();
  TRACE("context", "*** [context] Popped scope to level ", level(), "}");
//   IF_DEBUG(debugger.setElapsed(tmpTimer); accum += tmpTimer);
}


void Context::popto(int toLevel)
{
  //TODO: more efficient implementation?
  // TRACE("context", "Context::popto(", toLevel, ") {");
  while (toLevel < topScope()->level()) pop();
  // TRACE("context", "Context::popto(", toLevel, ") => }");
}


void Context::deleteNotifyObj(ContextNotifyObj* obj) {
  size_t i(0), iend(d_notifyObjList.size());
  for(; i<iend && d_notifyObjList[i]!=obj; ++i);
  if(i<iend) { // Found obj; delete it from the vector
    d_notifyObjList[i]=d_notifyObjList.back();
    d_notifyObjList.pop_back();
  }
}


///////////////////////////////////////////////////////////////////////////////
// ContextManager methods                                                    //
///////////////////////////////////////////////////////////////////////////////


ContextManager::ContextManager()
{
  d_curContext = createContext("default");
}


ContextManager::~ContextManager()
{
  while (d_contexts.size()) {
    delete d_contexts.back();
    d_contexts.pop_back();
  }
}


Context* ContextManager::createContext(const string& name)
{
  d_contexts.push_back(new Context(this, name, d_contexts.size()));
  return d_contexts.back();
}


Context* ContextManager::switchContext(Context* context)
{
  DebugAssert(false, "Multiple contexts not yet implemented");
  Context* old = d_curContext;
  DebugAssert(context && context == d_contexts[context->id()],
	      "Unknown context");
  d_curContext = context;
  // TODO: Fix up all Context Objects
  return old;
}
