
(*****************************************************************************
/*!
 * \file array.ml
 *
 * Theory of array and related functions and definitions.
 * 
 * <hr>
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation for any purpose is hereby granted without
 * royalty, subject to the terms and conditions defined in the \ref
 * README file provided with this distribution.
 * 
 * <hr>
 * 
 */
/*****************************************************************************)

let list x = [x];;

let CASES_ON tm = 
  let ty,_ = dest_type (type_of tm) in
    match ty with 
        "num" -> 
          DISJ_CASES_TAC (SPEC tm num_CASES) THENL 
          [
            POP_ASSUM SUBST1_TAC;
            POP_ASSUM STRIP_ASSUME_TAC THEN POP_ASSUM SUBST1_TAC
          ]
      | "bool" -> 
          DISJ_CASES_TAC (SPEC tm EXCLUDED_MIDDLE)
      | _ -> failwith "not a case type";;

let UNDISCH_ALL_TAC = REPEAT (POP_ASSUM MP_TAC);;

let labels_flag = ref false;;

let LABEL_ALL_TAC:tactic =
 let mk_label avoid =
  let rec mk_one_label i avoid  =
    let label = "Z-"^(string_of_int i) in
      if not(mem label avoid) then label else mk_one_label (i+1) avoid in
    mk_one_label 0 avoid in
 let update_label i asl =
  let rec f_at_i f j =
    function [] -> []
      | a::b -> if (j=0) then (f a)::b else a::(f_at_i f (j-1) b) in
  let avoid = map fst asl in
  let current = el i avoid in
  let new_label = mk_label avoid in
  if (String.length current > 0) then asl else
    f_at_i (fun (_,y) -> (new_label,y) ) i asl in
  fun (asl,w) ->
    let aslp = ref asl in
    (for i=0 to ((length asl)-1) do (aslp := update_label i !aslp) done;
    (ALL_TAC (!aslp,w)));;

let e tac = refine(by(VALID
   (if !labels_flag then (tac THEN LABEL_ALL_TAC) else tac)));;

let prove_by_refinement(t,(tacl:tactic list)) =
  let gstate = mk_goalstate ([],t) in
  let _,sgs,just = rev_itlist
    (fun tac gs -> by
       (if !labels_flag then (tac THEN LABEL_ALL_TAC) else tac) gs)
     tacl gstate in
  let th = if sgs = [] then just null_inst []
  else failwith "BY_REFINEMENT_PROOF: Unsolved goals" in
  let t' = concl th in
  if t' = t then th else
  try EQ_MP (ALPHA t' t) th
  with Failure _ -> failwith "prove_by_refinement: generated wrong
theorem";;


(* ====================================================================== *)
(*  Theory of Arrays                                                      *)
(* ====================================================================== *)

new_type("array",2);; (* index_type, data_type *)

new_constant("read",`:(I,D)array->I->D`);;

new_constant("write",`:(I,D)array->I->D->(I,D)array`);;

let read_over_write = new_axiom(`!(a:(I,D)array) (i:I) (j:I) (v:D).
            ((i = j) ==> (read(write a i v) j = v)) /\ 
            ((~(i = j)) ==> (read(write a i v) j = (read a j)))`);;

let array_extensionality = new_axiom(`!(a:(I,D)array) (b:(I,D)array).
            (!(i:I). read a i = read b i) ==> (a = b)`);;

let rewrite_same_store = prove_by_refinement(
  `!S1 S2 i. (write (S1:(A,B)array) i (read S2 i) = S1) = 
    (read S1 i = read S2 i)`,[
(* {{{ Proof *)

  REPEAT STRIP_TAC; 
  EQ_TAC;
  STRIP_TAC;
  POP_ASSUM (ONCE_REWRITE_TAC o list o GSYM);
  SIMP_TAC[read_over_write];
  (* done *)
  STRIP_TAC;
  POP_ASSUM (ONCE_REWRITE_TAC o list o GSYM);  
  MATCH_MP_TAC array_extensionality;
  STRIP_TAC;
  CASES_ON `(i:A) = i'`;
  ONCE_ASM_REWRITE_TAC[];
  SIMP_TAC[read_over_write];
  ASM_SIMP_TAC[read_over_write];
]);;

(* }}} *)


(* }}} *)

let redundant1 = prove(
  `!S i v. (v = read S i) ==> (write S i v = S)`,
(* {{{ Proof *)
  MESON_TAC[read_over_write;array_extensionality;rewrite_same_store]);;
(* }}} *)

let redundant2 = prove_by_refinement(
  `!(S:(I,D)array) i v1 v2.( write (write S i v1) i v2) = (write S i v2)`,
(* {{{ Proof *)
  [
  REPEAT STRIP_TAC;
  MATCH_MP_TAC array_extensionality;
  GEN_TAC; 
  CASES_ON `(i:I) = i'`;
  ASM_SIMP_TAC[read_over_write];                      
  ASM_SIMP_TAC[read_over_write];                      
  ]);;
(* }}} *)

let read_write = prove_by_refinement(
  `!(S:(I,D)array) i1 i2 v. (read (write S i1 v) i2) = 
    (if i1 = i2 then v else read S i2)`,
(* {{{ Proof *)
    [
      REPEAT STRIP_TAC;
      COND_CASES_TAC;
      ASM_MESON_TAC[read_over_write];
      ASM_MESON_TAC[read_over_write];
    ]);;
(* }}} *)

let rewrite_same_store2 = prove_by_refinement(
  `!(S:(I,D)array) i v. ((write S i v) = S) = (read S i = v)`,
(* {{{ Proof *)
  [
    REPEAT STRIP_TAC;
    EQ_TAC;
    DISCH_THEN  (ONCE_REWRITE_TAC o list o GSYM);
    SIMP_TAC[read_over_write];
    DISCH_THEN  (ONCE_REWRITE_TAC o list o GSYM);    
    MESON_TAC[rewrite_same_store]
  ]);;
(* }}} *)

let rewrite_write_write = prove_by_refinement(
  `!(S1:(I,D)array) S2 i1 i2 v1 v2. (write S1 i1 v1 = write S2 i2 v2) = 
           ((S1 = write (write S2 i2 v2) i1 (read S1 i1)) /\ 
            (v1 = read (write S2 i2 v2) i1))`,
(* {{{ Proof *)
  [
    REPEAT STRIP_TAC;
    EQ_TAC;
    STRIP_TAC;
    CONJ_TAC;
    POP_ASSUM (REWRITE_TAC o list o GSYM);
    MATCH_MP_TAC array_extensionality; 
    STRIP_TAC;
    CASES_ON `(i:I) = i1`;
    ASM_REWRITE_TAC[];
    SIMP_TAC[read_over_write];
    ASM_SIMP_TAC[read_over_write];
    POP_ASSUM (REWRITE_TAC o list o GSYM);
    ASM_SIMP_TAC[read_over_write];
    (* 1 subgoal *)
    STRIP_TAC;
    POP_ASSUM SUBST1_TAC;
    POP_ASSUM SUBST1_TAC;
    CASES_ON `(i1:I) = i2`;
    POP_ASSUM (fun x -> UNDISCH_ALL_TAC THEN REWRITE_TAC[x] THEN 
              REPEAT STRIP_TAC);
    SIMP_TAC[read_over_write];
    MATCH_MP_TAC array_extensionality; 
    STRIP_TAC;
    CASES_ON `(i:I) = i2`;
    ASM_REWRITE_TAC[];
    SIMP_TAC[read_over_write];
    ASM_SIMP_TAC[read_over_write];
    ASM_REWRITE_TAC[];
    MATCH_MP_TAC array_extensionality;     
    STRIP_TAC;
    CASES_ON `(i:I) = i1`;    
    ASM_SIMP_TAC[read_over_write];
    ASM_SIMP_TAC[read_over_write];
]);;
    (* }}} *)

let lemma_array_simp_write_read = prove_by_refinement(

`! (store:(I,D)array) (s:(I,D)array) (index1:I) (index2:I) (v1:D) (v2:D). 
  (read (write store index1 v1) index2) = (if index1 = index2 then v1 else read store index2)` ,
[
  MESON_TAC[read_over_write];
]
);;  
  

let interchangeindices = prove_by_refinement(
`! (store:(I,D)array) (index1:I) (index2:I) (v1:D) (v2:D). 
 (write (write store index1 v1) index2 v2) =  
  write (write store index2 v2) index1 (if (index1 = index2) then v2 else v1 )`, 
[ 
  REPEAT STRIP_TAC;
  MATCH_MP_TAC array_extensionality;
  REPEAT STRIP_TAC;
  SIMP_TAC[lemma_array_simp_write_read];
  MESON_TAC[];
]);;
