(*****************************************************************************
/*!
 * \file trans.ml
 *
 * Main function of the translator.  I made a class of proof_translate because I 
 * do not want to change too much to incorporate the "use_int" option. 
 * 
 * <hr>
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation for any purpose is hereby granted without
 * royalty, subject to the terms and conditions defined in the \ref
 * README file provided with this distribution.
 * 
 * <hr>
 * 
 */
/*****************************************************************************)

(* to do: pull out all quotations *)
(* prove more meta theorems *)

(* ---------------------------------------------------------------------- *)
(* Some support functions                                                 *)
(* ---------------------------------------------------------------------- *)

prioritize_real();;

let is_plus_int  = is_binop `(+):int->int->int` ;;
let is_minus_int  =  is_binop `(-):int->int->int` ;;
let is_mult_int = is_binop `( * ):int->int->int` ;;

let mk_plus = mk_binop `(+)`;; 
let mk_minus = mk_binop `(-)`;; 
let mk_mult = mk_binop `( * )`;; 
let mk_uminus = mk_combc `(--)`;; 

let mk_lt_int = mk_binop `(<):int->int->bool`;;
let mk_lt_real = mk_binop `(<):real->real->bool`  ;;
let mk_le_int = mk_binop `(<=):int->int->bool`  ;;
let mk_le_real = mk_binop `(<=):real->real->bool`  ;;
let mk_add_int = mk_binop `(+):int->int->int` ;;
let mk_add_real = mk_binop `(+):real->real->real` ;;
let dest_le_int = dest_binary "int_le" ;;
let dest_add_int = dest_binary "int_add";;
let mk_gt_int = mk_binop `(>):int->int->bool` ;;
let mk_gt_real = mk_binop `(>):real->real->bool` ;;
let mk_ge_int = mk_binop `(>=):int->int->bool` ;;
let mk_ge_real = mk_binop `(>=):real->real->bool` ;;
let mk_mult_int = mk_binop `( * ):int->int->int` ;;
let mk_mult_real = mk_binop `( * ):real->real->real` ;;
let mk_minus_int = (mk_binop `( - ):int->int->int`) ;;
let mk_minus_real = (mk_binop `( - ):real->real->real`) ;;
let mk_plus_int = mk_binop `( + ):int->int->int` ;;
let mk_plus_real = mk_binop `( + ):real->real->real` ;;

let mk_divide = mk_binop `( / ):real->real->real` ;;  
let uminus_int = `(--):int->int` ;;
let uminus_real = `(--):real->real` ;;
let mk_uminus_real = mk_combc uminus_real;; 
let mk_pow = mk_binop `(pow)` ;;

let dest_lt_int =  dest_binop `(<):int->int->bool` ;;

(*let mk_mul = mk_binary "real_mul";;*)

let ty_num = `:num`;;
let ty_real = `:real` ;;
let ty_int =  `:int` ;;
let ty_bool = `:bool` ;;
let ty_true = `T` ;;
let ty_false = `F` ;;
let ty_a = `:A` ;;
let ty_b = `:B` ;;
let array_string  = "array" ;;
let ty_i = `:I` ;;
let ty_d = `:D` ;;
let ty_read = `read:(I,D)array->I->D` ;;
let ty_write = `write:(I,D)array->I->D->(I,D)array` ;;

let bool_ex  = `ex:bool` ;;
let bool_a = `a:bool` ;;
let bool_b = `b:bool` ;;
let bool_c = `c:bool` ;;
let bool_a1 = `a1:bool` ;;
let bool_b1 = `b1:bool` ;;
let bool_c1 = `c1:bool` ;;
let bool_e0 = `e0:bool` ;;
let bool_e1 = `e1:bool` ;;

let real_a = `a:real`;;
let real_b = `b:real`;;
let int_a = `a:int`;; 
let int_b = `b:int`;; 
let int_0 = `&0:int`;;
let real_1 = ` &1`;;


let int_of_term t = Num.int_of_num(rat_of_term t);;
let real_of_nums t = mk_comb(`(&)`,t);;

let is_lt_int  = can (dest_binop `(<):int->int->bool`) ;;
let is_le_int  = can (dest_binop `(<=):int->int->bool`) ;;
let is_gt_int  = can (dest_binop `(>):int->int->bool`) ;;
let is_ge_int  = can (dest_binop `(>=):int->int->bool`) ;;

let is_lt  = can (dest_binop `(<)`) ;;
let is_le  = can (dest_binop `(<=)`) ;;
let is_gt  = can (dest_binop `(>)`) ;;
let is_ge  = can (dest_binop `(>=)`) ;;


let mk_real_binop op tm1 =
  let rtm1 = if type_of tm1 = ty_num then real_of_nums tm1 else tm1 in
  let f = mk_comb(op,rtm1) in
  fun tm2 -> 
    let rtm2 = if type_of tm2 = ty_num then real_of_nums tm2 else tm2 in
      mk_comb(f,tm2);;

let mk_real_divide = mk_real_binop  ` ( / ) ` ;;

let isPair vc exp = 
    kind exp = getKindInt vc "RAW_LIST";;

let rec is_proof vc exp = 
  let k = kind exp in
  k = getKindInt vc "|-" or 
  k = getKindInt vc "LAMBDA" or
  is_pair vc exp

and is_pair vc exp = 
  let k = kind exp in
    k = getKindInt vc "RAW_LIST" & 
    arity exp = 2 & 
    is_proof vc (child exp 1);;  

let rec get vc exp l =
  if isLambda vc exp then get vc (getBody exp) l 
  else if is_pair vc exp then get vc (child exp 1) l 
  else match l with 
      [] -> exp 
    | h::t -> get vc (child exp h) t;;

let rec is_real_const t = 
  let t' = randt (REAL_RAT_REDUCE_CONV t) in
  let t'' = 
    try
      let neg,c = dest_comb t' in
        if neg = `(--)` then c else t'
    with Failure _ -> t' in
  let is_simp_const t = 
    try 
      let a,n = dest_comb t in
      if a = `&` & is_numeral n then true else false
    with Failure _ -> false in
  if is_simp_const t'' then true else
    try
      let l,r = dest_binary "real_div" t'' in
        if is_real_const l & is_real_const r then true else false
    with Failure _ -> false;; 


(* ---------------------------------------------------------------------- *)
(*  For shadows                                                           *)
(* ---------------------------------------------------------------------- *)

let gray_shadow_thm = new_definition `! a b c d. GRAY_SHADOW a b c d <=> c <= a-b /\ a-b <= d:int `;;

let mk_gray_shadow v e c1 c2 = 
  rev_itlist (fun x y -> mk_combc y x) [v;e;c1;c2] `(GRAY_SHADOW)`  ;;

let dest_gray_shadow  shadow = 
  let res3,c2 = dest_comb shadow in
  let res2,c1 = dest_comb res3 in
  let res1, e = dest_comb res2 in
  let _,v = dest_comb res1 in
    v,e,c1,c2;;

let dark_shadow_int_thm = new_definition `! a b . DARK_SHADOW_INT a b  <=> a <= b:int `;;
let dark_shadow_real_thm = new_definition `! a b . DARK_SHADOW_REAL a b  <=> a <= b:real `;;

let dark_shadow_int_def = `(DARK_SHADOW_INT)`;;

let mk_dark_shadow_int a b = 
  mk_combc (mk_combc dark_shadow_int_def  a) b  ;;

let dark_shadow_real_def = `(GRAY_SHADOW_REAL)`;;

let mk_dark_shadow_real a b = 
  mk_combc (mk_combc dark_shadow_real_def  a) b  ;;

let dest_dark_shadow shadow =
  let re1,b = dest_comb shadow in
  let _,a = dest_comb re1 in
    a,b;;

(* ---------------------------------------------------------------------- *)
(*  Some functions for CNF translation                                    *)
(* ---------------------------------------------------------------------- *)

let safe_disch term thm =
  DISCH term thm
(* because HOL Light will eliminate alpha-equivalent terms in the assumption
 this code does not work any more 
  let assumps = hyp thm in
    if (exists (fun x -> x = term) assumps )
    then DISCH term thm
    else (pr "============ let us debug for safe disch ================\n";
	  pr "== the term is == \n";
	  pt term ;
	  pr "\n== the thm is == \n";
	  pthm thm ;
	  pr "\n== end == \n";
	  failwith "safe dish";)
*)
;;
(*
let check_disch term thm =
  let assumps = hyp thm in
    if (exists (fun x -> x = term) assumps )
    then DISCH term thm
    else (pr "============ let us debug for safe disch ================\n";
	  pr "== the term is == \n";
	  pt term ;
	  pr "\n== the thm is == \n";
	  pthm thm ;
	  pr "\n== end == \n";
	  failwith "safe dish";)
;;
*)


let hol_negate tm =
  try dest_neg tm with Failure _ -> mk_neg tm;;


let gen_pos_neg term is_and = 
  let pick_pos_neg term is_fun dest_fun  = 
    let rec all_terms term  = 
      if is_fun term then
	let l,r = dest_fun term in
	  (all_terms l)@(all_terms r) 
      else [term]  in
    let rec map_terms term map = 
      if is_fun term then
	let l,r = dest_fun term in
	  map_terms l (map_terms r map)
      else (term |-> true) map in
    let has_neg_pos term map = 
      try
	let has_pos = apply map term in
	let has_neg = apply map (hol_negate term) in
	  if has_pos && has_neg 
	  then true
	  else false 
      with Failure _ -> false in
    let rec try_find ts map =
      match ts with
	  [] -> failwith "not found"
	| h::t -> 
	    if (has_neg_pos h map) then h
	    else try_find t map in
      
    let tms = all_terms term in
    let tmap = map_terms term undefined in
      try 
	let t = try_find tms tmap in
	let ngt = hol_negate t in
	let others = filter (fun x -> x <> t && x <> ngt) tms in
	  t,others 
      with Failure _ -> failwith "no pos and neg"	    
  in
    try
      if is_and then 
	let t,others = pick_pos_neg term is_conj dest_conj in
	let newterm = 
	  if (is_neg t) 
	  then list_mk_conj (mk_conj( (hol_negate t),t)::others)
	  else list_mk_conj (mk_conj( t,(hol_negate t))::others) in
	  CONJ_ACI_RULE (mk_eqc term newterm)
      else 
	let t,others = pick_pos_neg term is_disj dest_disj in
	let newterm = 
	  if (is_neg t) 
	  then list_mk_disj (mk_disj((hol_negate t),t)::others)
	  else list_mk_disj (mk_disj(t,(hol_negate t))::others) in
	  DISJ_ACI_RULE (mk_eqc term newterm)	  
    with Failure _ -> REFL term ;;


(* In a chain of combs, replace the (uniform) operator op with f op
   and the argument arg with g arg, 
   eg: 
   walk_op_chain 
   (fun x -> if x = `(/\)` then `(\/)` else fail()) 
   (fun x -> mk_neg x) 
   `a /\ b /\ c` --> `~a \/ ~b \/ ~c` *)

let rec walk_op_chain op_pred op_fun arg_fun tm = 
  match tm with
      Var _ as v -> arg_fun v
    | Const _ as c -> arg_fun c
    | Abs _ as a -> arg_fun a
    | Comb(Comb(op,arg),t2) as c -> 
        if op_pred op then 
          let walked_tm = walk_op_chain op_pred op_fun arg_fun t2 in
            mk_comb(mk_comb(op_fun op,arg_fun arg),walked_tm)
        else arg_fun c 
    | Comb _ as c -> arg_fun c;;


(* f `~(P /\ Q /\ ... /\ ~Z /\ A)` = `(~P \/ ~Q \/ ... \/ Z \/ ~A)` *)
(* Note the subtlety here.  If Z is for the form (x /\ y) there is 
   no way to distinguish between that case and if x and y were top
   level elements.  Thus, if the CVC term is 
   (OR e1 ... en (OR a b)), this will get translated as
   e1 \/ ... \/ en \/ a \/ b and will thus be negated as
   ~e1 /\ ... /\ ~en /\ ~a /\ ~b instead of the correct proper 
   ~e1 /\ ... /\ ~en /\ ~(a \/ b)
   I`m not going to worry about it now.  If it causes problems down
   the road, I`ll fix it then.
 *)

let DIST_NOT_AND tm = 
  let and_to_not_and = 
    walk_op_chain 
      (fun x -> if x = `(/\)` then true else false)
      (fun x -> `(\/)`)
      (fun x -> if can dest_neg x then dest_neg x else (mk_neg x)) in
  let pos_tm = tm in  
  let neg_tm = and_to_not_and pos_tm in
    TAUT (mk_eq(tm,neg_tm));;


let DIST_NOT_OR tm = 
  let or_to_not_and = 
    walk_op_chain 
      (fun x -> if x = `(\/)` then true else false)
      (fun x -> `(/\)`)
      (fun x -> if can dest_neg x then dest_neg x else (mk_neg x)) in
  let pos_tm = dest_neg tm in
  let neg_tm = or_to_not_and pos_tm in
    TAUT (mk_eq(tm,neg_tm));;


(* ---------------------------------------------------------------------- *)
(*  Conver int to real                                                    *)
(* ---------------------------------------------------------------------- *)

let is_int x = (type_of x) = ty_int ;;
let is_real x = (type_of x) = ty_real ;;

let rec up_term x  = 
  if is_int x 
  then 
    (
      pr "warning: convert integers to reals:";  pt x;   pr "\n";    
      let is_neg,n = try(
	  let sign,n = dest_comb x in
	    if (sign = uminus_int )
	    then (true, n)
	    else (false, x) 
      )
	with Failure _ -> (false, x)   in 
      let is_numint, nint = try (
	  let l,r = dest_comb n in
	    if (l = `(&):num->int`)
	    then true, r
	    else false, n
      ) 
	with Failure _ -> false, n  in
      let res_num =( 
	  if is_numint 
	  then (mk_combc `(&):num->real` nint)
 	  else if (is_plus_int  nint )
	  then let ch2 = up_term (rand nint) and ch1 = up_term (rand (rator nint)) in
		 end_itlist mk_plus_real [ ch1; ch2]   
 	  else if (is_minus_int  nint )
	  then let ch2 = up_term (rand nint) and ch1 = up_term (rand (rator nint)) in
		 end_itlist mk_minus_real [ ch1; ch2]   
 	  else if (is_mult_int  nint )
	  then let ch2 = up_term (rand nint)  and ch1 = up_term (rand (rator nint)) in
		 end_itlist mk_mult_real  [ ch1; ch2]   
	  else (pr "Warining: real to int convertion: "; pt nint; pr "\n"; mk_combc `real_of_int` nint)
      ) in
	if is_neg 
	then mk_uminus_real res_num 
	else res_num  
    )
  else x;;

let up_convert terms =
  if (exists is_real terms) then
    ((map up_term terms ) , false)
  else terms, true ;;

let mk_eq_int_real e1 e2 =
  if ((is_int e1) && (is_int e2) ) 
  then mk_eq (e1,e2) 
  else mk_eq(up_term e1, up_term e2);;


let num_of_real t = 
  match t with 
    Comb(f,n) when f = `(&)` -> n
  | Var(name,_) -> (mk_var (name,`:num`)) 
  | _ -> failwith "num_of_real";;



(* ---------------------------------------------------------------------- *)
(*  deal with reals                                                       *)
(* ---------------------------------------------------------------------- *)
let to_delete x =
  x = ")" or x = "(" or x = " ";;

let is_neg_real x =
  x = "-" or x = "~" ;;

let deal_real s =
  let str_list = filter (fun x -> not (to_delete x)) (explode s) in
  if length str_list >= 1
  then
    if is_neg_real (hd str_list)
    then true, (implode (tl str_list))
    else false,(implode str_list)
  else false, (implode str_list) ;;


(* substitute select operator by fresh var, used for REAL_ARITH *)
let rec subs_select_op tm env = 
  match tm with
      Comb(s,t) -> 
	  if is_select tm then 
	    try let new_tm = assoc tm env in
		  new_tm,env
	    with Failure _ -> 
	    let ty = type_of tm in 
	    let new_tm = genvar ty in
	      (new_tm,((tm,new_tm)::env))
	  else 
	    let s',new_env1 = subs_select_op s env in
	    let t',new_env2  = subs_select_op t new_env1 in
	      if s' == s & t' == t 
	      then (tm, env) 
	      else ((mk_comb (s',t')), new_env2)

      | _ -> (tm, env) ;;


(* ------------------------------------------------------------------- *)
(*  A special SUBS_CONV and special subst                              *)
(*  For special_subst, it needs a_list, tm2 , tm1                      *)
(*  tm2 and tm1 must have the same syntax structure                    *)
(*  then, the exprs in tm1 will be replaced according to the pairs in  *)
(*  a_list when the crosponding parts of tm1 and tm2 are NOT the same  *)
(*  for example, if tm1 is a/\a and tm2 is a/\b, a_list is [(a,c)]     *)
(*  then special_subst will return a/\c                                *)
(*  notice that the noraml subst returns c/\c                          *)
(* ------------------------------------------------------------------- *)

let special_qcomb con fn (l2,r2) (l1,r1) =
  try let l' = fn l2 l1 in
      try let r' = fn r2 r1 in con(l',r')
      with Unchanged -> con(l',r1)
  with Unchanged ->
      let r' = fn r2 r1 in con(l1,r');;

let special_subst =
  let mk_qcomb = special_qcomb mk_comb in
  let rec ssubst ilist tm2 tm1=
    if ilist = [] then raise Unchanged else
    try fst (find (fun x -> (((aconv tm1) o snd) x) && (tm1 <> tm2)) ilist) with Failure _ ->
    if is_comb tm1 then(
      mk_qcomb (ssubst ilist) (dest_comb tm2) (dest_comb tm1)  )
    else if is_abs tm1 then
      let v1,bod1 = dest_abs tm1 in
      let v2,bod2 = dest_abs tm2 in
      mk_abs(v1,ssubst (filter(not o (vfree_in v1) o snd) ilist) bod2 bod1 )
    else raise Unchanged in
  fun ilist ->
    let ts,xs = unzip ilist in
    let gs = map (genvar o type_of) xs in
    fun tm2 tm1 -> try vsubst (zip ts gs) (ssubst (zip gs xs) tm2 tm1)
              with Unchanged -> tm1;;

let special_SUBS_CONV ths tm2 tm1 =
  try if ths = [] then REFL tm1 else
      let lefts = map (lhand o concl) ths in
      let gvs = map (genvar o type_of) lefts in
      let pat = special_subst (zip gvs lefts) tm2 tm1 in
      let abs = list_mk_abs(gvs,pat) in
      let th = rev_itlist
        (fun y x -> CONV_RULE (RAND_CONV BETA_CONV THENC LAND_CONV BETA_CONV)
                              (MK_COMB(x,y))) ths (REFL abs) in
      if rand(concl th) = tm1 then REFL tm1 else th
  with Failure x -> pr x; failwith "special SUBS_CONV";;

    
(* ---------------------------------------------------------------------- *)
(*  Some support functions                                                *)
(* ---------------------------------------------------------------------- *)


let dest_div = dest_binop ` ( / )`;;
let dest_or = dest_binop ` ( \/ )`;;

let rec prove_DIV_NOT_EQ_0 tm =
  try 
    let a,b = dest_div tm in
    let pf_a = prove_DIV_NOT_EQ_0 a in
    let pf_b =  prove_DIV_NOT_EQ_0 b in
     REWRITE_RULE [lemma_le7] (CONJ pf_a pf_b)
  with Failure x -> 
    if (is_int tm) then
      INT_ARITH (mk_neg (mk_eqc tm ` &0:int`))
    else
      prove (mk_neg (mk_eqc tm ` &0`) , REAL_ARITH_TAC);;


let or_dis_thm x =
  let ints = 1 -- x in
  let bs = map(fun x-> mk_var ("b" ^ (string_of_int x), ty_bool)) ints in
  let left_terms = map(fun x-> mk_conj (bool_a, x )) bs in
  let lhs = end_itlist (curry mk_disj) left_terms in
  let rhs = mk_conj (bool_a, (end_itlist (curry mk_disj) bs)) in
   bs, TAUT (mk_eqc lhs rhs) ;;


(* a ==> b to ~a \/ b *)
let IMP_OR aExpr thm =
(*  let impThm = DISCH aExpr thm in*)
  let impThm = safe_disch aExpr thm in
  let bExpr = concl thm in
  let impOrThm = 
    (if  bExpr = ty_false 
    then if is_neg aExpr 
      then INST [((hol_negate aExpr), bool_a)] IMP_FALSE_THM_NEG
      else INST [(aExpr, bool_a)] IMP_FALSE_THM_POS  
    else
      (if (is_neg aExpr)  
	then INST [((hol_negate aExpr), bool_a);(bExpr, bool_b)] IMP_OR_THM_NEG
	else INST [(aExpr, bool_a);(bExpr, bool_b)] IMP_OR_THM_POS ) )  in
    MP impOrThm impThm;;


(* ---------------------------------------------------------------------- *)
(*  Translation of CNF                                                    *)
(* ---------------------------------------------------------------------- *)


(* 
  let DISJ_REORDER thm specialTerm  =
  let thmExpr = (concl thm) in
  if thmExpr = specialTerm 
  then  thm, []
  else
  let disjs,checkList = partition (fun x -> x <> specialTerm ) (disjuncts thmExpr) in
  if ((uniq checkList) <> [specialTerm]) 
  then(
  print_string "\n debug for reordering\n";
  print_term specialTerm ;
  print_string "\n== thm ==\n";
  print_thm thm ;
  print_string "\n=== disjs ==\n";
  print_list_term disjs;
  print_string "\n-----\n";
  print_list_term checkList;
  print_string "\n end of debug \n";
  failwith "reordering disj" )
  else
  let remainExpr = list_mk_disj disjs in
  let newExpr = mk_disj (specialTerm, remainExpr )  in
  let disjEQUThm = DISJ_ACI_RULE (mk_iff (thmExpr, newExpr)) in
  (EQ_MP disjEQUThm thm),[remainExpr]
  ;;
*)


(*
let fix_disj l r term =
  if(l = `F`  ) 
  then INST [(r,`a:bool`)] lemma_or_false_l    
  else if ( l = `~T`)
  then INST [(r,`a:bool`)] lemma_or_not_true_l
  else if ( r = `F`)
  then INST [(l,`a:bool`)] lemma_or_false_r
  else if ( r = `~T`)
  then INST [(l,`a:bool`)] lemma_or_not_true_r
  else REFL term;;


let rec reorder_disj bigTerm smallTerm =
  if (smallTerm = bigTerm )
  then REFL bigTerm, true
  else if (not (is_disj bigTerm))
  then REFL bigTerm, false
  else
    let l,r = dest_disj bigTerm in
(*    let fix = fix_disj l r  bigTerm in
      if (rhs (concl fix)) <> bigTerm
      then fix,(l = smallTerm or r = smallTerm) (*this line has a problem. let me fix it later *)
      else 
*)	let leftCh, leftRes = reorder_disj l smallTerm and 
	    rightCh, rightRes = reorder_disj r smallTerm in
	  if( not leftRes && not rightRes)
	  then (REFL bigTerm, false)
	  else if ( leftRes && rightRes) 
	  then
	    let tempThm = MK_DISJ leftCh rightCh  in
	    let leftExpr = rhs (concl leftCh  ) and
		rightExpr = rhs (concl rightCh) in
	      if(leftExpr = smallTerm) && (rightExpr = smallTerm) 
	      then (
		  let temp2new = INST [(leftExpr, `a:bool`)] lemma_or_comb_nothing in
		  (TRANS tempThm temp2new), true
	      )
	      else if (leftExpr = smallTerm) 
	      then ( 
		  let rightLeft,rightRight = dest_disj rightExpr in
		  let temp2new = INST [(leftExpr, `a:bool`);(rightRight,`c:bool`)] lemma_or_comb_left in
		    (TRANS tempThm temp2new), true
	      )
	      else if (rightExpr = smallTerm)
	      then ( 
		  let leftLeft,leftRight = dest_disj leftExpr in
		  let temp2new = INST [(rightExpr, `a:bool`);(leftRight,`b:bool`)] lemma_or_comb_right in
		    (TRANS tempThm temp2new), true
	      )
	      else (
		  let leftLeft,leftRight = dest_disj leftExpr in
		  let rightLeft,rightRight = dest_disj rightExpr in
		  let temp2new = 
		    INST [(leftLeft, `a:bool`);(leftRight,`b:bool`);(rightRight,`c:bool`)] lemma_or_comb_both in
		    (TRANS tempThm temp2new), true
	      )

	  else
	    let tempThm = MK_DISJ leftCh rightCh in
	    let tempLeft, tempRight = dest_disj (rhs (concl tempThm)) in
	      assert (bigTerm = lhs (concl tempThm));
	      let newThm = 
		( 
		  if rightRes 
		  then 
		    let temp2new = INST [(tempLeft,`a:bool`);(tempRight,`b:bool`)] lemma_or_exchange  in
		      TRANS tempThm temp2new
		  else
		    tempThm
		) in
	      let newBigTerm = rhs (concl newThm) in	    
	      let newLeft, newRight = dest_disj newBigTerm in
		if(newLeft = smallTerm)
		then newThm,true
		else
		  ( assert (is_disj newLeft);
		    let leftLeft, leftRight =  dest_disj newLeft in
		      assert (leftLeft = smallTerm) ;
		      let new2final = INST [(leftLeft,`a:bool`);(leftRight,`b:bool`);(newRight,`c:bool`)] lemma_or_shift in
			(*
			  print_thm newThm;
			  print_string " \n " ;
			  print_thm new2final;
			  print_string " \n " ;
			*)
			(TRANS newThm  new2final, true)
		  )
;;

let DISJ_REORDER thm specialTerm  =
  if specialTerm = concl thm 
   then  (thm, [])
  else 
    let mpThm,succ = reorder_disj (concl thm) specialTerm in
      if  succ 
      then
	let  resThm = EQ_MP mpThm thm in  
	let  remains =  (
	  try let _, newRight = dest_disj (concl resThm) in [newRight]
	  with Failure x -> []) in
	  resThm, remains
      else failwith "cannot find special term";;

*)

(*
let is_false term =
 if term = `F` or term = `~T` then true else false ;;
*)

(* for reorder_disj, this version is working except the true and false 
stuff because DISJ_ACI_RULE require that 
*)

(*
let rec reorder_disj bigTerm smallTerm =
  if (smallTerm = bigTerm )
  then bigTerm, true
  else if (not (is_disj bigTerm))
  then bigTerm, false
  else
    let l,r = dest_disj bigTerm in
(*      if is_false l  then reorder_disj r smallTerm
      else if is_false r then reorder_disj l smallTerm
      else
*)
	let leftCh, leftRes = reorder_disj l smallTerm and 
	    rightCh, rightRes = reorder_disj r smallTerm in
	  if ( not leftRes && not rightRes)
	  then 
	    (bigTerm, false)
	  else if ( leftRes && rightRes) 
	  then  
	    (
	      if rightCh = smallTerm 
	      then leftCh, leftRes
	      else if leftCh = smallTerm
	      then rightCh, rightRes
	      else
		( 
		  let leftLeft, leftRight = dest_disj leftCh and
		      rightLeft, rightRight = dest_disj rightCh in
(*		    assert(leftLeft= rightLeft && leftLeft = smallTerm);*)
		    mk_disj(leftLeft, mk_disj(leftRight, rightRight)), true
		)
	    )
	  else 
	    let newLeft , newRight = 
	      (if rightRes then rightCh, leftCh else leftCh, rightCh) in
	      if newLeft <> smallTerm then
		let newLeftLeft, newLeftRight = dest_disj newLeft in
(*		  assert(smallTerm = newLeftLeft);*)
		  mk_disj(newLeftLeft, mk_disj(newLeftRight,newRight)), true
	      else mk_disj(newLeft, newRight), true 
;;


let DISJ_REORDER thm specialTerm  =
  let thmExpr = concl thm in
  if specialTerm = thmExpr 
  then   (thm, [])
  else 
    let mpThmExpr,succ = reorder_disj thmExpr  specialTerm in
      if succ 
      then 
	let resThm = EQ_MP (DISJ_ACI_RULE (mk_iff (thmExpr, mpThmExpr)))  thm in
	let remains = 
	  ( try let _, newRight = dest_disj mpThmExpr in [newRight]
	    with Failure x-> [] ) in
	  resThm,remains
      else failwith "cannot find special term" ;;
*)

(**)

(*
let rec rec_prebool thm =
  let term = concl thm in
    if(term = `F`) 
    then thm
    else 
      let newThm =  
	(try 
	    let left,right = dest_disj term in
	      if(is_neg left)
	      then (INST [((hol_negate left),e_a);(right,e_b)] disj_imp_neg)
	      else (INST [(left,e_a);(right,e_b)] disj_imp_pos)
	  with Failure x -> 
	    if (is_neg term)
	    then INST [((hol_negate term),e_a)] disj_false_neg
	    else INST [(term,e_a)] disj_false_pos
	) in
	rec_prebool (UNDISCH (EQ_MP newThm thm)) ;;
*)


let rec rec_trans_cnf thm orlist =
  let term = concl thm in
    if(term = ty_false ) 
    then ( 
	if orlist = [] 
	then thm 
	else if orlist = [ty_false]
	then thm
	else ( print_thm thm ; print_string "\n"; failwith "orlist not empty"))
    else 
      let newThm =  
	(try
	    let left,right = dest_disj term in
	      if ((hd orlist) <> left) then failwith "unknown error in translating CNF" 
	      else if(is_neg left)
	      then (INST [((hol_negate left),bool_a);(right,bool_b)] disj_imp_neg)
	      else (INST [(left,bool_a);(right,bool_b)] disj_imp_pos)
	  with Failure x -> 
(*	    assert ((length orlist) = 1);*)
	    assert ((hd orlist) = term);
	    if (is_neg term)
	    then INST [((hol_negate term),bool_a)] disj_false_neg
	    else INST [(term,bool_a)] disj_false_pos
	) in
	rec_trans_cnf (UNDISCH (EQ_MP newThm thm)) (tl orlist) ;;


let pre_bool thm =
  let term = concl thm in
    if(term <> ty_false )
    then (
	failwith "not a F";
(*
	let res = rec_prebool thm in
	  print_string (string_of_thm res);
	  print_newline();
	  res
*)
    )
    else thm ;;

let cnfthm_and_mid_l = TAUT `a/\b ==> a`;;
let cnfthm_and_mid_r = TAUT `a/\b ==> b`;;
let cnfthm_and_mid = TAUT `a==>a`;;
let cnfthm_and_mid_or = TAUT `a==>b <=> ~a\/b`;;

let rec cnf_and_mid_thm conjs term  =
  if (conjs = term) then  INST [term,bool_a] cnfthm_and_mid , true
  else if (is_conj conjs) then  
    let l,r = dest_conj conjs in
    let lthm,lsucc = cnf_and_mid_thm l term in
      if (not lsucc) then 
	let rthm,rsucc = cnf_and_mid_thm r term in
	  if (not rsucc) then REFL r, false
	  else IMP_TRANS (INST [l,bool_a;r,bool_b] cnfthm_and_mid_r) rthm, true
      else IMP_TRANS (INST [l,bool_a;r,bool_b] cnfthm_and_mid_l) lthm,true
  else
    let found = (term = conjs) in
      INST [term,bool_a] cnfthm_and_mid , found ;;

let cnfthm_or_mid_l = TAUT `~(a\/b) ==> ~a`;;
let cnfthm_or_mid_l_n = TAUT `~((~ a)\/b) ==> a`;;

let cnfthm_or_mid_r = TAUT `~(a\/b) ==> ~b`;;
let cnfthm_or_mid_r_n = TAUT `~(a\/(~ b)) ==> b`;;

let cnfthm_or_mid = TAUT `~a ==> ~a`;;
let cnfthm_or_mid_n = TAUT `a ==> a`;;

let cnfthm_or_mid_or = TAUT `(~a ==> ~b) <=> a\/ (~ b)`;;
let cnfthm_or_mid_or_n = TAUT `(~a ==> b) <=> a\/ (b)`;;


let rec cnf_or_mid_thm disjs term  =
  if (disjs = term) then  if (is_neg term) 
    then INST [(hol_negate term),bool_a] cnfthm_or_mid_n , true
    else INST [term,bool_a] cnfthm_or_mid , true
  else if (is_disj disjs) then  
    let l,r = dest_disj disjs in
    let lthm,lsucc = cnf_or_mid_thm l term in
      if (not lsucc) then 
	let rthm,rsucc = cnf_or_mid_thm r term in
	  if (not rsucc) then REFL r, false
	  else let thm = if (is_neg r)  
	    then INST [l,bool_a;(hol_negate r),bool_b] cnfthm_or_mid_r_n
	    else INST [l,bool_a;r,bool_b] cnfthm_or_mid_r in
		 IMP_TRANS thm  rthm, true
      else let thm = if (is_neg l) 
	then INST [(hol_negate l),bool_a;r,bool_b] cnfthm_or_mid_l_n
	else (INST [l,bool_a;r,bool_b] cnfthm_or_mid_l) in
	     IMP_TRANS thm lthm, true
  else
    let found = (term = disjs) in
      INST [term,bool_a] cnfthm_or_mid , found ;;


let cnfthm_ite_1 = TAUT `a=a1 ==> b=b1 ==> c=c1 ==> ~(if a then b else c) \/ a1 \/ c1 `;; 
let cnfthm_ite_2 = TAUT `a=a1 ==> b=b1 ==> c=c1 ==>  (if a then b else c) \/ a1 \/ ~c1` ;;
let cnfthm_ite_3 = TAUT `a=a1 ==> b=b1 ==> c=c1 ==>  (if a then b else c) \/ ~a1 \/ ~b1` ;;
let cnfthm_ite_4 = TAUT `a=a1 ==> b=b1 ==> c=c1 ==> ~(if a then b else c) \/ ~a1 \/ b1` ;;
let cnfthm_ite_5 = TAUT `a=a1 ==> b=b1 ==> c=c1 ==>  (if a then b else c) \/ ~b1 \/ ~c1` ;;
let cnfthm_ite_6 = TAUT `a=a1 ==> b=b1 ==> c=c1 ==> ~(if a then b else c) \/ b1 \/ c1` ;;

let cnfthm_imp_0 = TAUT `(a==>b) \/ a ` ;;

let cnfthm_imp_1 = TAUT `(a==>b) \/ (~b)` ;;
let cnfthm_imp_1_n = TAUT `(a==> (~ b)) \/ (b)` ;;

let cnfthm_imp_2 = TAUT `~(a==>b) \/ ~a \/ b` ;;
let cnfthm_imp_2_n = TAUT `~((~ a)==>b) \/ a \/ b` ;;

let cnfthm_iff_0 = TAUT `(a<=>b) \/ a \/ b` ;;
let cnfthm_iff_1 = TAUT `(a<=>b) \/ ~a \/ ~b` ;;
let cnfthm_iff_2 = TAUT `~(a<=>b) \/ ~a \/ b` ;;
let cnfthm_iff_3 = TAUT `~(a<=>b) \/ a \/ ~b` ;;

let dneg = TAUT `(~ ~x) <=> x ` ;;

let cnfthm_or_final_pos = TAUT `~b \/ b`;;  
let cnfthm_or_final_neg = TAUT `b \/ ~b`;;  

let cnf_or_final term =
  if is_neg term then INST [(hol_negate term),bool_b] cnfthm_or_final_neg
  else INST [term,bool_b] cnfthm_or_final_pos;; 

let cnfthm_and_final = TAUT `a /\ ~(a/\b) ==> ~b`;;
let cnfthm_and_final_1 = TAUT `a /\ (~ a) ==> F`;;

let rec cnf_and_final thin ands =
  match ands with 
      [] -> thin
    | a::[] -> let b = dest_neg (concl thin) in
		 if (b = a) then
		   let mpthm = INST [a,bool_a] cnfthm_and_final_1 in
		     MP mpthm (CONJ (ASSUME a) thin)
		 else failwith "final case in cnf_and_final"
    | a::t -> 
	let b = try snd (dest_conj (dest_neg (concl thin))) with Failure _ -> failwith "debug here"  in
	let mpthm = INST [a,bool_a;b,bool_b] cnfthm_and_final in
	let thout = MP mpthm (CONJ (ASSUME a) thin) in
	  cnf_and_final thout t ;;


(*

let distinct_thm = new_definition `DISTINCT p = !a b. a IN p /\ b IN p ==> ~(a=b)` ;; 

let in_op = `(IN)`;;

let mk_in ele set =
  mk_icomb (mk_icomb (in_op,ele),set);;

let distinct_pair distinct_term distinct_set  andterm =
  let a,b = dest_conj andterm in
  let th1 = ASSUME distinct_term  in
  let th2 = SPECL [a;b] (REWRITE_RULE [distinct_thm] th1) in
  let tha = SET_RULE (mk_in a distinct_set) in
  let thb = SET_RULE (mk_in b distinct_set) in
  let th3 = CONJ tha thb in
    MP th2 th2 ;;

*)

let debug1 = ref [];; 

(* ----------------------------------------------------------------------- *)
(*  DISTINCT from SMT LIB                                                  *)
(*  The definition used now is pretty slow because O(n^2) terms are needed *)
(*  for some cases, most time is spent on generate the definition          *)
(*  It would be better if we can use some 'dnyamic' ways to generate the   *)
(*  needed theorems                                                        *)
(* ----------------------------------------------------------------------- *)
let all_distinct_table = Hashtbl.create 100;;

let generate_distinct_def len =
  let l1 = 1--len in
    pr "\n DISTINCTDEF :";
    print_int len;
    pr "\n";
  let var_names = map (fun x -> "v"^(string_of_int x)) l1 in
  let def_name = "DISTINCT"^(string_of_int len) in
  let lhs = def_name ^ " " ^ (itlist (fun x y -> x ^" " ^ y) var_names "") in
  let rhs =  (
      let temp = ref "" in
	for i = 0 to len-1  do
	  for j = i+1 to len-1  do
	    let item = "~(" ^ (el i var_names) ^ "=" ^ (el j var_names) ^ ")"in
	      if (!temp = "") 
	      then temp := (!temp) ^ item   
	      else temp := !temp ^ " /\\ " ^ item 
	  done;
	  done;
	      !temp
  ) in
    
  let new_def_string  = (lhs ^ " = (" ^ rhs ^ ")") in
  let new_def = parse_term new_def_string in
  let dist_thm = new_definition new_def in
  let distinct_pred = parse_term def_name in
    Hashtbl.add all_distinct_table len dist_thm  ;
    dist_thm ;;

let term_prod t1 t2 = 
  let thm = REAL_RAT_REDUCE_CONV (mk_mult t1 t2) in
    rhs (concl thm);;

(* encode is from HOL Light to CVC *)
exception Kind;;
(*exception Encode_term;;*)
exception Translate_term;;

class encode_cvc = 
  let rec encode_type vc hol_type = 
    if not (is_type hol_type) 
    then failwith "Variable Types not supported" else    
      let name,list = dest_type hol_type in
	match name with 
            "real" -> realType vc 
          | "num" -> intType vc 
          | "bool" -> boolType vc 
          | "array" -> 
              if (length list <> 2) then failwith "bad array type args" 
              else 
		(* all indices are real for now *)
		let i,d = hd list,hd (tl list) in
		let index_type = encode_type vc i in
		let data_type = encode_type vc d in
		  arrayType vc index_type data_type             
          | _ -> failwith "No type match" in

  let en_tm_tbl = Hashtbl.create 100 in 

  let unary_fun_map vc = [
      (`(~)`,notExpr vc);
  ] in
    
  let binary_fun_map vc = [
      (`(=):bool->bool->bool`,iffExpr vc);
      (`(=):A->A->bool`,eqExpr vc);
      (`(==>)`,impliesExpr vc);
      (`(/\)`,andExpr vc);
       (`(\/)`,orExpr vc);
       (`(+):real->real->real`,plusExpr vc);
       (`(-):real->real->real`,minusExpr vc);
       (`( * ):real->real->real`,multExpr vc);
       (`(>):real->real->bool`,gtExpr vc);
       (`(>=):real->real->bool`,geExpr vc);
       (`(<):real->real->bool`,ltExpr vc);
       (`(<=):real->real->bool`,leExpr vc);
       (`(pow):real->num->real`,C (powExpr vc));
       (`read:(I,D)array->I->D`,readExpr vc);
      ] in
			    
  let ternary_fun_map vc = [
      (`write:(I,D)array->I->D->(I,D)array`,writeExpr vc);
  ] in
    
  let can_match t1 t2 = 
    can (term_match [] t1) t2 in
    
  let match_assoc x l = snd(find (fun p -> can_match (fst p) x) l) in 
    
  let lookup_unary_fun vc x = match_assoc x (unary_fun_map vc) in 
  let lookup_binary_fun vc x = match_assoc x (binary_fun_map vc) in 
  let lookup_ternary_fun vc x = match_assoc x (ternary_fun_map vc) in
  let unary_map f l = f (hd l) in
  let binary_map f l = f (hd l) (hd(tl l)) in
  let ternary_map f l = f (hd l) (hd(tl l)) (hd(tl(tl l))) in
    
  let rec  encode_term vc tm = 
    (if is_real_const tm then ratExpr vc (int_of_term tm) 1 else
	( match tm with 
	    Var(name,ty) -> ( try Hashtbl.find en_tm_tbl name
              with Not_found -> 
		let t =  varExpr vc name (encode_type vc ty) in
                  Hashtbl.add en_tm_tbl name t;
                  t )
	  | Const _ as c -> 
              if c = ty_false then falseExpr vc else 
		failwith "constant not supported"
	  | Abs _ -> failwith "abs not supported"
	  | Comb _ -> 
              let f,cvc_expr_list = gather vc tm in
		match length cvc_expr_list with
		    1 -> unary_map (lookup_unary_fun vc f) cvc_expr_list
		  | 2 -> binary_map (lookup_binary_fun vc f) cvc_expr_list
		  | 3 -> ternary_map (lookup_ternary_fun vc f) cvc_expr_list
		  | _ -> failwith "encode_term"
	))
      
  and gather vc tm = 
    (let rec gather vc tm store = 
      (match tm with
          Var _ | Const _ | Abs _ -> failwith "gather"
	| Comb(l,r) -> let cr = encode_term vc r in
			 match l with 
			     Var _ | Const _ -> l,cr::store
			   | Abs _ -> failwith "gather2"
			   | Comb _ -> gather vc l (cr::store))  in
       gather vc tm [] )  in
object 
  method encode_term vc t = 
(*    Hashtbl.clear en_tm_tbl; *)
    encode_term vc t
end ;;    

let debug_table = ref[] ;;

let string_or_final = "\"or_final\"" ;;
let string_and_final = "\"and_final\"" ;;
let string_ite = "\"ite\"";;
let string_iff = "\"iff\"";;
let string_and =  "\"and_mid\"" ;;
let string_or_mid = "\"or_mid\"";;
let string_imp = "\"imp\"" ;;


class translate_cvc vc use_ints  =  
  let kind_real =  getKindInt vc "_REAL" in
  let kind_int =  getKindInt vc "_INT" in
  let kind_array =  getKindInt vc "_ARRAY" in
  let kind_bool =  getKindInt vc "_BOOLEAN" in
  let kind_arrow = getKindInt vc "_ARROW" in 
  let kind_typedecl = getKindInt vc "_TYPEDECL" in 
  let kind_ufunc =  getKindInt vc "_UFUNC" in
  let kind_apply = getKindInt vc "_APPLY" in
  let kind_ite = getKindInt vc "_ITE" in
  let kind_bool_var = getKindInt vc "_BOOL_VAR" in
  let kind_rational_var = getKindInt vc "_RATIONAL_EXPR" in
  let kind_true_expr = getKindInt vc "_TRUE_EXPR" in
  let kind_false_expr =  getKindInt vc "_FALSE_EXPR" in
  let kind_uconst = getKindInt vc "_UCONST" in
  let kind_eq = getKindInt vc "_EQ" in
  let kind_not = getKindInt vc "_NOT" in
  let kind_and = getKindInt vc "_AND" in
  let kind_or = getKindInt vc "_OR" in
  let kind_iff = getKindInt vc "_IFF" in
  let kind_implies = getKindInt vc "_IMPLIES" in 
  let kind_lt = getKindInt vc "_LT" in
  let kind_le = getKindInt vc "_LE" in 
  let kind_gt = getKindInt vc "_GT" in
  let kind_ge = getKindInt vc "_GE" in 
  let kind_mult = getKindInt vc "_MULT" in 
  let kind_minus = getKindInt vc "_MINUS" in
  let kind_plus = getKindInt vc "_PLUS" in
  let kind_divide = getKindInt vc "_DIVIDE" in
  let kind_uminus = getKindInt vc "_UMINUS" in 
  let kind_pow = getKindInt vc "_POW" in
  let kind_read = getKindInt vc "_READ" in
  let kind_write = getKindInt vc "_WRITE" in
  let kind_array_var = getKindInt vc "_ARRAY_VAR" in
  let kind_constdef = getKindInt vc "_CONSTDEF" in
  let kind_bound_var = getKindInt vc "_BOUND_VAR" in
  let kind_skolem_var = getKindInt vc "_SKOLEM_VAR" in
  let kind_exists = getKindInt vc "_EXISTS" in
  let kind_forall = getKindInt vc "_FORALL" in
  let kind_distinct = getKindInt vc "_DISTINCT" in
  let kind_gray_shadow  = getKindInt vc "_GRAY_SHADOW" in
  let kind_dark_shadow =  getKindInt vc "_DARK_SHADOW" in

    
(* ====================================================================== *)
(*  Translate Types                                                       *)
(* ====================================================================== *)
  let rec translate_type vc type_exp =
    let k = kind type_exp in
    if k = kind_real then ty_real
    else if k = kind_int then if use_ints then ty_int else ty_real
    else if k = kind_bool then ty_bool
    else if k = kind_array then
      let index_type = translate_type vc (child type_exp 0) in
      let data_type = translate_type vc (child type_exp 1) in
        mk_type(array_string, [index_type;data_type])
    else if k = kind_arrow then
      (let ch1 = translate_type vc (child type_exp 0) in
       let ch2 = translate_type vc (child type_exp 1) in
	 mk_fun_ty ch1 ch2)
    else if k = kind_typedecl then ty_a
    else failwith (kind_string vc type_exp ^ " not an allowed type") in

(* ====================================================================== *)
(*  Translate Terms                                                       *)
(* ====================================================================== *)

  let arr_var_map = ref [] in

  let new_array_var,reset_array_var = 
    let arv_num = ref 0 in 
      (fun ty -> let n = !arv_num in 
		   (arv_num := n + 1;  
		    mk_var("__array_var_" ^ string_of_int n,ty))), 
    (fun () -> arv_num := 0) in
    
  let rev_assoc_cvc x l = fst(find (fun p -> exprString (snd p) = exprString x) l) in
    
  let add_array_var(var,expr) = 
    if can (rev_assoc_cvc expr) !arr_var_map then ()
    else arr_var_map := (var,expr)::!arr_var_map  in
    
  let reset_arr_map() = arr_var_map := [] in

  
  let term_size = ref 500000 in
  let term_table = Hashtbl.create !term_size in
  let distinct_table = Hashtbl.create 10 in

  (* the hash table to handle const def *)
  let const_def_tbl = Hashtbl.create 111 in
  let exists_term_table = ref [] in
  let exists_proof_table = ref [] in
  let assumptions_table = ref [] in

  
  let rec slow_translate_term vc expr = 
    let k = kind expr in
      try
      (* ---------------------------------------------------------------------- *)
      (*  Uninterpreted Functions                                               *)
      (* ---------------------------------------------------------------------- *)
      if k = kind_ufunc then 
        let name = exprString expr in
	  if (arity expr) = 0
	  then  mk_var (name,ty_bool)
	  else if (arity expr) > 1 then
	    let opname = exprString (getFun vc expr) in
	    let ret_ty = translate_type vc (typeExpr vc expr) in
	    let child =   map (translate_term vc) (child_expr_list vc expr) in
	    let par = end_itlist (curry mk_pair) child in
	    let par_ty = type_of par in
	    let ty = mk_fun_ty par_ty ret_ty in
	    let func = mk_var (opname, ty) in
	      mk_comb(func, par)   
	  else 
	    let opname = exprString (getFun vc expr) in
	    let ret_ty = translate_type vc (typeExpr vc expr) in
	    let ch0 = child expr 0 in
	    let par = translate_term vc ch0 in
	    let par_ty = translate_type vc (typeExpr vc ch0) in
	    let ty = mk_fun_ty par_ty ret_ty in
	    let func = mk_var (opname, ty) in
	      mk_comb (func, par)
      
      (* ---------------------------------------------------------------------- *)
      (*  APPLY                                                                 *)
      (* ---------------------------------------------------------------------- *)
		
    else if k = kind_apply then 
      if (arity expr) = 0
      then  failwith "APPLY got arity 0"
      else if (arity expr) > 1 then
	let opname = exprString (getFun vc expr) in
	let ret_ty = translate_type vc (typeExpr vc expr) in
	let child =   map (translate_term vc) (child_expr_list vc expr) in
	let par = end_itlist (curry mk_pair) child in
	let par_ty = type_of par in
	  
	let ty = mk_fun_ty par_ty ret_ty in
	let func = mk_var (opname, ty) in
	  mk_comb(func, par)   
      else 
	let opname = exprString (getFun vc expr) in
	let ret_ty = translate_type vc (typeExpr vc expr) in
	let ch0 = child expr 0 in
	let par = translate_term vc ch0 in
	let par_ty = translate_type vc (typeExpr vc ch0) in
	let ty = mk_fun_ty par_ty ret_ty in
	let func = mk_var (opname, ty) in
	  mk_comb (func, par)        
	    
      (* ---------------------------------------------------------------------- *)
      (*  Conditionals                                                          *)
      (* ---------------------------------------------------------------------- *)
	    
      else if k = kind_ite then  
	let test = translate_term vc (child expr 0) in
	let true_branch = translate_term vc (child expr 1) in
	let false_branch = translate_term vc (child expr 2) in
          mk_cond(test,true_branch,false_branch)
	    
      (* ---------------------------------------------------------------------- *)
      (*  BOOLEAN                                                          *)
      (* ---------------------------------------------------------------------- *)
	    
      else if k = kind_bool_var then  
	let name = exprString expr in
          mk_var (name,ty_bool)
	    
      (* ---------------------------------------------------------------------- *)
      (*  Rationals                                                             *)
      (* ---------------------------------------------------------------------- *)

      else if k = kind_rational_var then
	(
	  let type_expr  = typeExpr vc expr in
	  let type_kind = kind type_expr in
	    if (type_kind = kind_real ) 
	      or (type_kind = kind_int  && (not use_ints)) 
	    then 
	      ( let  s = exprString expr in
	      let rec trans_posreal s =
                (
		  let s_list = explode s in
		  let num_div = length (filter (fun x -> x = "/" ) s_list) in
		    if num_div = 0 then  mk_combc `(&)` (parse_term s)
		    else if num_div = 1 then
		      let ind = index "/" s_list in
		      let lhs,rhs = chop_list ind s_list in
		      let lh =  trans_posreal (implode lhs) and rh = trans_posreal (implode (tl rhs)) in
			( (*print_term (mk_real_binop  ` ( / ) ` lh rh);*)
			  mk_real_divide  lh rh)
		    else failwith "Cannot handle more than 1 divide here" 
	        ) in		  
	      let is_neg, pos = deal_real s in
		if is_neg then  
                  mk_uminus (trans_posreal pos )
		else trans_posreal s
	      )
	    else  if (type_kind = kind_int ) then 
	      (
		let s = exprString expr in
                let is_neg, pos = deal_real s in
		  if is_neg then  
                    mk_intconst (Num.num_of_string ("-" ^ pos))
		  else mk_intconst (num_of_string pos)
	      )
	    else
	      ( pr "\n Please check type in rational_expr: \n";
		print_detailed vc type_expr;
		failwith "No such type in rational expr"
	      )
	)
      (* ---------------------------------------------------------------------- *)
      (*  Boolean                                                               *)
      (* ---------------------------------------------------------------------- *)
  
        else if k = kind_true_expr then ty_true 
        else if k = kind_false_expr then ty_false
	  
      (* ---------------------------------------------------------------------- *)
      (*  Variables                                                             *)
      (* ---------------------------------------------------------------------- *)
	  
	else if k = kind_uconst then 
          let name = exprString expr in
          let ty = translate_type vc (typeExpr vc expr) in
	    (*  print_string "debug for UCONST\n"; *)
            mk_var(name,ty)
	      
       (* ---------------------------------------------------------------------- *)
       (*  Equations                                                             *)
       (* ---------------------------------------------------------------------- *)
	      
	else if k = kind_eq then 
          let left = translate_term vc (child expr 0) in
          let right = translate_term vc (child expr 1) in
	  let left_ty = type_of left in
	  let right_ty = type_of right in
	    if left_ty = right_ty then
              mk_eq (left,right)
	    else if ((is_int left) or (is_real left)) && 
		((is_int right) or (is_real right))  
	    then mk_eq ((up_term left), (up_term right))
	    else failwith "cannot mk eq "

      (* ---------------------------------------------------------------------- *)
      (*  Logical                                                               *)
      (* ---------------------------------------------------------------------- *)

      else if k = kind_not  then
        mk_neg(translate_term vc (child expr 0))
      else if k = kind_and then 
        end_itlist (curry mk_conj) (get_child_terms vc expr) 
      else if k = kind_or  then 
        end_itlist (curry mk_disj) (get_child_terms vc expr) 
      else if k = kind_iff then
        mk_eq(translate_term vc (child expr 0),
              translate_term vc (child expr 1))
      else if k = kind_implies then
        mk_imp(translate_term vc (child expr 0),
               translate_term vc (child expr 1))

      (* ---------------------------------------------------------------------- *)
      (*  Inequalities                                                          *)
      (* ---------------------------------------------------------------------- *)

      else if k = kind_lt then
	let lhs = translate_term vc (child expr 0) in
	let rhs = translate_term vc (child expr 1) in
	  if ((type_of lhs )= ty_int) then 
	    mk_lt_int  lhs rhs 
	  else 
	    mk_lt_real lhs rhs 
  
      else if k = kind_le then
	let lhs = translate_term vc (child expr 0) in
	let rhs = translate_term vc (child expr 1) in
	  if ((type_of lhs )= ty_int ) then 
	    mk_le_int lhs rhs 
	  else 
	    mk_le_real  lhs rhs 

      else if k = kind_gt then
	let lhs = translate_term vc (child expr 0) in
	let rhs = translate_term vc (child expr 1) in
	  if ((type_of lhs )= ty_int ) then 
	    mk_gt_int lhs rhs 
	  else 
	    mk_gt_real  lhs rhs 

      else if k = kind_ge  then
	let lhs = translate_term vc (child expr 0) in
	let rhs = translate_term vc (child expr 1) in

	  if ((type_of lhs )= ty_int) then 
	    mk_ge_int lhs rhs 
	  else 
	    mk_ge_real lhs rhs 

      (* ---------------------------------------------------------------------- *)
      (*  Arithmetic                                                            *)
      (* ---------------------------------------------------------------------- *)          

      else if k = kind_mult  then
	let chd = (get_child_terms vc expr) in
	let chs2,allint = up_convert chd in
	  if allint then 
            end_itlist mk_mult_int chs2
	  else           
	    end_itlist mk_mult_real chs2
      else if k = kind_minus then
	let chd = (get_child_terms vc expr) in
	let chs, allint = up_convert chd in
	  if allint then
            end_itlist mk_minus_int chs
	  else           
	    end_itlist mk_minus_real chs

      else if k = kind_plus  then
	let chd = (get_child_terms vc expr) in
	let chs , allint = up_convert chd in
	  if allint then 
            end_itlist mk_plus_int chs
	  else           
	    end_itlist mk_plus_real chs

      else if k = kind_divide  then
	let chd = (get_child_terms vc expr) in
	let chs,allint = up_convert chd in
	  if allint  then 
	    let chs = map up_term chd in 
	      end_itlist mk_divide chs  
	  else 
	    end_itlist mk_divide chs  

      else if k = kind_uminus then
	let ch = translate_term vc (child expr 0) in
	let chs,allint = up_convert [ch] in
	  if allint then 
          (curry mk_comb) uminus_int ch 
	else           
          (curry mk_comb) uminus_real (hd chs) 

      else if k = kind_pow then 
        let x = (translate_term vc (child expr 1)) in
          (* type of power could be real or num *)
        let raw_n = (translate_term vc (child expr 0)) in
        let n = if (type_of raw_n = ty_real) then num_of_real raw_n else raw_n in
          mk_pow x n

      (* ---------------------------------------------------------------------- *)
      (*  Arrays                                                                *)
      (* ---------------------------------------------------------------------- *)

      else if k = kind_read then
        let arr = translate_term vc (child expr 0) in
        let type_string,type_list = dest_type(type_of arr) in
        let ind_type,data_type = hd type_list,hd(tl type_list) in
        let read_tm = inst [(ind_type,ty_i);(data_type,ty_d)] 
          ty_read in              
        let ind = translate_term vc (child expr 1) in
          app_list read_tm [arr;ind]
	    
      else if k = kind_write  then
        let arr = translate_term vc (child expr 0) in
        let type_string,type_list = dest_type(type_of arr) in
        let ind_type,data_type = hd type_list,hd(tl type_list) in
        let write_tm = inst [(ind_type,ty_i);(data_type,ty_d)] 
          ty_write in              
        let ind = translate_term vc (child expr 1) in
        let data = translate_term vc (child expr 2) in
          app_list write_tm [arr;ind;data]
	    
      else if k = kind_array_var then 
        try 
          rev_assoc_cvc expr !arr_var_map
        with Failure _ ->
          let ty = translate_type vc (typeExpr vc expr) in 
          let hol_var = new_array_var ty in 
            add_array_var(hol_var,expr);
            hol_var

      else if  k = kind_constdef  then 
	( let the_const = translate_term vc (child expr 0) in
	  let the_def = translate_term vc (child expr 1) in
	    Hashtbl.add const_def_tbl the_def the_const;
	   the_const
	)
       else if  k = kind_bound_var then 
	(  
	  let ty = translate_type vc (typeExpr vc expr) in
	  let name = exprString expr in
            mk_var (name,ty);
	)    
       else if  k = kind_skolem_var  then 
	 (
	   let ty = translate_type vc (typeExpr vc expr) in

	   let exist = translate_term vc (getExistential expr) in
	     mk_select (dest_exists exist)	    

	 )
	   
       else if  k = kind_exists then 
	 (
	   
           let ty = translate_type vc (typeExpr vc expr) in
	   let opname = exprString  expr in
	     
           let numVar = (getNumVars  expr) in 
	     if (numVar >1) then failwith "Exits vars >1"
	     else let var = translate_term vc (getVar expr 0) in
	     let body = translate_term vc (getBody expr) in
	       mk_exists(var,body)
	 )
       else if  k = kind_forall then 
	 (
	  let body = translate_term vc (getBody expr) in
          let numVar = (getNumVars  expr) in 
	  let rec getAllVars n = match n with 
	      0 -> []
	    | n -> (translate_term vc (getVar expr (numVar -n)))::(getAllVars (n-1)) in
	  let bvs = getAllVars numVar in
	    let res = itlist (fun x y -> mk_forall(x,y)) bvs body in
	      res
	)

       else if  k = kind_distinct  then 
	 ( 
	  let ands = get_child_terms vc expr in
	  let len = length ands in
	    let def_thm = try Hashtbl.find all_distinct_table len
	      with _ -> generate_distinct_def len in
	    let def_name = "DISTINCT"^(string_of_int len) in
	    let distinct_pred = parse_term  def_name in
	      let res = rev_itlist (fun x y -> mk_icomb (y,x)) ands distinct_pred in
		Hashtbl.add distinct_table res def_thm  ;
		res
	)
	  
       else if k = kind_gray_shadow  then
	   let v = translate_term vc (child expr 0) in
	   let e = translate_term vc (child expr 1) in
	   let c1 = translate_term vc (child expr 2) in
	   let c2 = translate_term vc (child expr 3) in
	     rev_itlist (fun x y -> mk_combc y x) [v;e;c1;c2] `(GRAY_SHADOW)`   
	       
       else if k = kind_dark_shadow  then
	   (	
	     let lhs = translate_term vc (child expr 0) in
	     let rhs = translate_term vc (child expr 1) in
	       if (is_int lhs) && (is_int rhs) then
		 mk_dark_shadow_int lhs rhs
	       else if (is_real lhs) && (is_real rhs) then
		 mk_dark_shadow_real lhs rhs
	       else failwith "not same type in dark shadow"
	   )
	     
       else  raise Kind
with 
    Kind -> print_string ("No such kind: " ^ (string_of_int k) ^ "\n");
      print_string("and the expr is");
      print_expr vc expr;
      print_detailed vc expr;
      failwith "kind"
	
  | Failure x -> 
      pr "Can not translate term:\n";
      print_detailed vc expr; 
      pr "failed becasue :";
      pr x;
      pr"\n";
      failwith "tanslate_term"
and
    get_child_terms vc expr =
  let rec get_child_terms n store =
    match n with 
        0 -> store 
      | n -> get_child_terms (n-1) ((translate_term vc (child expr (n-1)))::store) in
    get_child_terms (arity expr) []
and
    translate_term vc expr = 
  try Hashtbl.find term_table expr
  with Not_found ->
    (
      let t = try slow_translate_term vc expr with Failure _ -> failwith " translate term" in
	Hashtbl.add term_table expr t;
	t
    ) in

let translate_or_term vc expr = 
  let k = kind expr in
    (get_child_terms vc expr) 
and 
  get_child_terms vc expr =
  let rec get_child_terms n store =
    match n with 
        0 -> store 
      | n -> get_child_terms (n-1) ((translate_term vc (child expr (n-1)))::store) in
    get_child_terms (arity expr) []
in
      
let var_term vc var =
  let temp = toExpr (getType vc var) in
  let res = translate_term vc temp in
      res in

let getVarList vc expr = 
  let numvars = getNumVars expr in
    let rec get_vars n =
    if n = 0 then [(var_term vc (getVar expr 0))]
    else get_vars (n-1) @  [(var_term vc (getVar expr n))]
  in get_vars (numvars -1) in

(* ====================================================================== *)
(*  Translate Proofs                                                      *)
(* ====================================================================== *)
  
let time_proof = ref false in
let max_time = ref 5.0 in
let proof_size = ref 500000 in
let proof_table = Hashtbl.create !proof_size in
let back_level = ref 1 in
let depth_level = ref 0 in

let print_name = ref false in

let (reset_count,translate_proof) = 

  (* -------------------------------  Util  ------------------------------- *)

  let get_rule_string rule = exprString(child rule 0) in
  let proof_count = ref 0 in
  let reset_count() = proof_count := 0 in
  let rec 
      
      (* ====================================================================== *)
      (*  Arith                                                                 *)
      (* ====================================================================== *)
      
      
      (* ----------------------------  mult_ineqn  ---------------------------- *)
      
      handle_mult_ineqn vc expr = 
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
    let eq = mk_eqc lhs rhs in
    let eq_mid , substs = subs_select_op eq [] in
    let eq_mid_thm = if (is_int (rand eq_mid)) then INT_ARITH eq_mid else REAL_ARITH eq_mid in 
    let eq_thm = 
      if (length substs) > 0 
      then  INST substs eq_mid_thm 
      else eq_mid_thm in
      assert ((concl eq_thm) = eq) ;
      eq_thm
(*
    let eq = mk_eqc lhs rhs in
      if (is_int (rand lhs)) then
	INT_ARITH eq
      else REAL_ARITH eq
*)
  and
      
      (* ---------------------------------------------------------------------- *)
      (* mult_eqn                                                               *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : x                                                            *)
      (*         : y                                                            *)
      (*         : z (~(z= 0))                                                            *)
      (*                                                                        *)
      (* returns : |- (x = y) = (x * z = y * z)                                 *)
      (* ---------------------------------------------------------------------- *)

      
      handle_mult_eqn vc expr =  
    let x = translate_term vc (child expr 1) in 
    let y = translate_term vc (child expr 2) in 
    let z = translate_term vc (child expr 3) in
    let chs, all_int = up_convert [x;y] in
      if all_int 
      then 
	let znz = prove_DIV_NOT_EQ_0 z in
	  SPECL[x;y] (MATCH_MP INT_NZ_RMUL znz) 
      else 
	let znz = prove_DIV_NOT_EQ_0 (up_term z) in
	  SPECL[(up_term x);(up_term y)] (MATCH_MP REAL_NZ_RMUL znz)  
  and 


      (* --------------------------  plus_predicate  -------------------------- *)
      
      handle_plus_predicate vc expr = 
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
    let eq = mk_eqc lhs rhs in
      if (is_int (rand lhs)) then INT_ARITH eq
      else REAL_ARITH eq
  and


      (* -------------------------  const_predicate  -------------------------- *)
      
      handle_const_predicate vc expr = 
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
    let eq = mk_eqc lhs rhs in
      if (is_int (rand lhs)) then INT_ARITH eq 
	else REAL_ARITH eq
  and


      (* ------------------------  canon_flatten_sum  ------------------------- *)
      
      handle_canon_flatten_sum vc expr = 
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
    let eq = mk_eqc lhs rhs in
      REAL_ARITH eq 
  and

      (* -----------------------  canon_mult_const_sum  ----------------------- *)
      
      handle_canon_mult_const_sum vc expr = 
    let c = translate_term vc (child expr 1) in
    let sum = translate_term vc (child expr 2) in
    let lhs = mk_mult c sum in
    let rhs = translate_term vc (child expr 3) in
    let eq = mk_eqc lhs rhs in
      REAL_ARITH eq 
  and

      (* ----------------------  canon_mult_const_term  ----------------------- *)
      
      handle_canon_mult_const_term vc expr = 
    let c1 = translate_term vc (child expr 1) in
    let c2 = translate_term vc (child expr 2) in
    let t = translate_term vc (child expr 3) in
    let cct = mk_mult c1 (mk_mult c2 t) in
    let prod = term_prod c1 c2 in
    let ct = mk_mult prod t in
    let eq = mk_eqc cct ct in
      REAL_ARITH eq 
  and

      (* ---------------------  canon_mult_const_const  ----------------------- *)

      handle_canon_mult_const_const vc expr = 
    let c1 = translate_term vc (child expr 1) in
    let c2 = translate_term vc (child expr 2) in
    let oc = mk_mult c1 c2 in
      REAL_RAT_REDUCE_CONV oc
  and


      (* --------------------------  canon_mult_one  -------------------------- *)

      handle_canon_mult_one vc expr = 
    let c = translate_term vc (child expr 1) in
    let oc = mk_mult real_1 c in
    let eq = mk_eqc oc c in
      REAL_ARITH eq 
  and

      (* ----------------------  canon_mult_term_const  ----------------------- *)

      handle_canon_mult_term_const vc expr = 
    let const = translate_term vc (child expr 1) in
    let body = translate_term vc (child expr 2) in
    let bc = mk_mult body const in
    let cb = mk_mult const body in
    let eq = mk_eqc bc cb in
      REAL_ARITH eq 
  and

      (* ----------------------  canon_combo_like_terms  ---------------------- *)

      handle_canon_combo_like_terms vc expr = 
    let pre = translate_term vc (child expr 1) in
    let post = translate_term vc (child expr 2) in
    let eq = mk_eqc pre post in
      REAL_ARITH eq 
  and

      (* ---------------------------------------------------------------------- *)
      (* canon_mult_mterm_mterm                                                 *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : e1 * e2                                                      *)
      (*                                                                        *)
      (* returns : |- e1 * e2 ,(value of e1 * e1)                              *)
      (* ---------------------------------------------------------------------- *)

      handle_canon_mult_mterm_mterm vc expr =  
    let prod = translate_term vc (child expr 1) in 
    let post = translate_term vc (child expr 2) in 
    let eq = mk_eqc prod post in
      REAL_ARITH eq 
  and 

      (* ---------------------------------------------------------------------- *)
      (* canon_mult                                                             *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : e1                                                           *)
      (*         : e2                                                           *)
      (*                                                                        *)
      (* returns : |- e1 = e2                                                   *)
      (* ---------------------------------------------------------------------- *)

      handle_canon_mult vc expr = 
    let e1 = translate_term vc (child expr 1) in
    let e2 = translate_term vc (child expr 2) in
      (* this is a lazy solution, if it is too slow, i will optimize it *)
    let eq = mk_eqc e1 e2 in
    let eq_mid , substs = subs_select_op eq [] in
    let eq_mid_thm = if (is_int (rand eq_mid)) then INT_ARITH eq_mid else REAL_ARITH eq_mid in 
    let eq_mid_thm = if (is_int (rand eq_mid)) then INT_ARITH eq_mid else REAL_ARITH eq_mid in 
    let eq_thm = 
      if (length substs) > 0 
      then  INST substs eq_mid_thm 
      else eq_mid_thm in
      assert ((concl eq_thm) = eq) ;
      eq_thm
	
  and

      (* ---------------------------------------------------------------------- *)
      (* canon_plus                                                             *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : e1                                                           *)
      (*         : e2                                                           *)
      (*                                                                        *)
      (* returns : |- e1 = e2                                                   *)
      (* ---------------------------------------------------------------------- *)

      handle_canon_plus vc expr = 
    let e1 = translate_term vc (child expr 1) in
    let e2 = translate_term vc (child expr 2) in
    let eq = mk_eqc e1 e2 in
    let eq_mid , substs = subs_select_op eq [] in
    let eq_mid_thm = if (is_int (rand eq_mid)) then INT_ARITH eq_mid else REAL_ARITH eq_mid in 
    let eq_thm = 
      if (length substs) > 0 
      then  INST substs eq_mid_thm 
      else eq_mid_thm in
      assert ((concl eq_thm) = eq) ;
      eq_thm

(*    let chs, all_int = up_convert [e1;e2] in
      if all_int then 
	let eq = mk_eqc e1 e2 in
	  INT_ARITH eq
      else 
	let eq = mk_eqc (up_term e1) (up_term e2) in
	  INT_ARITH eq
*)
  and

      (* -------------------------  flip_inequality  -------------------------- *)

      handle_flip_inequality vc expr = 
    let e = translate_term vc (child expr 1) in
    let ret = translate_term vc (child expr 2) in
    let _,e2 = dest_comb e in
      if( is_int e2) then
	PURE_ONCE_REWRITE_CONV [rw_flip_ineq_int_1;rw_flip_ineq_int_2] e 
      else
	PURE_ONCE_REWRITE_CONV [rw_flip_ineq_real_1;rw_flip_ineq_real_2] e 
  and

      (* -------------------------  implyEqualities  -------------------------- *)

      handle_implyEqualities vc expr = 
    let conjs = end_itlist (curry mk_conj) (get_child_terms vc (child expr 1)) in 
    let pfs  = map (fun x -> translate_proof  vc x) (get_args 3 expr) in
    let th1 = end_itlist (fun x y -> CONJ x y) pfs in
    let th1_concl = concl th1 in
    let th2 = INT_ARITH (mk_imp (th1_concl, conjs)) in
      MP th2 th1

  and


      (* ------------------------  negated_inequality  ------------------------ *)

      handle_negated_inequality vc expr = 
      let t = translate_term vc (child expr 1) in
      let ty = type_of (snd (dest_comb (snd (dest_comb t)))) in
	if ( ty = ty_int ) then 
          PURE_ONCE_REWRITE_CONV[rw_neg_ineq_int_1;rw_neg_ineq_int_2;rw_neg_ineq_int_3;rw_neg_ineq_int_4] t
	else           
	  PURE_ONCE_REWRITE_CONV[rw_neg_ineq_real_1;rw_neg_ineq_real_2;rw_neg_ineq_real_3;rw_neg_ineq_real_4] t

  and
      (* ------------------------  implyWeakerInequalityDiffLogic  ------------------------ *)
      
     handle_implyWeakerInequalityDiffLogic vc expr =
      let e1 = translate_term vc (child expr 1) in
      let pfs = map (fun x -> translate_proof vc x) (get_args 3 expr) in
      let th1 = end_itlist (fun x y -> CONJ x y) pfs in
      let th1_concl = concl th1 in
      let th2 = INT_ARITH (mk_imp (th1_concl, e1)) in
	MP th2 th1

  and
      (* ------------------------  implyWearkerInequality  ------------------------ *)
      


     handle_implyWeakerInequality vc expr =
      let e1 = translate_term vc (child expr 1) in
      let e2 = translate_term vc (child expr 2) in
      let imp = mk_imp (e1,e2) in
	if (is_int (rand e1)) then INT_ARITH imp else REAL_ARITH imp

  and
      (* ------------------------  implyNegatedInequalityDiffLogic  ------------------------ *)
      
     handle_implyNegatedInequalityDiffLogic vc expr =
      let e1 = translate_term vc (child expr 1) in
      let pfs = map (fun x -> translate_proof vc x) (get_args 3 expr) in
      let th1 = end_itlist (fun x y -> CONJ x y) pfs in
      let th1_concl = concl th1 in
      let th2 = INT_ARITH (mk_imp (th1_concl, e1)) in
	MP th2 th1

  and
      (* ------------------------  implyNegatedInequality  ------------------------ *)
      
     handle_implyNegatedInequality vc expr =
      let e1 = translate_term vc (child expr 1) in
      let e2neg = translate_term vc (child expr 3) in
      let imp = mk_imp (e1,e2neg) in
	if (is_int (rand e1)) then INT_ARITH imp else REAL_ARITH imp




  and
      (* ------------------------  cycleConflict     ------------------------ *)

      handle_cycleConflict vc expr = 
      let num_pfs = (arity expr)/2 in  
      let pfs = map (fun x -> translate_proof vc x) (get_args (num_pfs + 1) expr  ) in
      let th1 = end_itlist (fun x y -> CONJ x y) pfs in
      let th1_concl = concl th1 in
      let concl_false = mk_imp (th1_concl, ty_false) in
      let th2 = INT_ARITH concl_false in
	MP th2 th1 


  and
      (* ---------------------------  real_shadow  ---------------------------- *)
      
      handle_real_shadow vc expr = 
    let lhs_thm = translate_proof vc (child expr 3) in
    let rhs_thm = translate_proof vc (child expr 4) in
    let conj = CONJ lhs_thm rhs_thm in
    let thm = 
      if is_lt (concl lhs_thm) & is_lt (concl rhs_thm) then REAL_LT_TRANS 
      else if is_lt (concl lhs_thm) & is_le (concl rhs_thm) then REAL_LTE_TRANS 
      else if is_le (concl lhs_thm) & is_lt (concl rhs_thm) then REAL_LET_TRANS 
      else if is_le (concl lhs_thm) & is_le (concl rhs_thm) then REAL_LE_TRANS 
      else if is_lt_int (concl lhs_thm) & is_lt_int (concl rhs_thm) then INT_LT_TRANS 
      else if is_lt_int (concl lhs_thm) & is_le_int (concl rhs_thm) then INT_LTE_TRANS 
      else if is_le_int (concl lhs_thm) & is_lt_int (concl rhs_thm) then INT_LET_TRANS 
      else if is_le_int (concl lhs_thm) & is_le_int (concl rhs_thm) then INT_LE_TRANS 
      else failwith "real_shadow" in
      MATCH_MP thm conj 
  and 

      (* ---------------------------  real_shadow_eq --------------------------- *)
      
      handle_real_shadow_eq vc expr = 
    let lhs_thm = translate_proof vc (child expr 3) in
    let rhs_thm = translate_proof vc (child expr 4) in
    let conj = CONJ lhs_thm rhs_thm in
    let l =  (rand o rator) (concl lhs_thm) in
    let r =  (rand o rator) (concl rhs_thm) in
      if(is_int l) then 
	let thm = SPECL [l;r] INT_LE_ANTISYM in
          EQ_MP thm conj 
      else
	let thm = SPECL [l;r] REAL_LE_ANTISYM in
          EQ_MP thm conj 
  and 
      
      handle_lessThan_To_LE_rhs vc expr = 
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
    let less_thm  = translate_proof vc (child expr 3) in
    let dest_less =  dest_lt_int in
    let a,b = dest_less (concl less_thm) in
      if(not (is_int a) or  not (is_int b)) then 
	failwith  "not integers in lessthan" 
      else 
	let thm = SPECL [a;b] lessThanEqRhs in
	  thm
	    
  and 

      handle_lessThan_To_LE_lhs vc expr = 
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
    let less_thm  = translate_proof vc (child expr 3) in
    let dest_less = dest_lt_int  in
    let a,b = dest_less (concl less_thm) in
      if(not (is_int a) or  not (is_int b)) then 
	failwith  "not integers in lessthan" 
      else 
	let thm = SPECL [a;b] lessThanEqLhs in
	  thm

  and 
      
      handle_lessThan_To_LE_rhs_rwr vc expr = 
    let less = translate_term vc (child expr 1) in
    let le = translate_term vc (child expr 2) in
    let dest_less =  dest_lt_int in
    let a,b = dest_less less  in
      if(not (is_int a) or  not (is_int b)) then 
	failwith  "not integers in lessthan" 
      else 
	let thm = SPECL [a;b] lessThanEqRhs in
	  assert(le = (rand (concl thm)));
	  thm
	    
  and 

      handle_lessThan_To_LE_lhs_rwr vc expr = 
    let less = translate_term vc (child expr 1) in

    let le = translate_term vc (child expr 2) in
    let dest_less = dest_lt_int  in
    let a,b = dest_less less in
      if(not (is_int a) or  not (is_int b)) then 
	failwith  "not integers in lessthan" 
      else 
	let thm = SPECL [a;b] lessThanEqLhs in
	  assert(le = (rand (concl thm)));
	  thm
  and 

      (* -------------------------  right_minus_left  ------------------------- *)
      
      handle_right_minus_left vc expr =
    let t = translate_term vc (child expr 1) in
    let ty = (snd (dest_comb t)) in
      if (is_int ty) then 
        PURE_ONCE_REWRITE_CONV[rw_right_minus_left_int_1;
			       rw_right_minus_left_int_2;
			       rw_right_minus_left_int_3;
			       rw_right_minus_left_int_4;
			       rw_right_minus_left_int_5] t 
      else
        PURE_ONCE_REWRITE_CONV[rw_right_minus_left_real_1;
			       rw_right_minus_left_real_2;
			       rw_right_minus_left_real_3;
			       rw_right_minus_left_real_4;
			       rw_right_minus_left_real_5] t 
	  
  and
      
      (* --------------------------  minusToPlus  --------------------------- *)
      
      handle_minus_to_plus vc expr =
    let l = translate_term vc (child expr 1) in
    let r = translate_term vc (child expr 2) in
    let chs = [l;r] in
    let chs2,allint = up_convert chs in
      if allint then
	let res  = INST [l,int_a;r,int_b] lemma_minus_to_plus_int in
	  res
      else 
	INST [(up_term l),real_a;(up_term r),real_b] lemma_minus_to_plus_real
  and      
      

      (* ====================================================================== *)
      (*  Common                                                                *)
      (* ====================================================================== *)


      (* ---------------------------------------------------------------------- *)
      (* not_to_iff                                                             *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : e                                                            *)
      (*         : |- ~e                                                        *)
      (*                                                                        *)
      (* returns : |- e = F                                                     *)
      (* ---------------------------------------------------------------------- *)

      handle_not_to_iff vc expr =
    let not_e = translate_proof vc (child expr 2) in
    let e = translate_term vc (child expr 1) in
      EQ_MP (INST [(e,bool_a)] lemma_not_iff) not_e 
	
  and      


      (* ---------------------------------------------------------------------- *)
      (* conflict_clause                                                        *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : B                                                            *)
      (*         : !A_1 \/ ... \/ !A_n \/ B  unless B=F in which case we have   *)
      (*           !A_1 \/ ... \/ !A_n                                          *)
      (*         : Gamma, A_1,...,A_n |- B                                      *)
      (*         :                                                              *)
      (*                                                                        *)
      (* returns : Gamma |- or_expr                                             *)
      (* ---------------------------------------------------------------------- *)

      handle_conflict_clause vc expr = 
    let expr_child = child expr 1 in
    let bthm = translate_proof vc expr_child in
    let var_term = getVarList vc expr_child in
    let assums = var_term in
    let assums' = filter (fun x -> x <> ty_false ) assums in
    let itfun tm thm = DISCH tm thm in       
    let imp_thm = itlist itfun assums' bthm in
    let res2 = PURE_REWRITE_RULE[lem9__;lem10__;lem11__;lem12__] imp_thm  in
      res2
  and 

      (* ---------------------------------------------------------------------- *)
      (* unit_prop                                                              *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : index                                                        *)
      (*         :  ~e_1\/~e_2\/ .\/e_i\/.. \/ ~e_k                                *)
      (*         : G_1 |- e_1,...,G_k |- e_k, clause_thm                        *)
      (*                                                                        *)
      (* returns : /\{G_i} |- e_i                                             *)
      (* ---------------------------------------------------------------------- *)

      handle_unit_prop vc expr = 
    
    let name = "unit_prop" in
      failwith "Error: unit_prop no longer supported." 
(*
    let rule = get_rule_string expr in   
      (* assert (name = rule); *) 
      let ar = arity expr in
      let num_thms = (ar - 2)/2 in 
      let thms = ref [] in
	for i = (2 + num_thms) to (ar - 1) do
	  let proof = translate_proof vc (child expr i) in
	    thms := proof::!thms
	done;
	let or_thm = hd !thms in
	let neg_thms = rev(tl !thms) in
	  REPEAT_DISJ_ELIM or_thm neg_thms
*)
  and

      (* ---------------------------------------------------------------------- *)
      (* conflict                                                               *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : e_1,...,e_k,c1 \/ ... \/ c_k                                 *)
      (*         : G_1 |- !e_1,...,G_n |= !e_n                                  *)
      (*                                                                        *)
      (* returns : |- {G_i} |- F                                                *)
      (* ---------------------------------------------------------------------- *)

      handle_conflict vc expr = 
    failwith "Error: conflict no longer supported." 
(*
    let name = "conflict" in
    let rule = get_rule_string expr in   
      (* assert (ame = rule); *) 
      let ar = arity expr in
      let num_thms = (ar - 1)/2 in 
      let thms = ref [] in
	for i = (1 + num_thms) to (ar - 1) do
	  let proof = translate_proof vc (child expr i) in
	    thms := proof::!thms
	done;
	let or_thm = hd !thms in
	let neg_thms = rev(tl !thms) in
	  REPEAT_DISJ_ELIM or_thm neg_thms
*)
  and

      (* ---------------------------------------------------------------------- *)
      (* iff_contrapositive                                                     *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : |- a = b                                                     *)
      (*                                                                        *)
      (* returns : |- ~a = ~b                                                   *)
      (* ---------------------------------------------------------------------- *)
      
      handle_iff_contrapositive vc expr = 
    let aEQb = translate_proof vc (child expr 2) in
    let lhs,rhs = (dest_eq o concl) aEQb in
    let thm = lemma_iff_contro  in
    let ret = EQ_MP (ISPECL [lhs;rhs] thm) aEQb in
      (* slight hack here... cvc calls negate, not ~ *)
      PURE_REWRITE_RULE[dneg] ret
          
  and

      (* ---------------------------------------------------------------------- *)
      (* neg_intro                                                              *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : ~ a                                                          *)
      (*         : a |- F                                                       *)
      (*                                                                        *)
      (* returns : |- ~a                                                        *)
      (* ---------------------------------------------------------------------- *)

      handle_neg_intro vc expr =    
    let NOT_RULE a_imp_f = 
      let a = fst(dest_imp(concl a_imp_f)) in
      let t1 = MK_COMB (SYM NOT_DEF,REFL a) in
      let t2 = (LAND_CONV BETA_CONV) (concl t1) in
      let t3 = EQ_MP t2 t1 in 
      let t4 = EQ_MP t3 a_imp_f in
        t4 in
    let not_a = translate_term vc (child expr 1) in  
    let a_proves_f = translate_proof vc (child expr 2) in  
    let a_imp_f = DISCH (hol_negate not_a) a_proves_f in
      NOT_RULE a_imp_f
	
  and 

      (* -----------------------------  iff_refl  ----------------------------- *)
      
      handle_iff_refl vc expr =    
    let e = translate_term vc (child expr 2) in
      REFL e
  and 

      (* -----------------------------  refl  ----------------------------- *)
      
      handle_refl vc expr =    
    let e = translate_term vc (child expr 1) in
      REFL e
  and 

      (* -----------------------------  eq_refl  ----------------------------- *)
      
      handle_eq_refl vc expr =    
    let e = translate_term vc (child expr 2) in
      REFL e
	
  and 

      (* ---------------------------  rewrite_and  ---------------------------- *)
(*
	handle_rewrite_and vc expr =     
	let name = "rewrite_and" in    
	let rule = get_rule_string expr in    
	(* assert (name = rule); *)
	let lhs = translate_term vc (child expr 1) in
	let rhs = translate_term vc (child expr 2) in
        TAUT (mk_eqc lhs rhs)
*)  
      handle_rewrite_and vc expr =     
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
      if rhs = ty_false then 
	let th1 = gen_pos_neg lhs true in
	let th2 = REWRITE_CONV [rewrite_and_thm] (rand (concl th1)) in
	let th3 = TRANS th1 th2 in
	  try
	    if (rhs <> rand (concl th3)) then failwith "strange case"  
	    else th3  
	  with Failure _ ->
	    pr "\n -------- strange case in rewrite_and ---------- \n";
	    TAUT (mk_eq (lhs,rhs)) 
      else
	try  CONJ_ACI_RULE (mk_eqc lhs rhs)
	with Failure x -> 
	  let thl = REWRITE_CONV [] lhs in
	  let thr = REWRITE_CONV [] rhs in
	  let ll,lr = dest_iff (concl thl) in
	  let rl,rr = dest_iff (concl thr)in
	    try 
	      let th = CONJ_ACI_RULE (mk_eqc lr rr) in
		TRANS (TRANS thl th) (SYM thr) 
	    with Failure x ->
	      pr "\n -------- strange case in rewrite_and ---------- \n";
	      TAUT (mk_eqc lhs rhs)  
		
  and  

      (* ----------------------------  rewrite_or  ---------------------------- *)

      handle_rewrite_or vc expr =
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
      if rhs = ty_true then 
	let th1 = gen_pos_neg lhs false in
	let th2 = REWRITE_CONV [rewrite_or_thm] (rand (concl th1)) in
	let th3 = TRANS th1 th2 in
	  try
	    if (rhs <> rand (concl th3)) then failwith "strange case"  
	    else th3  
	  with Failure _ ->
	    pr "\n -------- strange case 2 in rewrite_or ---------- \n";
	    TAUT (mk_eq (lhs,rhs)) 
      else
	try DISJ_ACI_RULE (mk_eqc lhs rhs)
	with Failure x -> 
	  let thl = REWRITE_CONV [] lhs in
	  let thr = REWRITE_CONV [] rhs in
	  let ll,lr = dest_iff (concl thl) in
	  let rl,rr = dest_iff (concl thr)in
	    try 
	      let th = DISJ_ACI_RULE (mk_eqc lr rr) in
		TRANS (TRANS thl th) (SYM thr) 
	    with Failure x ->
	      pr "\n -------- strange case 1 in rewrite_or ---------- \n";
	      TAUT (mk_eqc lhs rhs)  
		
  and
      
      (* -------------------------  rewrite_iff  ------------------------- *)

      handle_rewrite_iff vc expr = 
    let e0 = translate_term vc (child expr 1) in
    let e1 = translate_term vc (child expr 2) in
      if      ( ty_true = e0 ) then INST [ (e0,bool_e0);(e1,bool_e1)] lemma_rewrite_1
      else if ( ty_false = e0 ) then INST [ (e0,bool_e0);(e1,bool_e1)] lemma_rewrite_2
      else if ( (is_neg e0) & (e1 = (dest_neg e0))) then  INST [ (e0,bool_e0);(e1,bool_e1)]  lemma_rewrite_5
      else if ( ty_true = e1 ) then INST [ (e0,bool_e0);(e1,bool_e1)] lemma_rewrite_3
      else if ( ty_false = e1 ) then INST [ (e0,bool_e0);(e1,bool_e1)] lemma_rewrite_4
      else if ( (is_neg e1) & (e0 = (dest_neg e1))) then  INST [ (e0,bool_e0);(e1,bool_e1)]  lemma_rewrite_6
      else failwith "unsupported case in rewrite_iff" 
        
  and

      (* -------------------------  rewrite_iff_symm  ------------------------- *)  

      handle_rewrite_iff_symm vc expr = 
    let e0 = translate_term vc (child expr 1) in
    let e1 = translate_term vc (child expr 2) in
      INST [(e0,bool_a);(e1,bool_b)] lemma_rewrite_iff_symm
	
  and

      (* -------------------------  false_implies_anything  ------------------------- *)

      handle_false_implies_anything vc expr = 
    let pFalse = translate_proof vc (child expr 2) in
    let result = translate_term vc (child expr 1) in
      if ty_false = (concl pFalse) then
	let tp = TAUT (mk_imp (ty_false,result)) in
	  MP  tp pFalse 
      else failwith "false_implies_anytihng, FALSE is not proved"
	  
  and
      
      (* -------------------------  rewrite_constdef ------------------------- *)
      handle_rewrite_constdef vc expr =
    let the_def = translate_term vc (child expr 1) in
      print_string "Debug for rewrite_constdef\n";
      let the_const = Hashtbl.find const_def_tbl the_def in
	ASSUME (mk_eqc the_const the_def)
  and

      (* -------------------------  rewrite_not_ite ------------------------- *)

      handle_rewrite_not_ite vc expr =
    let not_ite = child expr 1 in
    let ite = child not_ite 0 in 
    let ite_cond = translate_term vc (child ite 0) in
    let ite_true = translate_term vc (child ite 1) in
    let ite_false = translate_term vc (child ite 2) in
      INST [(ite_cond,bool_a);(ite_true,bool_b);(ite_false,bool_c)] lemma_rewrite_not_ite
  and 
      
      (* ------------------------- uminus_to_mult ------------------------- *)
      
      handle_uminus_to_mult vc expr = 
    let e = translate_term vc (child expr 1) in
      if (is_int e) then  INST [e,int_a] INT_UMINUS_TO_MULT
      else INST [e,real_a ] REAL_UMINUS_TO_MULT
  and 

      (* ------------------------- rewrite_ite_same_iff  ------------------------- *)
      
      handle_rewrite_ite_same_iff  vc expr = 
    let e = translate_term vc (child expr 1) in
    let branch = translate_term vc (child expr 2) in
      ISPECL [e;branch] lemma_ite_same_iff
	
  and 


      (* ------------------------- rewrite_ite_same      ------------------------- *)
      
      handle_rewrite_ite_same  vc expr = 
    let e = translate_term vc (child expr 2) in
    let branch = translate_term vc (child expr 3) in 
      ISPECL [e;branch] lemma_ite_same
  and 

      (* ------------------------- cannon_invert_divide  ------------------------- *)
      
      handle_canon_invert_divide  vc expr = 
    let e = child expr 1 in
    let lhs = translate_term vc e in
      if (kind e) = getKindInt vc "_DIVIDE" then(
	  let e0 =up_term (translate_term vc (child e 0)) in
	  let e1 =up_term (translate_term vc (child e 1)) in
	  let rhs = mk_mult_real e0 ( mk_real_divide real_1  e1)   in
   	    REAL_ARITH (mk_eqc lhs rhs)  )
      else failwith "Cannot handle this in canon_invert_divide" 
  and 

      (* -------------------------  rewrite_iff_refl  ------------------------- *)

      handle_rewrite_iff_refl vc expr = 
    let tm = translate_term vc (child expr 1) in
      INST [(tm,bool_a)] lemma_rewrite_iff_refl
  and
      
      (* -----------------------------  iff_true  ----------------------------- *)
      
      handle_iff_true vc expr = 
    let pf = translate_proof vc (child expr 2) in
      EQT_INTRO pf
  and

      (* -----------------------------  iff_not_not  ----------------------------- *)

      handle_rewrite_not_not vc expr = 
    let thm = lemma_not_not  in
    let e = translate_term vc (get vc expr [1]) in
      ISPEC e thm    
	
  and


      (* ---------------------------------------------------------------------- *)
      (* iff_true_elim                                                          *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : |- e = T                                                     *)
      (*                                                                        *)
      (* returns : |- e                                                         *)
      (* ---------------------------------------------------------------------- *)

      handle_iff_true_elim vc expr = 
    let pf = translate_proof vc (child expr 2) in
      EQT_ELIM pf
  and


      (* ---------------------------------------------------------------------- *)
      (* iff_false_elim                                                          *)
      (*                                                                        *)
      (* args    :                                                              *)
      (*         : |- e = F                                                     *)
      (*                                                                        *)
      (* returns : |- ~e                                                         *)
      (* ---------------------------------------------------------------------- *)

      handle_iff_false_elim vc expr = 
    let pf = translate_proof vc (child expr 2) in
      EQF_ELIM pf

  and


      (* ---------------------------------------------------------------------- *)
      (* eq_trans                                                               *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : :A                                                           *)
    (*         : a1:A                                                         *)
    (*         : a2                                                           *)
    (*         : a3                                                           *)
    (*         : |- a1 = a2                                                   *)
    (*         : |- a2 = a3                                                   *)
    (*                                                                        *)
    (* returns : |- a1 = a3                                                   *)
    (* ---------------------------------------------------------------------- *)

      handle_eq_trans vc expr = 
    let t1 = translate_term vc (child expr 2) in
    let t2= translate_term vc (child expr 3) in
    let t3 = translate_term vc (child expr 4) in
    let t1_t2 = translate_proof vc (child expr 5) in
    let t2_t3 = translate_proof vc (child expr 6) in
      TRANS t1_t2 t2_t3
  and


    (* ---------------------------------------------------------------------- *)
    (* iff_trans                                                              *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : a1                                                           *)
    (*         : a2                                                           *)
    (*         : a3                                                           *)
    (*         : |- a1 = a2                                                   *)
    (*         : |- a2 = a3                                                   *)
    (*                                                                        *)
    (* returns : |- a1 = a3                                                   *)
    (* ---------------------------------------------------------------------- *)

      handle_iff_trans vc expr =
    let t1_t2 = translate_proof vc (child expr 4) in
    let t2_t3 = translate_proof vc (child expr 5) in
      try  TRANS t1_t2 t2_t3
      with Failure x ->
	(
	  try
	    let t21 = rand (concl t1_t2) in
	    let t22 = rand (rator (concl t2_t3)) in
	    let th = INT_ARITH (mk_eqc t21 t22) in
	      TRANS (TRANS t1_t2 th) t2_t3 
	  with Failure x -> 
	    failwith "Error in iff_trans" ;
	)
  and


    (* ---------------------------------------------------------------------- *)
    (* rewrite_eq_symm                                                        *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : :A                                                           *)
    (*         : a:A                                                          *)
    (*         : b:A                                                          *)
    (*                                                                        *)
    (* returns : |- (a = b) = (b = a)                                         *)
    (* ---------------------------------------------------------------------- *)

      handle_rewrite_eq_symm vc expr = 
    let a = translate_term vc (child expr 2) in 
    let b = translate_term vc (child expr 3) in 
      ISPECL[a;b] SYMM_THM
          
  and



    (* ---------------------------------------------------------------------- *)
    (* subst_op                                                               *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : f(t1,t2,...,tn)                                              *)
    (*         : f(r1,r2,...,rn)                                              *)
    (*         : |- t1 = r1, |- t2 = r2, ..., |- tn = rn                      *)
    (*                                                                        *)
    (* returns : |- f(t1,...,tn) = f(r1,...,rn)                               *)
    (* ---------------------------------------------------------------------- *)

      handle_subst_op vc expr = 
    let pre_term = translate_term vc (child expr 1) in
    let post_term = translate_term vc (child expr 2) in 
    let expected_tm = mk_eqc pre_term post_term in
      
    let rewrites = map (fun x -> (translate_proof vc x)) (get_args 3 expr) in
    let  temp_res =  special_SUBS_CONV rewrites post_term pre_term in
    let  _, temp_concl = dest_eq (concl temp_res) in
      if temp_concl  = post_term then temp_res else
	(let res_eqc = mk_eqc temp_concl post_term in
	   try  TRANS temp_res (prove (res_eqc, INT_ARITH_TAC ))
	   with Failure x -> print_string x ; failwith "Wrong in subst_op"
	)
  and
      (* ---------------------------------------------------------------------- *)
      (* subst_op0                                                              *)
      (*                                                                        *)
      (* args    : e(p)                                                         *)
      (*         : e(d)                                                         *)
      (*         : p == d                                                       *)
      (*                                                                        *)
      (* returns : |- e(p) == e(d)                                              *)
      (* ---------------------------------------------------------------------- *)
      (*      
	      handle_subst_op0 vc expr = 
	      let name = "subst_op0" in 
	      failwith "subst_op0 no longer used";
	      let rule = get_rule_string expr in 
      (* assert (name = rule); *) 
	      let e_p  = translate_term vc (child expr 1) in
      (*      let epd_thm  = translate_proof vc (child expr 2) in *)
	      REFL e_p
	      and
      *)
      
    (* ---------------------------------------------------------------------- *)
    (* andE                                                                   *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : i                                                            *)
    (*         : |- a1 /\ a2 /\ ... /\ an                                     *)
    (* returns : |- ai                                                        *)
    (* ---------------------------------------------------------------------- *)

      handle_andE vc expr = 
    let index = getInt(child expr 1) in
    let and_thm = translate_proof vc (child expr 3) in
      el index (CONJUNCTS and_thm)
  and


    (* ---------------------------------------------------------------------- *)
    (* andE                                                                   *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : i                                                            *)
    (*         : |- a1 /\ a2 /\ ... /\ an                                     *)
    (* returns : |- ai                                                        *)
    (* ---------------------------------------------------------------------- *)

    handle_andI vc expr = 
    let children = get_child_list expr in
    let pfs = map (fun x -> translate_proof vc x) (tl (tl children)) in
    let res =   end_itlist (fun x y -> CONJ x y) pfs  in
      res
	
  and

      handle_int_const_eq vc expr = 
    let lhs = translate_term vc (child expr 1) in
    let rhs = translate_term vc (child expr 2) in
    let eq = mk_eqc lhs rhs in
    let eq_mid , substs = subs_select_op eq [] in
    let eq_mid_thm = if (is_int (rand (rand eq_mid))) then INT_ARITH eq_mid else REAL_ARITH eq_mid in 
    let eq_thm = 
      if (length substs) > 0 
      then  INST substs eq_mid_thm 
      else eq_mid_thm in
      assert ((concl eq_thm) = eq) ;
      eq_thm
(*
    let res = INT_ARITH (mk_eqc lhs rhs) in 
      res
*)
  and

    (* --------------------------  rewrite_not_or  -------------------------- *)
    (* rewrite_not_or(NOT (OR e1 ... en)) --> (|- ~e1 /\ ~e2 /\ ... /\ ~en) *)
    (* note: double negations are removed *)
    (* note: see note for DIST_NOT_AND *)
      handle_rewrite_not_or vc expr = 
    let tm = child expr 1 in 
    let tm' = translate_term vc tm in
      DIST_NOT_OR tm'
  and

    (* -------------------------  rewrite_not_and  -------------------------- *)

    (* rewrite_not_and(NOT (AND e1 ... en)) --> (|- ~e1 \/ ~e2 \/  ... \/ ~en) *)
    (* note: double negations are removed *)
    (* note: see note for DIST_NOT_AND *)
    handle_rewrite_not_and vc expr = 
    let tm = child expr 1 in 
    let tm' = translate_term vc tm in
      try DIST_NOT_AND tm'
      with x -> print_expr vc expr;failwith "rewrite_not_and"
  and    
   
    (* ---------------------------------------------------------------------- *)
    (*  rewrite_ite_true_iff                                                  *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : e1                                                           *)
    (*         : e2                                                           *)
    (*                                                                        *)
    (* returns : |- IF TRUE THEN E1 ELSE E2 = E1                              *)
    (* ---------------------------------------------------------------------- *)

     handle_rewrite_ite_true_iff vc expr = 
    let e1 = child expr 1 in 
    let e2  = child expr 2 in
    let e1_term = translate_term vc e1 in
    let e2_term = translate_term vc e2 in
      ISPECL [e1_term;e2_term] lemma_rewrite_ite_true_iff
	
  and  
    (* ---------------------------------------------------------------------- *)
    (*  rewrite_ite_false_iff                                                 *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : e1                                                           *)
    (*         : e2                                                           *)
    (*                                                                        *)
    (* returns : |- IF FALSE THEN E1 ELSE E2 = E2                             *)
    (* ---------------------------------------------------------------------- *)

     handle_rewrite_ite_false_iff vc expr = 
    let e1 = child expr 1 in 
    let e2  = child expr 2 in
    let e1_term = translate_term vc e1 in
    let e2_term = translate_term vc e2 in
      
      ISPECL [e1_term;e2_term] lemma_rewrite_ite_false_iff

  and 
    (* ---------------------------------------------------------------------- *)
    (* rewrite_eq_refl                                                        *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : :A                                                           *)
    (*         : x:A                                                          *)
    (*                                                                        *)
    (* returns : |- (x = x) = T                                               *)
    (* ---------------------------------------------------------------------- *)

    handle_rewrite_eq_refl vc expr = 
    let x = translate_term vc (child expr 2) in
      EQT_INTRO(REFL x)
  and
      
    (* -------------------------  rewrite_implies  -------------------------- *)

    (* rewrite_implies(a,b) --> (|- (a ==>b) = (~a \/b) ) *)
      handle_rewrite_implies vc expr =
    let tm1 = translate_term vc (child expr 1) in
    let tm2 = translate_term vc (child expr 2) in
    let taut = lemma_rewrite_implies in
      INST[(tm1,bool_a);(tm2,bool_b)] taut
  and 

    (* -----------------------------  iff_symm  ----------------------------- *)

    (* iff_symm(a,b,(|- a = b)) --> (|- b = a) *)
      handle_iff_symm vc expr = 
    let aEQb_proof = child expr 3 in
    let aEQb_thm = translate_proof vc aEQb_proof in
      SYM aEQb_thm
  and 

    (* ---------------------------------------------------------------------- *)
    (* eq_symm                                                                *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : ty                                                           *)
    (*         : a1                                                           *)
    (*         : a2                                                           *)
    (*         : |- a1 = a2                                                   *)
    (*                                                                        *)
    (* returns : |- a2 = a1                                                   *)
    (* ---------------------------------------------------------------------- *)

      handle_eq_symm vc expr = 
    let aEQb = translate_proof vc (child expr 4) in
      SYM aEQb
  and 

    (* -------------------------  rewrite_not_true  ------------------------- *)

    (* rewrite_not_true() --> (|- ~T = F)  *)
    handle_rewrite_not_true vc expr =
    let thm = lemma_not_true in
      thm
  and

    (* ------------------------  rewrite_not_false  ------------------------- *)


    (* rewrite_not_false() --> (|- ~F = T)  *)
    handle_rewrite_not_false vc expr =
    let thm = lemma_not_false in
      thm
  and

    (* ---------------------- or_distribuitivity_rule ----------------------- *)


      handle_or_distribuitivity_rule vc expr =
    let chls = child_expr_list vc (child expr 1) in
    let a = translate_term vc (child (hd chls) 0) in
    let bs = map (fun x -> translate_term vc (child x 1)) chls in
    let fakebs,fake_thm = or_dis_thm (length chls) in
    let insts = map2 (fun x y -> (x,y)) bs fakebs in
    let res = INST ([a,bool_a]@ insts)  fake_thm in
      res
  and


    (* ---------------------------------------------------------------------- *)
    (* iff_mp                                                                 *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : p                                                            *)
    (*         : p = q                                                        *)
    (*         : |- p                                                         *)
    (*         : |- p = q                                                     *)
    (* returns : |- q                                                         *)
    (* ---------------------------------------------------------------------- *)

    
      handle_iff_mp vc expr = 
    
    let p = translate_proof vc (child expr 3) in
    let p_eq_q =  translate_proof vc (child expr 4)   in
      EQ_MP p_eq_q p
	
  and

      handle_impl_mp vc expr = 
    let e1 = translate_term vc (child expr 1) in
    let e2 = translate_term vc (child expr 2) in
    let pe1 = translate_proof vc (child expr 3) in
    let pe1e2 = translate_proof vc (child expr 4)   in
      MP pe1e2 pe1
	
  and

    
    (* ------------------------------  lambda  ------------------------------ *)

    handle_lambda vc expr = 
    assert (isLambda vc expr);
    let body = getBody expr in
      translate_proof vc body 
  and

    (* ---------------------------------------------------------------------- *)
    (* pf_by_contradiction                                                    *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : a                                                            *)
    (*         : |- ~a |- F                                                   *)
    (* returns : |- a                                                         *)
    (* ---------------------------------------------------------------------- *)
    
    handle_pf_by_contradiction vc expr = 
    let goal_term = translate_term vc (child expr 1) in
      ( 
        let na_imp_f = child expr 2 in 
	let isl = isLambda vc na_imp_f in
	let contr_thm = 
          if (isLambda vc na_imp_f) then handle_lambda vc na_imp_f
          else translate_proof vc na_imp_f in
          CCONTR goal_term contr_thm    
      )
  and

    (* ---------------------------------------------------------------------- *)
    (* rewrite_ite_true                                                       *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : if T then e1 else e2                                         *)
    (*                                                                        *)
    (* returns : |- if T then e1 else e2 = e1                                 *)
    (* ---------------------------------------------------------------------- *)


      handle_rewrite_ite_true vc expr = 
    let e1 = translate_term vc (child expr 2) in
    let e2 = translate_term vc (child expr 3) in
      ISPECL [e1;e2] lemma_ite_true
  and

    (* ---------------------------------------------------------------------- *)
    (* rewrite_ite_false                                                      *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : if F then e1 else e2                                         *)
    (*                                                                        *)
    (* returns : |- if F then e1 else e2 = e1                                 *)
    (* ---------------------------------------------------------------------- *)


      handle_rewrite_ite_false vc expr = 
    let e1 = translate_term vc (child expr 2) in
    let e2 = translate_term vc (child expr 3) in
      ISPECL [e1;e2] lemma_ite_false 

  and 
    (* ---------------------------------------------------------------------- *)
    (* var intro                                                              *)
    (* ---------------------------------------------------------------------- *)

      
      handle_var_intro vc expr = 
    let phi = translate_term vc (child expr 1) in
    let boundVar = translate_term vc (child expr 2) in
    let bvtype = translate_type vc (typeExpr vc (child expr 2)) in
    let my_y = mk_var ("y",bvtype)  in
    let th = prove(mk_exists(boundVar,mk_eq (my_y,boundVar)), MESON_TAC[]) in
    let result = INST [(phi,my_y)] th in
      (exists_proof_table := result :: !exists_proof_table; result)
	
  and

    (* ====================================================================== *)
    (*  Arrays                                                                *)
    (* ====================================================================== *)

    (* ---------------------------------------------------------------------- *)
    (* rewrite_same_store                                                     *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : write(S, i,v) = S                                            *)
    (*                                                                        *)
    (* returns : |- (write(S, i,v) = S) = read(store,i) = v                   *)
    (* ---------------------------------------------------------------------- *)
    
      handle_rewrite_same_store vc expr = 
    let write = get vc expr [1;0] in
    let s = translate_term vc (get vc write [0]) in
    let i = translate_term vc (get vc write [1]) in
    let v = translate_term vc (get vc write [2]) in
      ISPECL[s;i;v] rewrite_same_store2

  and

    (* ---------------------------------------------------------------------- *)
    (* rewriteRedundantWrite1                                                 *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : write(store,index,value)                                     *)
    (*         : |- value = read(store, index)                                *)
    (*                                                                        *)
    (* returns : |- write(store,index,value) = store                          *)
    (* ---------------------------------------------------------------------- *)

      handle_rewrite_redundant_write_1 vc expr = 
    let v_eq_r = translate_proof vc (child expr 2) in
      MATCH_MP redundant1 v_eq_r
  and

    (* ---------------------------------------------------------------------- *)
    (* rewriteRedundantWrite2                                                 *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : write(write(store, index, v1), index, v2)                    *)
    (*                                                                        *)
    (* returns : |- write(write(store, index, v1), index, v2) =               *) 
    (*                 write(store, index, v2)                                *)
    (* ---------------------------------------------------------------------- *)

      handle_rewrite_redundant_write_2 vc expr = 
    let store = translate_term vc (get vc  expr [1;0;0]) in
    let index = translate_term vc (get vc expr [1;1]) in
    let v1 = translate_term vc (get vc expr [1;0;2]) in
    let v2 = translate_term vc (get vc expr [1;2]) in
      ISPECL [store;index;v1;v2] redundant2

  and

    (* ---------------------------------------------------------------------- *)
    (* renameRead                                                             *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : read                                                         *)
    (*         : rename_var                                                   *)
    (*                                                                        *)
    (* returns : read = rename_var |- read = rename_var                       *)
    (* ---------------------------------------------------------------------- *)

      handle_rename_read vc expr =  
    let read = translate_term vc (child expr 1) in 
    let ty = translate_type vc (typeExpr vc (child expr 2)) in 
    let arr_var = child expr 2 in 
    let hol_var = new_array_var ty in 
      add_array_var(hol_var,arr_var);
      failwith "why";
      ASSUME (mk_eqc read hol_var) 

  and 

    (* ---------------------------------------------------------------------- *)
    (* rewriteReadWrite                                                       *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : read(write(store, index1, value), index2)                    *)
    (*                                                                        *)
    (* returns : |- read(write(store, index1, value), index2) =               *)
    (*                ite(index1 = index2, value, read(store, index2))        *)
    (* ---------------------------------------------------------------------- *)

      handle_rewrite_read_write vc expr = 
    let read_expr = child expr 1 in
    let store = translate_term vc (get vc read_expr [0;0]) in
    let index1 = translate_term vc (get vc read_expr [0;1]) in
    let index2 = translate_term vc (get vc read_expr [1]) in
    let value = translate_term vc (get vc read_expr [0;2]) in
      ISPECL [store;index1;index2;value] read_write
	
  and


    (* ---------------------------------------------------------------------- *)
    (* rewriteWriteWrite                                                      *)
    (*                                                                        *)
    (* args    :                                                              *)
    (*         : e                                                            *)
    (*                                                                        *)
    (* returns : |-                                                           *)
    (* ---------------------------------------------------------------------- *)

      handle_rewrite_write_write vc expr = 
    let lhs,rhs = get vc expr [1;0],get vc expr [1;1] in
    let s1 = translate_term vc (get vc lhs [0]) in
    let i1 = translate_term vc (get vc lhs [1]) in
    let v1 = translate_term vc (get vc lhs [2]) in
    let s2 = translate_term vc (get vc rhs [0]) in
    let i2 = translate_term vc (get vc rhs [1]) in
    let v2 = translate_term vc (get vc rhs [2]) in
      ISPECL [s1;s2;i1;i2;v1;v2] rewrite_write_write

  and

    (* ---------------------------------------------------------------------- *)
    (*                                                                        *)
    (* (write (write store index1 v1) index2 v2) =                            *)
    (*  write (write store index2 v2) index1                                  *)
    (*            (if (index1 = index2) then v2 else v1                       *)                 
    (*                                                                        *)
    (* ---------------------------------------------------------------------- *)

      handle_interchangeindices vc expr = 
    let store = translate_term vc (get vc expr [1;0;0]) in
    let index1 = translate_term vc (get vc expr [1;0;1]) in
    let index2 = translate_term vc (get vc expr [1;1]) in
    let v1 = translate_term vc (get vc  expr [1;0;2]) in
    let v2 = translate_term vc (get vc expr [1;2]) in

      ISPECL [store;index1;index2;v1;v2] interchangeindices

  and
      
      handle_minisat_proof vc expr = 
    let result = translate_term vc (child expr 1) in
    let proof = translate_proof vc (child expr 2) in
    let curResult = concl proof in
      if (result <> curResult) 
      then 
	if(curResult = ty_false) 
	then 
	  let move_term = 
	    let hyp_list = hyp proof in
	      if exists (fun x -> x = (hol_negate result)) hyp_list then (hol_negate result)
	      else if exists (fun x -> x = mk_neg result) hyp_list then (mk_neg result) 
	      else failwith "cannot find the desired term in hyp list" in
	  let fixedThm = IMP_OR move_term proof in
	    if (concl fixedThm = result) 
	    then fixedThm
	    else (
		print_thm fixedThm ;
		print_string "\n";
		failwith "failed to fix minisat proof"
	    )
	else (
	    print_thm proof;
	    print_string "\n";
	    print_flush();
	    failwith "Pleasea debug minisat proof"
	)
      else proof

  and
      handle_normalizeQuant vc expr = 
    let before = translate_term vc (child expr 1) in
    let after = translate_term vc (child expr 2) in
      ALPHA before after
	
  and
      
      handle_bool_resolution vc expr = 
    let e = translate_term vc (child expr 1) in
    let negE = hol_negate e in
    let leftThm = translate_proof vc (child expr 2) in
    let rightThm = translate_proof vc (child expr 3) in
    let leftTmpThm = pre_bool leftThm and 
	rightTmpThm = pre_bool rightThm in
    let leftFinalThm = try safe_disch negE leftTmpThm with Failure _ -> debug_table := e::(hyp leftThm) ; Gc.compact(); print_name := true ; debug1 := [vc, expr] ; failwith "left failed"  and
	rightFinalThm = try safe_disch  e rightTmpThm  with Failure _-> debug_table := [] ; Gc.compact(); print_name := true ;failwith "right failed" in
    let taut = 
      if (is_neg e) 
      then INST [(negE,bool_a)] bool_final_pos_2 
      else INST [(e,bool_a)] bool_final_neg_2 in
    let tmpth0 = EQ_MP taut leftFinalThm in
    let tmpth1 = EQ_MP tmpth0 rightFinalThm in
    let final = tmpth1 in
      assert(concl final = ty_false) ;
      final
	
  and
      
      handle_assumptions vc expr = 
    let hol_expr = translate_term vc (child expr 2) in
      ( assumptions_table :=  !assumptions_table @ [hol_expr] ; 
	ASSUME hol_expr)
	
  and
      
      handle_CNF vc expr = 
    let reason = exprString (child expr 1) in
    let before = translate_term vc (child expr 2) in
    let after = translate_term vc (child expr 3) in
    let idx = getInt (child expr 4) in
      if(reason = string_or_final) 
	then
	let t = cnf_or_final before in
	let ors =(hol_negate before)::(get_child_terms vc (child expr 2)) in 
	let ret = rec_trans_cnf t ors in
	  ret
      else if (reason = string_and_final ) 
      then
	let ands =get_child_terms vc (child expr 2) in 
	let th0 = ASSUME (hol_negate before) in
	let ret = cnf_and_final th0 ands in
	  (*	    assert (`F:bool` = (concl ret));*)
	  ret
      else if (reason = string_ite )
      then failwith "why ite is here"
      else if (reason = string_iff )
      then
	(	  let arg0 = translate_term vc (child (child expr 2) 0) in
	let arg1 = translate_term vc (child (child expr 2) 1) in
	  if 0 = idx then 
	    let thm =  INST [(arg0,bool_a);(arg1,bool_b)] cnfthm_iff_0 in
	    let ret = PURE_REWRITE_RULE [dneg] thm  in
	      rec_trans_cnf ret [before; arg0; arg1] 
	  else if 1 = idx then 
	    let thm =  INST [(arg0,bool_a);(arg1,bool_b)] cnfthm_iff_1 in
	    let ret = PURE_REWRITE_RULE [dneg] thm  in
	      rec_trans_cnf ret [before;(hol_negate arg0); (hol_negate arg1)] 
	  else if 2 = idx then 
	    let thm =  INST [(arg0,bool_a);(arg1,bool_b)] cnfthm_iff_2 in
	    let ret = PURE_REWRITE_RULE [dneg] thm  in
	      rec_trans_cnf ret [(hol_negate before);(hol_negate arg0); arg1] 
	  else if 3 = idx then 
	    let thm =  INST [(arg0,bool_a);(arg1,bool_b)] cnfthm_iff_3 in
	    let ret = PURE_REWRITE_RULE [dneg] thm  in
	      rec_trans_cnf ret [(hol_negate before);arg0; (hol_negate arg1)] 
		
 	  else failwith "unknown idx"
	)
	  
      else if (reason = string_and )
      then
	(	  
	  let conjs = before in
	  let term  = after in
	  let th1,succ = cnf_and_mid_thm conjs term  in
	    if (not succ) then failwith "cannot find the term "
	    else 
	      let th2 = INST [before,bool_a;after,bool_b] cnfthm_and_mid_or in
	      let res = rec_trans_cnf (EQ_MP th2 th1) [(hol_negate before);after]  in
		res
		  
	)
      else if (reason = string_or_mid )
      then
	(	  
	  let disjs = before in
	  let term  = after in
	  let th1,succ = cnf_or_mid_thm disjs term  in
	    if (not succ) then failwith "cannot find the term "
	    else 
	      let th2 = if (is_neg after) 
		then INST [before,bool_a; (hol_negate after),bool_b] cnfthm_or_mid_or_n 
		else INST [before,bool_a;after,bool_b] cnfthm_or_mid_or in
	      let th3 = EQ_MP th2 th1 in
		rec_trans_cnf th3 [before;(hol_negate after)] 
	)
      else if (reason = string_imp )
      then
	(	
	  let arg0 = translate_term vc (child (child expr 2) 0) in
	  let arg1 = translate_term vc (child (child expr 2) 1) in
	    if 0 = idx then 
	      let thm =  INST [(arg0,bool_a);(arg1,bool_b)] cnfthm_imp_0 in
		rec_trans_cnf thm [before; arg0] 
	    else if 1 = idx then 
	      let thm = if (is_neg arg1) 
		then INST [(arg0,bool_a);(hol_negate arg1),bool_b] cnfthm_imp_1_n 
		else INST [(arg0,bool_a);(arg1,bool_b)] cnfthm_imp_1 in
		rec_trans_cnf thm [before;(hol_negate arg1)] 
	    else if 2 = idx then 
	      let thm =  if (is_neg arg0) 
		then INST [((hol_negate arg0),bool_a);(arg1,bool_b)] cnfthm_imp_2_n
		else INST [(arg0,bool_a);(arg1,bool_b)] cnfthm_imp_2 in
		rec_trans_cnf thm [(hol_negate before);(hol_negate arg0); arg1] 
 	    else failwith "unknown idx"
	)
      else 
	(print_string reason;
	 failwith "unknown CNF" )
  and
      
      handle_learned_clause vc expr = 
    let res_expr = translate_term vc (child expr 1) in
    let assups = if (is_disj res_expr) 
      then get_child_terms vc (child expr 1)
      else [res_expr] in
    let th =  translate_proof vc (child expr 2) in
    let ors = 
      if ((concl th) <> ty_false) then 
	((* assert ((last assups) = (concl th));*)
	  butlast assups
	)
      else assups in
    let res = itlist (fun x y -> IMP_OR (hol_negate x) y ) ors th in
      if ( [] <> (hyp res)) 
      then (failwith "error in learned clause"  )
      else ();
      (*	assert (res_expr = (concl res)); *)
      res
  and
      handle_CNFITE vc expr = 
    let before = translate_term vc (child expr 1) in
    let newarg0 = translate_term vc (child expr 2) in
    let newarg1 = translate_term vc (child expr 3) in
    let newarg2 = translate_term vc (child expr 4) in
    let idx = getInt (child expr 5) in
    let e0thm = translate_proof vc (child expr 6) in
    let e1thm = translate_proof vc (child expr 7) in
    let e2thm = translate_proof vc (child expr 8) in
    let  oldarg0, (oldarg1, oldarg2) = dest_cond before in
      if 1 = idx then 
	let th1 = INST [(oldarg0,bool_a); (oldarg1,bool_b); (oldarg2,bool_c);
			(newarg0,bool_a1);(newarg1,bool_b1);(newarg2,bool_c1)] cnfthm_ite_1 in
	let th2 = MP (MP (MP th1 e0thm) e1thm) e2thm in
	let ret = PURE_REWRITE_RULE [dneg] th2  in
	  if ((concl th2) != (concl ret)) then 
	    ()
	  else ();
	  rec_trans_cnf ret [(hol_negate before); newarg0; newarg2] 
	    
      else if (2 = idx) then
	let th1 = INST [(oldarg0,bool_a); (oldarg1,bool_b); (oldarg2,bool_c);
			(newarg0,bool_a1);(newarg1,bool_b1);(newarg2,bool_c1)] cnfthm_ite_2 in
	let th2 = MP (MP (MP th1 e0thm) e1thm) e2thm in
	let ret = PURE_REWRITE_RULE [dneg] th2  in
	  if ((concl th2) != (concl ret)) then 
	    ( 	    )
	  else ();
	  rec_trans_cnf ret [before; newarg0; (hol_negate newarg2)] 
	    
      else if (3 = idx) then
	let th1 = INST [(oldarg0,bool_a); (oldarg1,bool_b); (oldarg2,bool_c);
			(newarg0,bool_a1);(newarg1,bool_b1);(newarg2,bool_c1)] cnfthm_ite_3 in
	let th2 = MP (MP (MP th1 e0thm) e1thm) e2thm in
	let ret = PURE_REWRITE_RULE [dneg] th2  in
	  if ((concl th2) != (concl ret)) then 
	    (  )
	  else ();
	  rec_trans_cnf ret [before; (hol_negate newarg0); (hol_negate newarg1)] 
	    
      else if (4 = idx) then
	let th1 = INST [(oldarg0,bool_a); (oldarg1,bool_b); (oldarg2,bool_c);
			(newarg0,bool_a1);(newarg1,bool_b1);(newarg2,bool_c1)] cnfthm_ite_4 in
	let th2 = MP (MP (MP th1 e0thm) e1thm) e2thm in
	let ret = PURE_REWRITE_RULE [dneg] th2  in
	  if ((concl th2) != (concl ret)) then 
	    (  )
	  else ();
	  rec_trans_cnf ret [(hol_negate before); (hol_negate newarg0); newarg1] 
	    
      else if (5 = idx) then
	let th1 = INST [(oldarg0,bool_a); (oldarg1,bool_b); (oldarg2,bool_c);
			(newarg0,bool_a1);(newarg1,bool_b1);(newarg2,bool_c1)] cnfthm_ite_5 in
	let th2 = MP (MP (MP th1 e0thm) e1thm) e2thm in
	let ret = PURE_REWRITE_RULE [dneg] th2  in
	  
	  if ((concl th2) != (concl ret)) then 
	    (  )
	  else ();
	  
	  rec_trans_cnf ret [before; (hol_negate newarg1); (hol_negate newarg2)] 
      else if (6 = idx) then
	let th1 = INST [(oldarg0,bool_a); (oldarg1,bool_b); (oldarg2,bool_c);
			(newarg0,bool_a1);(newarg1,bool_b1);(newarg2,bool_c1)] cnfthm_ite_6 in
	let th2 = MP (MP (MP th1 e0thm) e1thm) e2thm in
	let ret = PURE_REWRITE_RULE [dneg] th2  in
	  
	  if ((concl th2) != (concl ret)) then 
	    (  )
	  else ();
	  rec_trans_cnf ret [(hol_negate before); newarg1; newarg2] 
      else failwith "unknown idx"
	
  and
      
      handle_cnf_add_unit vc expr = 
    let unitExpr = translate_term vc (child expr 1) in
    let thm = translate_proof vc (child expr 2) in
      if ((concl thm) <> unitExpr) then (
	  failwith "unexpected in add unit" )
      else 
	let newThm = (
	    if (is_neg unitExpr)
	    then INST [((hol_negate unitExpr),bool_a)] disj_false_neg
	    else INST [(unitExpr,bool_a)] disj_false_pos
	) in
	  UNDISCH (EQ_MP newThm thm) 
	    
  and
      
      handle_cnf_convert vc expr = 
    let unitExpr = translate_term vc (child expr 1) in
    let ors = get_child_terms vc (child expr 2) in
    let thm = translate_proof vc (child expr 3) in
      if  (unitExpr = (concl thm)) then 
	let res = rec_trans_cnf thm ors in
	  (*	  assert (`F` = (concl res));*)
	  res
      else(
	  failwith "why convert fails")
  and
      
      handle_if_lift_rule vc expr = 
    let tp = `p:A->A->B` and tx = `x:A` and
	ta = bool_a and tb = `b:A` and  tc = `c:A` in
    let e = translate_term vc (child expr 1) in
    let newite = translate_term vc (child expr 2 ) in
    let idx = getInt (child expr 3) in
      if (is_eq e) then
	let eq_lhs, rhs = dest_comb e in
	let eq_op, lhs = dest_comb eq_lhs in
	let (cond,(brt, brf)),x,meta_thm = 
	  if ( 0 = idx ) 
	  then (dest_cond lhs), rhs, lemma_if_lift_2_0 
	  else if (1 = idx) 
	  then (dest_cond rhs), lhs, lemma_if_lift_2_1
	  else failwith "error in if_lift" in
	let real_a_type = type_of x in
	let real_b_type = type_of e in	  
	let res =   
	  PINST
	    [(real_a_type,ty_a);(real_b_type,ty_b)] 
	    [(eq_op,tp);(cond,ta);(brt,tb);(brf,tc);(x,tx)] 
	    meta_thm in
	  res
      else failwith "not handled case in if_lift\n"
	
  and
      
      handle_universal_elimination1 vc expr = 
    let univ_pf =  translate_proof vc (child expr 5) in
    let univ = translate_term vc (child expr 1) in
    let res_expr = translate_term vc (child expr 3) in
    let vars = get_child_terms vc (child expr 2) in
    let res =  rev_itlist (fun x y -> ISPEC x y) vars univ_pf  in
      res
  and
      
      handle_rewrite_distinct vc expr = 
    let dist_term  =  translate_term  vc (child expr 1) in
    let dist_thm = Hashtbl.find distinct_table dist_term in
    let res_term = translate_term vc (child expr 2) in
    let th1 = REWRITE_CONV [dist_thm] dist_term in
    let th2 = CONJ_ACI_RULE (mk_eqc (rhs (concl th1)) res_term) in
    let res_thm = TRANS th1 th2 in
      res_thm
	
  and
      handle_eq_elim_int vc expr = 
    let before = translate_term vc (child expr 1) in
    let after = translate_term vc (child expr 2) in
    let before_pf = translate_proof vc (child expr 3) in
      (*	assert ((concl before_pf) = before);*)
    let iff = mk_iff (before,after) in
      EQ_MP (ASSUME iff)  before_pf 
  and
      
      handle_expand_gray_shadowconst0 vc expr = 
    let shadow = translate_term vc (child expr 1) in
    let v,e,c1,c2  = dest_gray_shadow shadow in
    let res = mk_eqc v (mk_plus_int e c2)   in
    let shadow_pf = translate_proof vc (child expr 2) in 
    let shadow_def = SPECL [v;e;c1;c2] gray_shadow_thm in
    let shadow_def_pf = EQ_MP shadow_def shadow_pf in
      (* here I use a lazy method, will improve this if necessary *)
    let mpthm = INT_ARITH (mk_eqc (concl shadow_def_pf) res) in
      EQ_MP mpthm shadow_def_pf 
  and
      
      handle_expand_gray_shadow vc expr = 
    let shadow = translate_term vc (child expr 1) in
    let v,e,c1,c2 = dest_gray_shadow shadow in
    let shadow_pf = translate_proof vc (child expr 2) in
    let shadow_def = SPECL [v;e;c1;c2] gray_shadow_thm in
    let shadow_def_pf = EQ_MP shadow_def shadow_pf in
    let shadow_concl = concl shadow_def_pf in
    let expr1 = mk_le_int (mk_add_int e c1) v in
    let expr2 = mk_le_int v (mk_add_int e c2) in
    let pre_th1 = MP (INT_ARITH (mk_imp (shadow_concl,expr1))) shadow_def_pf in
    let pre_th2 = MP (INT_ARITH (mk_imp (shadow_concl,expr2))) shadow_def_pf in
    let res = CONJ pre_th1 pre_th2 in
      res
  and
      
      handle_expand_dark_shadow vc expr = 
    let shadow = translate_term vc (child expr 1) in
    let shadow_pf = translate_proof vc (child expr 2) in
    let lhs,rhs = dest_dark_shadow shadow in
      if(is_int lhs) then 
	let shadow_def = SPECL [lhs;rhs] dark_shadow_int_thm in
	  EQ_MP shadow_def shadow_pf
      else 
	let shadow_def = SPECL [lhs;rhs] dark_shadow_real_thm in
	  EQ_MP shadow_def shadow_pf
  and

      handle_gray_shadow_const   vc expr = 
    let oldg = translate_term vc (child expr 1) in
    let newg = translate_term vc (child expr 2) in
    let v,e,c1,c2 = dest_gray_shadow oldg in
    let oldg_pf = translate_proof vc (child expr 3) in
    let nv,ne,nc1,nc2 = dest_gray_shadow newg in
    let old_shadow_def = SPECL [v;e;c1;c2] gray_shadow_thm in
    let new_shadow_def = SPECL [nv;ne;nc1;nc2] gray_shadow_thm in
    let actual_old_shadow = rand (concl old_shadow_def) in
    let actual_new_shadow = rand (concl new_shadow_def) in
    let th1 = INT_ARITH (mk_eqc actual_old_shadow actual_new_shadow) in
    let th2 = TRANS old_shadow_def th1 in
    let th3 = TRANS th2 (SYM  new_shadow_def) in
    let th4 = EQ_MP th3 oldg_pf in
      th4
	
  and

      handle_split_gray_shadow   vc expr = 
    let oldg = translate_term vc (child expr 1) in
    let oldg_pf = translate_proof vc (child expr 4) in
    let newg1 = translate_term vc (child expr 2) in
    let newg2 = translate_term vc (child expr 3) in
    let face_g1_or_g2 = mk_disj (newg1, newg2) in
    let face_ng1_or_ng2 = mk_disj ((mk_neg newg1), (mk_neg newg2)) in
    let face_finalg  = mk_conj (face_g1_or_g2,face_ng1_or_ng2) in
      
    let n1v,n1e,n1c1,n1c2 = dest_gray_shadow newg1 in
    let n2v,n2e,n2c1,n2c2 = dest_gray_shadow newg2 in
    let new1_shadow_def = SPECL [n1v;n1e;n1c1;n1c2] gray_shadow_thm in
    let new2_shadow_def = SPECL [n2v;n2e;n2c1;n2c2] gray_shadow_thm in
      
    let actual_finalg_thm = SUBS_CONV [new1_shadow_def;new2_shadow_def] face_finalg in
      
    let v,e,c1,c2 = dest_gray_shadow oldg in
    let old_shadow_def = SPECL [v;e;c1;c2] gray_shadow_thm in
    let actual_old_shadow = rand (concl old_shadow_def) in
      
    let actual_new1_shadow = rand (concl new1_shadow_def) in
    let actual_new2_shadow = rand (concl new2_shadow_def) in
    let actual_g1_or_g2 = mk_disj (actual_new1_shadow,actual_new2_shadow) in
    let actual_ng1_or_ng2 = mk_disj ((mk_neg actual_new1_shadow), (mk_neg actual_new2_shadow)) in  
    let actual_finalg_2 = mk_conj (actual_g1_or_g2,  actual_ng1_or_ng2) in
      
    (*	assert (actual_finalg_2 = rand (concl actual_finalg_thm));*)
    let rev_actual_finalg_thm = SYM actual_finalg_thm in
      
    let eq = mk_imp (actual_old_shadow, actual_finalg_2)  in
    let eq_thm = INT_ARITH eq in
    let th0 = EQ_MP old_shadow_def oldg_pf in
    let th1 = MP eq_thm th0 in
    let th2 = EQ_MP rev_actual_finalg_thm th1 in
      th2
	
  and

      handle_dark_grayshadow_2ab   vc expr = 
    let expr1 = translate_term vc (child expr 1) in
    let expr2 = translate_term vc (child expr 2) in
    let newd =  translate_term vc (child expr 3) in
    let newg =  translate_term vc (child expr 4) in
    let ex1_pf = translate_proof vc (child expr 5) in
    let ex2_pf = translate_proof vc (child expr 6) in
      
    let face_d_or_g = mk_disj (newd, newg) in
    let face_final  = face_d_or_g in 
      
    let lhs,rhs = dest_dark_shadow newd in
    let v,e,c1,c2 = dest_gray_shadow newg in
      
    let d_shadow_def = 
      if (is_int lhs) then SPECL [lhs;rhs] dark_shadow_int_thm 
      else failwith "error in dard_grayshadow_2ab"  in
      
    let g_shadow_def = SPECL [v;e;c1;c2] gray_shadow_thm in
    let actual_finalg_thm = SUBS_CONV [d_shadow_def;g_shadow_def] face_final in
    let input_thm = CONJ ex1_pf ex2_pf in
    let actual_input_term = concl input_thm in
      
    (*	assert (actual_input_term = mk_conj (expr1,expr2));*)
      
    let actual_d_shadow = rand (concl d_shadow_def) in
    let actual_g_shadow = rand (concl g_shadow_def) in
    let actual_d_or_g = mk_disj (actual_d_shadow,actual_g_shadow) in
    let actual_finalg_2 = actual_d_or_g in 
      
    (*	assert (actual_finalg_2 = rand (concl actual_finalg_thm));*)
      
    let rev_actual_finalg_thm = SYM actual_finalg_thm in
    let eq = mk_imp  (actual_input_term, actual_finalg_2)  in
    let eq_thm = INT_ARITH eq in 
    let th1 = MP eq_thm input_thm in
    let th2 = EQ_MP rev_actual_finalg_thm th1 in
      th2
	
  and

      handle_finite_interval   vc expr = 
    let e1 = translate_term vc (child expr 1) in
    let e2 = translate_term vc (child expr 2) in
    let e1_pf = translate_proof vc (child expr 5) in
    let e2_pf = translate_proof vc (child expr 6) in
    let a,t = dest_le_int  e1 in
    let _,ac = dest_le_int e2 in
    let _,c = dest_add_int  ac in
    let shadow_def = SPECL [t; a; int_0; c] gray_shadow_thm in
    let actual_shadow_expr = rand (concl shadow_def) in
    let th1 = INT_ARITH (mk_imp (mk_conj (e1,e2), actual_shadow_expr)) in
    let th2 = CONJ e1_pf e2_pf in
    let th3 = MP th1 th2 in
    let rev_shadow_def = SYM shadow_def in
    let th4 = EQ_MP rev_shadow_def th3 in
      th4
  and

      handle_rewrite_not_forall vc expr = 
    let e =  translate_term vc (child expr 1) in
      ONCE_REWRITE_CONV [NOT_FORALL_THM] e
  and

      handle_rewrite_not_exists vc expr = 
    let e =  translate_term vc (child expr 1) in
      ONCE_REWRITE_CONV [NOT_EXISTS_THM] e
	
  and 
      
      (* =========================  translate_proof  ========================== *)

    basic_translate_proof vc expr = 
(* 
   (* count *) print_flush();
    proof_count := !proof_count + 1;
    if (!proof_count) mod 5000 = 0 then 
(*      (print_string ("translations attempted: " ^ string_of_int !proof_count ^ "\n");
      print_flush();)
*)
      (  print_int !proof_count;
	 print_string " times\n";
	 print_string "Depth ";
	 print_int !depth_level;
	 print_newline();)
     
    else ();
*)  
    (* if expr is a bound variable, it is an assumption *)
    if (isVar expr) or (kind_bound_var = (kind expr)) then 
      (
	let term = toExpr(getType vc expr) in
	let hol_expr = translate_term vc term in
	  if (is_iff hol_expr) then
	    let l,r = dest_iff hol_expr in
	      if (is_exists l) then
		(	
		let assup_eq_assup_easy = (PURE_REWRITE_CONV [iff_th] hol_expr) in
		let assup_easy = rand (concl assup_eq_assup_easy ) in
		let l2r,r2l = dest_conj assup_easy in
		let l2r_th = MESON [] l2r in
		let r2l_th = MESON [] r2l in
		let assup_easy_th = CONJ l2r_th r2l_th in
		let th2 = EQ_MP (SYM assup_eq_assup_easy) assup_easy_th in
		  th2 
		)
	      else ASSUME hol_expr
	  else ASSUME hol_expr
      )
      else if isLambda vc expr then handle_lambda vc expr      
      else if isPair vc expr then translate_proof vc (child expr 1)
      else 

try
      (* otherwise, go to work *)
      let rule = get_rule_string expr in
(*
	print_flush();
  print_string "now doing: ";
  print_string rule;
  print_string "  ";
  print_int !depth_level;
  print_string "\n";
  print_flush();
*)
        match rule with
            "rewrite_eq_refl" -> handle_rewrite_eq_refl vc expr
	  | "bool_resolution" -> handle_bool_resolution  vc expr 
	  | "assumptions" -> handle_assumptions  vc expr 
	  | "refl" -> handle_refl  vc expr 
	  | "CNF" -> handle_CNF  vc expr 
	  | "CNFITE" -> handle_CNFITE  vc expr 
	  | "learned_clause" -> handle_learned_clause  vc expr 
	  | "cnf_add_unit" -> handle_cnf_add_unit vc expr 
	  | "cnf_convert" -> handle_cnf_convert vc expr (*please debug this *) 
	  | "if_lift_rule" -> handle_if_lift_rule vc expr 
          | "pf_by_contradiction" -> handle_pf_by_contradiction vc expr
          | "iff_contrapositive" -> handle_iff_contrapositive vc expr
          | "neg_intro" -> handle_neg_intro vc expr
          | "unit_prop" -> handle_unit_prop vc expr
          | "conflict" -> handle_conflict vc expr
          | "conflict_clause" -> handle_conflict_clause vc expr
          | "iff_symm" -> handle_iff_symm vc expr
          | "eq_symm" -> handle_eq_symm vc expr
          | "not_to_iff" -> handle_not_to_iff vc expr
          | "iff_mp" -> handle_iff_mp vc expr
          | "impl_mp" -> handle_impl_mp vc expr
          | "rewrite_not_true" -> handle_rewrite_not_true vc expr
          | "rewrite_not_not" -> handle_rewrite_not_not vc expr
          | "rewrite_not_false" -> handle_rewrite_not_false vc expr
          | "rewrite_implies" -> handle_rewrite_implies vc expr
          | "rewrite_not_and" -> handle_rewrite_not_and vc expr
          | "rewrite_not_or" -> handle_rewrite_not_or vc expr
          | "andE" -> handle_andE vc expr
          | "andI" -> handle_andI vc expr
          | "int_const_eq" -> handle_int_const_eq vc expr
          | "basic_subst_op" -> handle_subst_op vc expr
          | "basic_subst_op0" -> handle_subst_op vc expr
          | "basic_subst_op1" -> handle_subst_op vc expr
          | "optimized_subst_op" -> handle_subst_op vc expr
          | "rewrite_eq_symm" -> handle_rewrite_eq_symm vc expr
          | "iff_trans" -> handle_iff_trans vc expr
          | "eq_trans" -> handle_eq_trans vc expr
          | "iff_true" -> handle_iff_true vc expr
          | "iff_true_elim" -> handle_iff_true_elim vc expr
          | "iff_false_elim" -> handle_iff_false_elim vc expr
          | "rewrite_iff_refl" -> handle_rewrite_iff_refl vc expr
          | "rewrite_iff" -> handle_rewrite_iff vc expr
	  | "rewrite_iff_symm" -> handle_rewrite_iff_symm vc expr
          | "rewrite_and" -> handle_rewrite_and vc expr 
          | "rewrite_or" -> handle_rewrite_or vc expr 
          | "real_shadow" -> handle_real_shadow vc expr
	  | "cycleConflict" -> handle_cycleConflict vc expr
	  | "implyEqualities" -> handle_implyEqualities vc expr
          | "negated_inequality" -> handle_negated_inequality vc expr
	  | "implyWeakerInequality" -> handle_implyWeakerInequality vc expr
	  | "implyNegatedInequality" -> handle_implyNegatedInequality vc expr
	  | "implyNegatedInequalityDiffLogic" -> handle_implyNegatedInequalityDiffLogic vc expr
	  | "implyWeakerInequalityDiffLogic" -> handle_implyWeakerInequalityDiffLogic vc  expr
          | "flip_inequality" -> handle_flip_inequality vc expr
          | "right_minus_left" -> handle_right_minus_left vc expr
          | "minus_to_plus" -> handle_minus_to_plus vc expr
          | "canon_combo_like_terms" -> handle_canon_combo_like_terms vc expr
          | "iff_refl" -> handle_iff_refl vc expr
          | "eq_refl" -> handle_eq_refl vc expr
          | "mult_ineqn" -> handle_mult_ineqn vc expr
          | "mult_eqn" -> handle_mult_eqn vc expr
          | "plus_predicate" -> handle_plus_predicate vc expr
          | "const_predicate" -> handle_const_predicate vc expr
          | "canon_flatten_sum" -> handle_canon_flatten_sum vc expr
          | "canon_mult_const_sum" -> handle_canon_mult_const_sum vc expr
          | "canon_mult_const_term" -> handle_canon_mult_const_term vc expr
          | "canon_mult_const_const" -> handle_canon_mult_const_const vc expr
          | "canon_mult_mterm_mterm" -> handle_canon_mult_mterm_mterm vc expr
          | "canon_mult_one" -> handle_canon_mult_one vc expr
          | "canon_mult" -> handle_canon_mult vc expr
          | "canon_plus" -> handle_canon_plus vc expr
          | "canon_mult_term_const" -> handle_canon_mult_term_const vc expr
          | "rewriteSameStore" -> handle_rewrite_same_store vc expr
          | "rewriteRedundantWrite1" -> handle_rewrite_redundant_write_1 vc expr
          | "rewriteRedundantWrite2" -> handle_rewrite_redundant_write_2 vc expr
          | "rewriteReadWrite" -> handle_rewrite_read_write vc expr
          | "rewriteWriteWrite" -> handle_rewrite_write_write vc expr
          | "rewrite_ite_true" -> handle_rewrite_ite_true vc expr
          | "rewrite_ite_false" -> handle_rewrite_ite_false vc expr
          | "renameRead" -> handle_rename_read vc expr 
          | "false_implies_anything" -> handle_false_implies_anything vc expr 
          | "rewrite_ite_true_iff"  -> handle_rewrite_ite_true_iff vc expr
          | "rewrite_ite_false_iff"  -> handle_rewrite_ite_false_iff vc expr
          | "rewrite_constdef"  -> handle_rewrite_constdef vc expr
          | "rewrite_not_ite"  -> handle_rewrite_not_ite vc expr
          | "uminus_to_mult"  -> handle_uminus_to_mult vc expr
          | "rewrite_ite_same_iff" -> handle_rewrite_ite_same_iff vc expr
          | "rewrite_ite_same" -> handle_rewrite_ite_same vc expr
          | "canon_invert_divide" -> handle_canon_invert_divide vc expr
          | "real_shadow_eq" -> handle_real_shadow_eq vc expr
	  | "lessThan_To_LE_rhs" -> handle_lessThan_To_LE_rhs vc expr 
	  | "lessThan_To_LE_lhs" -> handle_lessThan_To_LE_lhs vc expr 
	  | "lessThan_To_LE_rhs_rwr" -> handle_lessThan_To_LE_rhs_rwr vc expr 
	  | "lessThan_To_LE_lhs_rwr" -> handle_lessThan_To_LE_lhs_rwr vc expr 
          | "var_intro" -> handle_var_intro vc expr
	  | "interchangeIndices" -> handle_interchangeindices vc expr
	  | "or_distribuitivity_rule" -> handle_or_distribuitivity_rule  vc expr 
	  | "universal_elimination1" -> handle_universal_elimination1 vc expr
	  | "rewrite_not_forall" -> handle_rewrite_not_forall vc expr
	  | "rewrite_not_exists" -> handle_rewrite_not_exists vc expr
	  | "rewrite_distinct" -> handle_rewrite_distinct vc expr
	  | "eq_elim_int" -> handle_eq_elim_int vc expr 
	  | "expand_gray_shadowconst0" -> handle_expand_gray_shadowconst0 vc expr 
	  | "expand_gray_shadow" -> handle_expand_gray_shadow vc expr 
	  | "expand_dark_shadow" -> handle_expand_dark_shadow vc expr 
	  | "gray_shadow_const" -> handle_gray_shadow_const vc expr 

	  | "dark_grayshadow_2ab" -> handle_dark_grayshadow_2ab  vc expr 
	  | "split_gray_shadow" -> handle_split_gray_shadow vc expr 
	  | "finite_interval" -> handle_finite_interval vc expr 
	  | "normalizeQuant" ->  handle_normalizeQuant vc expr  
	  | "minisat_proof" -> handle_minisat_proof  vc expr 
          | _ ->  (* print_expr vc expr ;*) failwith ("Error: " ^ rule ^ " not supported.") 
with Failure x ->
  reset();
  if (! back_level >=1 ) then
    (print_flush();
     print_string "back level ";
     print_int !back_level;
     print_string "\n";
     print_expr vc expr; 
     back_level := (!back_level - 1);

     print_string ("Fail " ^ x ^ ":current expr is ");
     print_string ((get_rule_string expr) ^":");
     print_string "\n"; 
    ); 
  failwith "translate error"
    
  and  

  translate_proof vc expr =
      print_flush();   (*flush stdout;*)
    try Hashtbl.find proof_table expr
    with Not_found ->   
      let time_fun expr = 
        let rule = 
          if not (isLambda vc expr) then
            child expr 0 
          else
            trueExpr vc in
          print_expr vc rule in 
      if !time_proof then 
        (let thm,time = time_save (basic_translate_proof vc) expr in
           Hashtbl.add proof_table expr thm;  
           if time > !max_time then
             (report("Took too long: ");
              time_fun expr;
              print_flush();)
           else 
             ();
           thm)
      else
	(depth_level := !depth_level+1;
        let thm = basic_translate_proof vc expr in  
          Hashtbl.add proof_table expr thm;
	  depth_level := !depth_level - 1;
	  let rule_string = (if (isVar expr) then "ASSUM" else (get_rule_string expr)) in
	    if (!print_name or (true or  rule_string = "bool_resolution" )) 
	    then (* 
		print_string ">> found proof \n" ;
		print_string ("when doing " ^ rule_string ^ "\n");

		print_flush();
		print_string ("when doing " ^ rule_string ^ "   "); print_int (!depth_level + 1); nl(); 
		print_flush();

		print_thm thm ;
		print_newline(); 
		print_flush();
		 *)
		thm
	    else thm
	)
  in
    reset_count,translate_proof in


let  translate_pair vc exp = 
(*  assert(kind exp = getKindInt vc "RAW_LIST");*)
  print_string "tranalate_pair";
  print_expr vc exp;
  let n = getInt (child exp 0) in
  let p = translate_proof vc (child exp 1) in
    (n,p) in
  

object 
  method translate_proof vc p = 
    Hashtbl.clear proof_table;
    Hashtbl.clear term_table;
    Hashtbl.clear const_def_tbl;
    exists_term_table := [];
    assumptions_table := [];
    debug_table :=[];
    back_level := 1;  
    translate_proof vc p
end;;

(* ---------------------------------------------------------------------- *)
(*  Debug                                                                 *)
(* ---------------------------------------------------------------------- *)


(* ====================================================================== *)
(*  CVC_Prove                                                             *)
(* ====================================================================== *)

class has_int = fun cvc ->
  let int_kind = getKindInt cvc "_INT"  in
  let real_kind = getKindInt cvc "_REAL" in
  let list_kind = getKindInt cvc "_RAW_LIST" in
  let rec has_int_type expr  =
    if (kind expr) = list_kind
    then 	  
      let chs = child_expr_list cvc expr in
	itlist (fun x y ->  y && (has_int_type x) ) chs true 
    else 
      let type_kind = kind (typeExpr cvc expr) in 
	if type_kind = int_kind 
	then true   
	else if  type_kind = real_kind 
	then false
	else 
	  let chs = child_expr_list cvc expr in
	    itlist (fun x y ->  y && (has_int_type x) ) chs true  in
object 
  method check = has_int_type
end ;;


let check_ints mc = 
  let cvc_assup = getAssumptions mc in
  let cvc_query = getQuery mc in
  let has_int_funcs = new has_int mc in 
    if (has_int_funcs#check cvc_assup) 
      && (has_int_funcs#check cvc_query) 
    then (pr "all int\n"; true) 
    else (pr "no int\n" ;false) ;;


let cvc_prove tm =
  let vc = getVC_nodagCVC() in
  let encoder = new encode_cvc in
  let expr = encoder#encode_term vc tm in
    ignore(query vc expr);
    let p = getProof vc in
(*      print_expr vc p;*)
    let has_ints = check_ints vc in
    let translator = (new translate_cvc vc has_ints ) in
    let pf =  translator#translate_proof vc p in
      destroy_cvc vc;
      pf;;

let cvc_prove_file fname =
  let vc = 
    if is_smt fname 
    then getVC_nodagSMT()
    else getVC_nodagCVC() in
  let p = ml_vc_getProofOfFile vc fname in
  let has_ints = check_ints vc in
  let translator = (new translate_cvc vc has_ints ) in
  let pf =  translator#translate_proof vc p in
    destroy_cvc vc;
    pf;;


let batch_prove s =
  let file_list = (all_smt_files s)@(all_cvc_files s) in
  let it_fun x =
    pr "====================================\n";
    pr "file: "; pr x; nl(); print_flush();
    try 
      let pf =  time cvc_prove_file x in
	pr "\nproof translation success\n"
    with  x -> pr "\nproof translation failed\n" in
    do_list it_fun file_list;;

(*
let get_handle_file fname =
  let vc = 
    if is_smt fname 
    then getVC_nodagSMT()
    else getVC_nodagCVC() in
    try 
      vc, ml_vc_getProofOfFile vc fname 
    with Failure x -> (destroy_cvc vc; failwith "not a valid theorem");;

let get_cvc_assups mc = 
  let org_assumptions = getAssumptions mc in
  let org_list = (child_expr_list mc org_assumptions) in
  let org_assups =  map (fun x -> translate_term mc x) org_list in
    org_assups;;

let get_cvc_query mc = 
  let org_query = translate_term mc (getQuery mc) in
    org_query;;
    
let compare_assup_list biglist smalist =
  let count = ref 0 in
  let rec compare_list l1 l2 =
    if (length l1) > 0 && (length l2) > 0 then
      (let hd1 = hd l1 and hd2 = hd l2 in
	 if hd1 = hd2 then (
	     compare_list (tl l1) (tl l2)
	 )
	 else if hd1 < hd2 then compare_list (tl l1) l2
	 else (
	     if hd2 = `T` then compare_list  l1 (tl l2)
	     else ( 	 failwith "fail compare"
	     )
	 )
      )
    else if (length l1) > 0 then true
    else if (length l2) > 0 then false
    else true in
  let res = compare_list (sort (<) biglist) (sort (<) smalist) in
    res;;


let cvc_check_handle mc me pf  = 
  let org_query = get_cvc_query mc in
  let org_assups = get_cvc_assups mc in
  if (org_query = (concl pf)) then
    let res= 
      try (compare_assup_list org_assups (hyp pf))
      with Failure _ -> false in 
      res
  else false;;

let cvc_prove_handle vc p = 
  Hashtbl.clear proof_table;
  Hashtbl.clear term_table;
  Hashtbl.clear en_tm_tbl;
  Hashtbl.clear const_def_tbl;
  let res =  translate_proof vc p in
    res;; 

let batch_prove s =
  let file_list = all_smt_files s in
  let it_fun x =
    print_string x; print_newline(); print_flush();
    let pf =
      try cvc_prove_file x
      with x -> failwith "failed"
    in
      print_thm pf;print_newline() in
    do_list it_fun file_list;;

let rec list_at n list =
  if n = 0 then list else list_at (n - 1) (tl list);;


let list_prove_fun  x =
  Hashtbl.clear proof_table;
  Hashtbl.clear term_table;
  Hashtbl.clear en_tm_tbl;
  Hashtbl.clear const_def_tbl;
  pr "===================================\n"; 
  print_string "translating "; pr x; nl(); 
  print_flush();
  Gc.compact();
  try 
    let mc,me = time get_handle_file x in
    let nowhere = try
	let pf =  try_time_out2 translate_proof mc me 600.0 in
	  pr "good theorem\n";
(*	  let res = time3 cvc_check_handle mc me pf  in
	    if res then pr "checked\n"
	    else pr "failed check\n" 
*)
      with Failure x -> pr x; pr " failed to prove\n" in
      destroy_cvc mc 
  with Failure x -> pr" not a valid theorem\n";; 
  
let list_prove file_list  = 
  generate_distinct_def 2;
  generate_distinct_def 3;
  generate_distinct_def 4;
  generate_distinct_def 5;
  generate_distinct_def 6;
  generate_distinct_def 7;
  generate_distinct_def 8;
  generate_distinct_def 9;
  do_list list_prove_fun  file_list;;

let list_prove_from list index = 
  let flist = list_at index list in
    do_list list_prove_fun flist;;

let list_pos list term =
  let rec position list term index = 
    if list = [] then failwith "not found"
    else if (hd list) = term then index
    else position (tl list) term index+1 in
    position list term 0;;


let list_prove_from_file list file  =
  let pos = list_pos list file in
    list_prove_from list pos;;

*)
