
(*****************************************************************************
/*!
 * \file util.ml
 *
 * Print functions, time functions, etc.
 * 
 * <hr>
 *
 * License to use, copy, modify, sell and/or distribute this software
 * and its documentation for any purpose is hereby granted without
 * royalty, subject to the terms and conditions defined in the \ref
 * README file provided with this distribution.
 * 
 * <hr>
 * 
 */
/*****************************************************************************)

(* ---------------------------------------------------------------------- *)
(*  File and time functions                                               *)
(* ---------------------------------------------------------------------- *)

let dir_files dir_name =
  try
    let files = Sys.readdir dir_name in
      map (fun x-> dir_name ^ "/" ^ x) (Array.to_list files) 
  with x-> raise x;; 

unset_jrh_lexer;; (* hol light does not like id in upper case *)	

open Unix;;

let is_dir fname =
  try
    let fstat = Unix.stat fname in
      if fstat.st_kind = S_DIR
      then true
      else false
  with x -> false ;;

set_jrh_lexer;;

let is_ext fname ext =
  Filename.check_suffix fname ext ;;

let rec all_smt_files dir_name =
  let cur_dir_files = dir_files dir_name in
  let deal_one fname rest =
    if is_dir fname then (all_smt_files fname) @ rest
    else if is_ext fname "smt" then [fname]@rest
    else rest in
    itlist deal_one cur_dir_files [] ;;


let rec all_cvc_files dir_name =
  let cur_dir_files = dir_files dir_name in
  let deal_one fname rest =
    if is_dir fname then (all_cvc_files fname) @ rest
    else if is_ext fname "cvc" then [fname]@rest
    else rest in
    itlist deal_one cur_dir_files [] ;;


let is_smt fname = is_ext fname "smt" ;;

unset_jrh_lexer;;	

let unixiterm = Unix.ITIMER_REAL;;

set_jrh_lexer;;

let set_timer tsecs =
  ignore (Unix.setitimer  unixiterm { Unix.it_interval = 0.0; Unix.it_value = tsecs });;

exception Timeout;;

let handle_sigalrm signo = raise Timeout;;

let try_time_out1 f x tsecs =
  let oldsig = Sys.signal Sys.sigalrm  (Sys.Signal_handle handle_sigalrm) in
    try 
      Sys.set_signal Sys.sigalrm oldsig;
      set_timer tsecs ;
      Sys.set_signal Sys.sigalrm oldsig;
      let res = f x in
	set_timer 0.0;
	Sys.set_signal Sys.sigalrm oldsig;
	res
    with Failure _ -> 	Sys.set_signal Sys.sigalrm oldsig; failwith "timeout";;

let try_time_out2 f x y tsecs =

  print_string "-n time out \n";
  print_flush();
  let oldsig = Sys.signal Sys.sigalrm  (Sys.Signal_handle handle_sigalrm) in
  let start_time = Sys.time() in
    try 
      set_timer tsecs ;
      let res = f x y in
	set_timer 0.0;
	Sys.set_signal Sys.sigalrm oldsig;
	let finish_time = Sys.time() in
	  report("CPU time (user): "^(string_of_float(finish_time -. start_time)));
	  res
    with Timeout -> 
      let finish_time = Sys.time() in
	Sys.set_signal Sys.sigalrm oldsig; 
	print_string("Failed after (user) CPU time of "^(string_of_float(finish_time -. start_time))^": ");
	failwith "timeout"
      | Failure _ -> set_timer 0.0; Sys.set_signal Sys.sigalrm oldsig;
	  let finish_time = Sys.time() in
	    Sys.set_signal Sys.sigalrm oldsig; 
	    print_string("Failed after (user) CPU time of "^(string_of_float(finish_time -. start_time))^": ");
	    failwith "translation error";;


let user_time f x =
   let t1=Unix.times () in
     for i = 10 to 30000000 do ignore(f x) done;
     let t2=Unix.times () in
       eprintf "Function %s: user time: %f; system time: %f\n%!" "data" (t2.tms_utime -. t1.tms_utime) (t2.tms_stime -. t1.tms_stime) ;;


let time_save_p = ref true;;
let time_save f x =
  if not (!time_save_p) then (f x,0.) else
  let start_time = Sys.time() in
  try let result = f x in
      let finish_time = Sys.time() in
      let time = finish_time -. start_time in
      result,time
  with e ->
      let finish_time = Sys.time() in
      print_string("Failed after (user) CPU time of "^
                   (string_of_float(finish_time -. start_time))^": ");
      raise e;;

let time f x =
  if not (!report_timing) then f x else
 let start_time = Sys.time() in
  try let result = f x  in
      let finish_time = Sys.time() in
      report("CPU time (user): "^(string_of_float(finish_time -. start_time)));
      result
  with e ->
      let finish_time = Sys.time() in
      print_string("Failed after (user) CPU time of "^
                   (string_of_float(finish_time -. start_time))^": ");
      raise e;;

let time2 f x y =
  if not (!report_timing) then f x  y else
 let start_time = Sys.time() in
  try let result = f x y in
      let finish_time = Sys.time() in
      report("CPU time (user): "^(string_of_float(finish_time -. start_time)));
      result
  with e ->
      let finish_time = Sys.time() in
      print_string("Failed after (user) CPU time of "^
                   (string_of_float(finish_time -. start_time))^": ");
      raise e;;



let time3 f x y z=
  if not (!report_timing) then f x  y z else
 let start_time = Sys.time() in
  try let result = f x y z in
      let finish_time = Sys.time() in
      report("CPU time (user): "^(string_of_float(finish_time -. start_time)));
      result
  with e ->
      let finish_time = Sys.time() in
      print_string("Failed after (user) CPU time of "^
                   (string_of_float(finish_time -. start_time))^": ");
      raise e;;

	 
(* ---------------------------------------------------------------------- *)
(*  Printing                                                              *)
(* ---------------------------------------------------------------------- *)

let rec mk_string s n = 
  match n with
      0 -> "" 
    | n -> s ^ mk_string s (n-1);;

let space n = mk_string " " n;;

let print_expr vc e = 
  let step = 2 in
  let rec print_expr vc e indent = 
    let sp = space (step * indent) in
    if isVar e then 
      let name = exprString e in
        if String.length name > 5 & String.sub name 0 5 = "assum" then
          print_expr vc (toExpr(getType vc e)) indent 
        else
          print_string (sp ^ (exprString e) ^ "\n") 
    else if isClosure e then print_expr vc (getBody e) indent 
    else if kind e = getKindInt vc "PF_APPLY" then 
      let name = exprString(child e 0) in
        print_string (sp ^ name ^ "\n");
        if arity e > 0 then 
          for i = 1 to arity e - 1 do 
            print_expr vc (child e i) (indent + 1)
          done 
        else () 
    else print_string (sp ^ (exprString e) ^ "\n") in
    print_expr vc e 0;;

let print_detailed vc exp = 
  print_string ("value: " ^ (exprString exp) ^ "\n");
  let untyped = map (getKindInt vc) ["PF_APPLY";"UCONST";"RAW_LIST"] in
  (if not (mem (kind exp) untyped) then
    print_string ("type: " ^ (getTypeString vc exp) ^ "\n")
  else ());
  let kind_val = kind exp in 
  print_string ("kind: " ^ getKindString vc kind_val ^ "\n");
  print_string "\n";;

let num_list n = 
  let rec num_list n =
    match n with 
        0 -> []
      | n -> [n-1] @ num_list (n-1) in
  rev (num_list n);;

let pchild exp i =  
  let exp = child exp i in 
    print_string ((exprString exp) ^ "\n"); 
    exp;;

let print_child vc exp i = 
  let kid = child exp i in
    print_string ("---------- child " ^ (string_of_int i) ^ " ---------\n");
    print_detailed vc kid;;

let print_children vc exp = 
  let i = arity exp in
  let nl = num_list i in
    map (print_child vc exp) nl;;

let print_child_kinds vc exp = 
  let n = arity exp in
  print_string "child kinds:\n";
  for i = 0 to n-1 do
    print_string ((kind_string vc (child exp i)) ^ "\n");
  done;;

let kind_child exp i = (kind o child exp) i;;

let rule_name exp = exprString (child exp 0);;


let rec print_list_thm list_thms = 
 match list_thms with 
 [] -> print_string "\n"
   | h::tl -> print_thm h ; print_string "\n"; print_list_thm tl ;;

let print_list_term termList =  
  do_list (fun x -> print_term x; print_string " ## \n" ) termList ; 
  print_string "\n" ;;

let print_term term = print_string (string_of_term term);;
let print_thm thm = print_string (string_of_thm thm);;

let pr = print_string;;
let nl() = pr "\n";;
let pt = print_term ;;
let pthm = print_thm ;;


(* ---------------------------------------------------------------------- *)
(*  Others                                                                *)
(* ---------------------------------------------------------------------- *)

(*
  only one is needed, get rid of the other 
*)

let get_child_list expr = 
  let n = arity expr in
  let rec get_child_list n store = 
    match n with
        0 -> store 
      | n -> get_child_list (n-1) ((child expr (n-1))::store) in
    get_child_list n [];;


let child_expr_list vc expr = 
  let ar = arity expr in
  let children = ref [] in
    (for i = 0 to ar - 1 do 
       children := !children @ [child expr i]
     done);
    !children;;

(* get all children of expr after index *)
let get_args index expr = 
  let ar = arity expr in
  let rec get_args index expr store = 
    if ar = index then store else
      get_args (index + 1) expr (store @ [child expr index]) in
    get_args index expr [];;


let mk_combc = curry mk_comb;;
let mk_eqc = curry mk_eq;;
let mk_impc = curry mk_imp;;
let mk_conjc = curry mk_conj;;
let mk_disjc = curry mk_disj;;


(* f,[x1;x2;...;xn] -> (f x1 x2 ...) *)
let rec app_list f l = 
  match l with 
      []-> f 
    | h::t -> app_list (mk_combc f h) t;;


let randt = snd o dest_eq o concl;;
let landt = fst o dest_eq o concl;;
